// -*- c++ -*-
// File: util.h
// Author: Suvrit Sra
// Date: 14 Jan 04

// Copyright (C) 2003 Suvrit Sra (suvrit@cs.utexas.edu)

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


#ifndef UTIL_H
#define UTIL_H


#include <gsl/gsl_matrix.h>
#include <gsl/gsl_matrix_float.h>
#include <gsl/gsl_vector.h>
#include <gsl/gsl_randist.h>
#include <gsl/gsl_cdf.h>
#include <gsl/gsl_rng.h>
#include <gsl/gsl_blas.h>
#include <ostream>
#include <cstdlib>
#include "ssutil_error.h"
#include "matrix.h"
#include "matrix_common.h"

#ifndef _dbg
#define _dbg(x) std::cerr << x << "\n"
#endif

/**
 * @namespace SSUtil This namespace contains a collection of utility
 * functions. These functions are of general utility and find use in many of
 * the libraries and softwares that i write.
 */
namespace SSUtil {

  std::ostream& operator << (std::ostream& os, const error_msg& e);
  void error(int num);
  void sign(const gsl_vector*, gsl_vector*);
  int sign(double t);
  int sign2(double t);

  SSLib::matrix* new_matrix(int mtype);
  int load_matrix(SSLib::matrix*, int mtype, char* fname);

  int rand_int (double n);

  /**
   * Not working as of now......
  template <typename T> T**  alloc_2d_array(size_t, size_t);
  template <typename T> T**  calloc_2d_array(size_t M, size_t N);
  template <typename T> void free_2d_array(T** ary, size_t M);
  template <typename T> void print_2d_array(T** array, size_t M, size_t N);
  */

  float**  alloc_2d_array_float(size_t, size_t);
  float**  calloc_2d_array_float(size_t M, size_t N);
  void     free_2d_array_float(float** ary, size_t M);
  void     print_2d_array_float(float** array, size_t M, size_t N);

  int argmax(double* v, int N);
  std::string matrix_type(int);
  int matrix_type_max();
  void matrix_normalize(gsl_matrix*);
  gsl_vector** alloc_array_vectors(size_t M, size_t N);
  void free_array_vectors(gsl_vector** ary, size_t M);

  int gsl_vector_setmin(gsl_vector* v, const gsl_vector* v2);
  int gsl_vector_sub2(gsl_vector* v, const gsl_vector* v2);
  int gsl_vector_add2(gsl_vector* v, const gsl_vector* v2);

  int gsl_vector_abs(gsl_vector* v);
  int gsl_vector_abs(gsl_vector_float* v);
  int gsl_vector_setmin(gsl_vector_float* v, const gsl_vector_float* v2);
  int gsl_vector_sub2(gsl_vector_float* v, const gsl_vector_float* v2);
  int gsl_vector_add2(gsl_vector_float* v, const gsl_vector_float* v2);
  
  double sum (double* v, size_t n);
  double sum (float* v, size_t n);

  double abs_sum (double* v, size_t n);
  double abs_sum (float* v, size_t n);

  double euclidean_norm (double *v, size_t n);
  double euclidean_norm (float* v, size_t n);

  double lp_norm (double* v, size_t n, double p, bool useAbs = false);
  double lp_norm (float* v, size_t n, double p, bool useAbs = false);

  double linf_norm (double* v, size_t n);
  double linf_norm (float* v, size_t n);

  size_t   num_nonzeros (double* v, size_t n);
  size_t   num_nonzeros (float* v, size_t n);

  gsl_vector* gsl_rand_vector(size_t);
  gsl_vector* read_gsl_vector(const char* filename);
  gsl_vector_float* read_gsl_vector_float(const char*);
  gsl_vector* read_gsl_vector(const char* filename, bool asbin);
  gsl_vector_float* read_gsl_vector_float(const char*, bool asbin);
  
  int         write_gsl_vector(gsl_vector*, const char* file);
  int         write_gsl_vector_float(gsl_vector_float*, const char* file);

  int         write_gsl_vector(gsl_vector*, const char* file, bool asbin);
  int         write_gsl_vector_float(gsl_vector_float*, const char* file, bool);

  void        print_vector(gsl_vector*);
  void  print_vector_float(gsl_vector_float*);
  /**
   * MATRIX, I/O FUNCTIONS. These function are wrappers around existing GSL
   * function since the GSL functions for I/O assume that the size of the
   * matrix is known in advance, which need not be the case. The matrix
   * expected by these functions is of the form:
   * M N
   * row1
   * row2
   * etc.
   * where M, N denote the number of rows and columns in the matrix
   * respectively. The matrices are all stored and accessed in C-style
   * (row-major) unless otherwise specified.
   */

  // Read an augmented GSL matrix from a file of the specified name.
  // This function not only reads the matrix, it allocate a gsl_matrix and
  // returns a pointer to it (with the contents being read from the file.
  gsl_matrix* read_gsl_matrix(char* filename);
  gsl_matrix_float* read_gsl_matrix_float(char* filename);

  // This function can be used to speed up the reading of a matrix that is
  // STORED in the format described above (which is clearly the row-major
  // format) and reading it and converting it to column major format (done by
  // doing a transpose operation and then tweaking the size1, size2 fields by
  // hand. If the input matrix is stored on the disk in column major format
  // then one should use read_gsl_matrix_raw
  gsl_matrix* read_gsl_matrix_fortran(char* filename);

  // Basically all this function does is, is to read in the dimensions of the
  // matrix into memory and then just read the entire m*n array into
  // memory...which if was stored in col major order will now be col major in
  // memory!
  gsl_matrix* read_gsl_matrix_raw(char* filename);


  // The following functions are just binary file counterparts of the
  // functions above...
  gsl_matrix* fread_gsl_matrix        (char* filename);
  gsl_matrix* fread_gsl_matrix_fortran(char* filename);
  gsl_matrix* fread_gsl_matrix_raw    (char* filename);

  gsl_matrix_float* fread_gsl_matrix_float (char* filename);
  gsl_matrix_float* fread_gsl_matrix_float_raw (char* filename);

  // The following functions read the matrix into a preallocated GSL matrix
  // of appropriate dimensions. If dimensions do not match an error will
  // occur. 

  int read_gsl_matrix        (gsl_matrix*, FILE*);
  int read_gsl_matrix_raw    (gsl_matrix*, FILE*);
  int read_gsl_matrix_fortran(gsl_matrix*, FILE*);

  int fread_gsl_matrix       (gsl_matrix*, FILE*);
  int fread_gsl_matrix_raw   (gsl_matrix*, FILE*);
  int fread_gsl_matrix_fortran(gsl_matrix*,FILE*);

  int read_gsl_matrix_float  (gsl_matrix_float*, const char*);
  int read_gsl_matrix        (gsl_matrix*, const char*);
  int read_gsl_matrix_raw    (gsl_matrix*, char*);
  int read_gsl_matrix_fortran(gsl_matrix*, char*);

  int fread_gsl_matrix       (gsl_matrix*, char*);
  int fread_gsl_matrix_raw   (gsl_matrix*, char*);
  int fread_gsl_matrix_fortran(gsl_matrix*,char*);

  int read_gsl_matrix        (gsl_matrix*, std::istream&);
  int read_gsl_matrix_raw    (gsl_matrix*, std::istream&);
  int read_gsl_matrix_fortran(gsl_matrix*, std::istream&);


  // The following functions write out the specified GSL matrix onto a file
  // or a filestream....they behave as exact counter parts to the read
  // functions listed above....
  int write_gsl_matrix        (gsl_matrix*, FILE*);
  int write_gsl_matrix_raw    (gsl_matrix*, FILE*);
  int write_gsl_matrix_fortran(gsl_matrix*, FILE*);

  int fwrite_gsl_matrix_float (gsl_matrix_float*, const char*);
  int fwrite_gsl_matrix       (gsl_matrix*, const char*);
  int fwrite_gsl_matrix       (gsl_matrix*, FILE*);
  int fwrite_gsl_matrix_raw   (gsl_matrix*, FILE*);
  int fwrite_gsl_matrix_fortran(gsl_matrix*,FILE*);

  int write_gsl_matrix        (gsl_matrix*, std::ostream&);
  int write_gsl_matrix_raw    (gsl_matrix*, std::ostream&);
  int write_gsl_matrix_fortran(gsl_matrix*, std::ostream&);
  int write_gsl_matrix        (gsl_matrix* m, char* fname);

  /*
   * UTILITY FUNCTIONS
   */

  /// Unif. random vector
  gsl_vector* random_vector( int m);
  void gsl_rand_vector(gsl_vector* v);
  void gsl_rand_vector_float(gsl_vector_float* v);

  /// rand vector
  void set_rand(double* v, size_t sz);
  /// randn type vector
  void set_randn(double* v, int sz);

  /// A uniformly random matrix with entries in [0..1]
  gsl_matrix* random_matrix (int m, int n);

  /// A random matrix, with various type of random distributions supported by
  /// means of allowing one to pass a gsl_rng object
  gsl_matrix* random_matrix (size_t m, size_t n, gsl_rng*);

  // Converts a row major matrix to col major and vice versa!
  gsl_matrix* toggle_rowcol_major(gsl_matrix*);


  /* ******************************************
   * OPERATIONS, FUNCTIONALS, ACTIONS etc...
   * ******************************************/

  // If no = 0, we use the default norm, which is the L_2 norm for
  // normlizing...else we normalize as per the specified norm....
  int normalize_matrix_cols(gsl_matrix*, int no = 0);
  int normalize_matrix_rows(gsl_matrix*, int no = 0);

  int vector_normalize(gsl_vector*);
  int vector_normalize(double*, int sz);
  int vector_float_normalize(gsl_vector_float*);

  double fnorm(gsl_matrix*   );	// Frobenius norm
  double norm (gsl_matrix*, int p);	// p, norm.

  /* **********************************************
   * Interface and utility functions for converting to SparseMatrix
   * Format... and other such functions...
   * **********************************************/

  // Writes out the given gsl matrix as a sparse matrix in CCS format, with
  // file name being specified by prefix.
  int  gsl_to_sparse(gsl_matrix*, char* prefix);
  long nnzero       (gsl_matrix*);

  /* ************************************
   * TOTALLY MISCELLANEOUS FUNCTIONS....
   * ************************************/
  // Converts a GSL matrix to a PGM image!!! Useful when doing simple image
  // processing work on PGM's that have been read in as a GSL matrix
  // See the tools pgmtogsl.cc pgmtogsl.pl
  int gsl_matrix_to_pgms(gsl_matrix*, char*, int r, int c, int scale);

  // Deprecated functions, held in here till code over-haul happens
  int         printon(gsl_matrix*, std::ostream&);
  int         printon_float(gsl_matrix_float*, std::ostream&);

  // Change a row major matrix to a column major matrix
  gsl_matrix* prepare_for_clapack(gsl_matrix*);


  // Apply scalar function to each element of the matrix and return result
  // in the second matrix
  int apply(double (*f)(double), gsl_matrix* src, gsl_matrix* dest);


  int binary_search (size_t* v, size_t& key, size_t N);
  //  template <class T>  int counting_sort (T* v, const size_t end, const size_t max, const size_t min);


}
#endif
