// util.cc: Implements functions in util.h
// Copyright (C) 2003 Suvrit Sra

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#include <cstdio>
#include <cmath>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <fstream>
#include "util.h"
#include <ostream>
#include <algorithm>
#include "ssutil_error.h"
#include "dense_matrix.h"
#include "ccs.h"
#include "crs.h"
#include "coord_ccs.h"
#include "coord_crs.h"
#include "matlab.h"
#include "harwell_boeing.h"
#include "row_coord.h"
#include "col_coord.h"

using namespace SSLib;


/**
 * Not working as of now, so i'll just create by hand..
template<> float**  SSUtil::alloc_2d_array<float>(uint, uint);
template<> void  SSUtil::free_2d_array<float>(float** ary, uint M);
template<> void  SSUtil::print_2d_array<float>(float** array, uint M, uint N);
template<> float**  SSUtil::calloc_2d_array<float>(uint M, uint N);
template<> double**  SSUtil::alloc_2d_array<double>(uint, uint);
template<> void  SSUtil::free_2d_array<double>(double** ary, uint M);
template<> void  SSUtil::print_2d_array<double>(double** array, uint M, uint N);
template<> double**  SSUtil::calloc_2d_array<double>(uint M, uint N);

*/

std::ostream& SSUtil::operator << (std::ostream& os, const SSUtil::error_msg& e)
{
  os << "#" << e.err << " " << e.msg;
  return os;
}

void SSUtil::error(int t)
{
  std::cerr << "SSUtil: Error ... " << ssutil_errors[t] << std::endl;
}

int SSUtil::sign(double t)
{
  if (t < 0)
    return -1;
  if (t > 0)
    return 1;
  return 0;
}

int SSUtil::matrix_type_max()
{
  return M_COLCOORD;
}
int SSUtil::sign2(double t)
{
  if (t < 0)
    return -1;
  return 1;
}

int SSUtil::argmax(double* v, int N)
{
  double m = v[0]; 
  int idx = 0;

  for (int i = 1; i < N; i++) {
    if (v[i] > m) { m = v[i]; idx = i;}
  }
  return idx;
}

void SSUtil::sign(const gsl_vector* v, gsl_vector *r)
{
  if (v->size != r->size)
    error(SSUtil::SSUTIL_VDIM_MISMATCH);
}

int SSUtil::rand_int (double n)
{
  int r = (int) (n * (rand() / (RAND_MAX + 1.0)));
  return r;
}

float** SSUtil::alloc_2d_array_float(size_t M, size_t N)
{
  float** ary = new float* [M];
  if (ary != 0) {
    for (size_t i = 0; i < M; i++)
      ary[i] = new float[N];
  } 
  return ary;
}

void SSUtil::print_2d_array_float(float** array, size_t M, size_t N) 
{
  for (size_t i = 0; i < M; i++) {
    for (size_t j = 0; j < N; j++)
      std::cout << array[i][j] << " ";
    std::cout << "\n";
  }
}
float** SSUtil::calloc_2d_array_float(size_t M, size_t N)
{
  float** ary = new float* [M];
  if (ary != 0) {
    for (size_t i = 0; i < M; i++) {
      ary[i] = new float[N];
      memset(ary, 0, sizeof(float)*N);
    }
  }
  return ary;
}

void SSUtil::free_2d_array_float(float** ary, size_t M)
{
  for (size_t i = 0; i < M; i++)
    delete[] ary[i];
  delete[] ary;
}

int SSUtil::gsl_vector_add2(gsl_vector* v, const gsl_vector* v2)
{
  size_t sz = std::min(v->size, v2->size);
  for (size_t i = 0; i < sz; i++)
    v->data[i] += v2->data[i];
  return 0;
}


int SSUtil::gsl_vector_sub2(gsl_vector* v, const gsl_vector* v2)
{
  size_t sz = std::min(v->size, v2->size);

  for (size_t i = 0; i < sz; i++)
    v->data[i] -= v2->data[i];

  return 0;
}

int SSUtil::gsl_vector_setmin(gsl_vector* v, const gsl_vector* v2)
{
  if (v->size != v2->size) {
    std::cerr << "gsl_vector_setmin: Dimension mismatch\n";
    return -1;
  }

  for (size_t i = 0; i < v->size; i++) 
    v->data[i] = std::min ( v->data[i], v2->data[i]);

  return 0;
  
}

int SSUtil::gsl_vector_setmin(gsl_vector_float* v, const gsl_vector_float* v2)
{
  if (v->size != v2->size) {
    std::cerr << "gsl_vector_setmin: Dimension mismatch\n";
    return -1;
  }

  for (size_t i = 0; i < v->size; i++) 
    v->data[i] = std::min ( v->data[i], v2->data[i]);
    return 0;
}

int SSUtil::gsl_vector_abs(gsl_vector* v)
{
  for (size_t i = 0; i < v->size; i++)
    v->data[i] = fabs(v->data[i]);

  return 0;
}

int SSUtil::gsl_vector_abs(gsl_vector_float* v)
{
  for (size_t i = 0; i < v->size; i++)
    v->data[i] = fabs(v->data[i]);

  return 0;
}

int SSUtil::gsl_vector_sub2(gsl_vector_float* v, const gsl_vector_float* v2)
{
  size_t sz = std::min(v->size, v2->size);
  
  for (size_t i = 0; i < sz; i++)
    v->data[i] -= v2->data[i];
  
  return 0;
}

int SSUtil::gsl_vector_add2(gsl_vector_float* v, const gsl_vector_float* v2)
{
  size_t sz = std::min(v->size, v2->size);
  for (size_t i = 0; i < sz; i++)
    v->data[i] += v2->data[i];
  return 0;
}


gsl_vector* SSUtil::read_gsl_vector(const char* fname)
{
  FILE* fp = fopen(fname, "r");
  if (fp == NULL) {
    std::cerr << "read_gsl_vector: Could not open `" << fname << "'\n";
    return 0;
  }

  // This is to allow future additions where the first line of the vector
  // file might have some other useful information aboutthe matrix...such
  // as whether matrix is triangular, symmetric, diagonal, rowmajor etc., etc.
  char line[1024];
  if (!fgets(line, 1000, fp)) {
    std::cerr << "read_gsl_vector: Error reading vector\n";
    return 0;
  }
  
  size_t m;
  sscanf(line, "%zu", &m);
  gsl_vector* v = gsl_vector_alloc(m);
  gsl_vector_fscanf(fp, v);
  fclose(fp);
  //cerr << mat->size1 << "***" << mat->size2 << endl;
  //gsl_matrix_fprintf(stdout, mat, "%f");
  return v;
}

gsl_vector* SSUtil::read_gsl_vector(const char* fname, bool asbin)
{
  if (!asbin)
    return read_gsl_vector(fname);

  FILE* fp = fopen(fname, "rb");
  if (fp == NULL) {
    std::cerr << "read_gsl_vector (bin): Could not open " << fname << std::endl;
    return 0;
  }

  size_t sz;
  size_t jnk;
  size_t st;
  st=fread(&sz, sizeof(size_t), 1, fp);
  st=fread(&jnk,sizeof(size_t), 1, fp);
  gsl_vector* v = gsl_vector_alloc(sz);
  gsl_vector_fread(fp, v);
  fclose(fp);
  return v;
}

gsl_vector_float* SSUtil::read_gsl_vector_float(const char* fname, bool asbin)
{
  if (!asbin)
    return read_gsl_vector_float(fname);

  FILE* fp = fopen(fname, "rb");
  if (fp == NULL) {
    std::cerr << "read_gsl_vector (bin): Could not open " << fname << std::endl;
    return 0;
  }

  size_t sz;
  size_t jnk;
  size_t st;
  st=fread(&sz, sizeof(size_t), 1, fp);
  st=fread(&jnk,sizeof(size_t), 1, fp);

  if (jnk != 0xaabb) {
    std::cerr << "Magic number missing from header, Potentially corrupt file\n";
  }
  gsl_vector_float* v = gsl_vector_float_alloc(sz);
  gsl_vector_float_fread(fp, v);
  fclose(fp);
  return v;
}

gsl_vector_float* SSUtil::read_gsl_vector_float(const char* fname)
{
  FILE* fp = fopen(fname, "r");
  if (fp == NULL) {
    std::cerr << "read_gsl_vector_float: Could not open " 
              << fname << std::endl;
    return 0;
  }

  // This is to allow future additions where the first line of the vector
  // file might have some other useful information aboutthe matrix...such
  // as whether matrix is triangular, symmetric, diagonal, rowmajor etc., etc.
  char line[1024];
  if (!fgets(line, 1000, fp)) {
    std::cerr << "read_gsl_vector_float: Error reading vector\n";
    return 0;
  }
  
  size_t m;
  sscanf(line, "%zu", &m);
  gsl_vector_float* v = gsl_vector_float_alloc(m);
  gsl_vector_float_fscanf(fp, v);
  fclose(fp);
  return v;
}

int SSUtil::write_gsl_vector(gsl_vector* v, const char* fname, bool asbin)
{
  if (!asbin)
    return write_gsl_vector(v, fname);

  FILE* fp = fopen(fname, "wb");
  if (fp == NULL) {
    std::cerr << "wrte_gsl_vector (bin): Could not open " << fname << std::endl;
    return 1;
  }

  size_t sz = v->size;
  size_t jnk = 0xaabb;
  size_t st;
  st=fwrite(&sz, sizeof(size_t), 1, fp);
  st=fwrite(&jnk, sizeof(size_t), 1, fp);

  gsl_vector_fwrite(fp,v);
  fclose(fp);
  return 0;

}

int SSUtil::write_gsl_vector(gsl_vector* v, const char* fname)
{
  FILE* fp = fopen(fname, "w");
  if (fp == NULL) {
    std::cerr << "wrte_gsl_vector: Could not open " << fname << std::endl;
    return 1;
  }

  // This is to allow future additions where the first line of the vector
  // could have some useful information
  fprintf(fp, "%zu\n", v->size);
  for (size_t i = 0; i < v->size; i++)
    fprintf(fp, "%lf\n", v->data[i]);
  fclose(fp);
  return 0;
}

int SSUtil::write_gsl_vector_float(gsl_vector_float* v, 
                                   const char* fname, bool asbin)
{
  if (!asbin)
    return write_gsl_vector_float(v, fname);

  FILE* fp = fopen(fname, "wb");
  if (fp == NULL) {
    std::cerr << "wrte_gsl_vector_float (bin): Could not open " << fname << std::endl;
    return 1;
  }

  size_t sz = v->size;
  size_t jnk = 0xaabb;
  size_t st;
  st=fwrite(&sz,  sizeof(size_t), 1, fp);
  st=fwrite(&jnk, sizeof(size_t), 1, fp);

  gsl_vector_float_fwrite(fp,v);
  fclose(fp);
  return 0;
}

int SSUtil::write_gsl_vector_float(gsl_vector_float* v, const char* fname)
{
  FILE* fp = fopen(fname, "w");
  if (fp == NULL) {
    std::cerr << "wrte_gsl_vector: Could not open " << fname << std::endl;
    return 1;
  }

  // This is to allow future additions where the first line of the vector
  // could have some useful information
  fprintf(fp, "%zu\n", v->size);
  for (size_t i = 0; i < v->size; i++)
    fprintf(fp, "%f\n", v->data[i]);
  fclose(fp);
  return 0;
}



void SSUtil::print_vector(gsl_vector* v)
{
  gsl_vector_fprintf(stdout, v, "%lf");
}

void SSUtil::print_vector_float(gsl_vector_float* v)
{
  gsl_vector_float_fprintf(stdout, v, "%f");
}


/**
 * Function to read in a gsl matrix from file specified by the global
 * 'fname'. This function allocates the matrix and returns a pointer to the
 * allocated matrix.
 */
gsl_matrix* SSUtil::read_gsl_matrix(char* fname)
{
  FILE* fp = fopen(fname, "r");
  if (fp == NULL) {
    std::cerr << "read_gsl_matrix: Could not open " << fname << std::endl;
    return 0;
  }

  // This is to allow future additions where the first line of the matrix
  // file might have some other useful information aboutthe matrix...such
  // as whether matrix is triangular, symmetric, diagonal, rowmajor etc., etc.
  char line[1024];
  if (!fgets(line, 1000, fp)) {
    std::cerr << "read_gsl_matrix: Error reading matrix\n";
    return 0;
  }
  
  int m, n;
  sscanf(line, "%d %d", &m, &n);
  gsl_matrix* mat = gsl_matrix_alloc(m,n);
  gsl_matrix_fscanf(fp, mat);
  fclose(fp);
  //cerr << mat->size1 << "***" << mat->size2 << endl;
  //gsl_matrix_fprintf(stdout, mat, "%f");
  return mat;
}


gsl_matrix_float* SSUtil::read_gsl_matrix_float(char* fname)
{
  FILE* fp = fopen64(fname, "r");
  if (fp == NULL) {
    std::cerr << "read_gsl_matrix: Could not open " << fname << std::endl;
    return 0;
  }

  // This is to allow future additions where the first line of the matrix
  // file might have some other useful information aboutthe matrix...such
  // as whether matrix is triangular, symmetric, diagonal, rowmajor etc., etc.
  char line[1024];
  if (!fgets(line, 1000, fp)) {
    std::cerr << "read_gsl_matrix: Error reading matrix\n";
    return 0;
  }
  
  int m, n;
  sscanf(line, "%d %d", &m, &n);
  gsl_matrix_float* mat = gsl_matrix_float_alloc(m,n);
  gsl_matrix_float_fscanf(fp, mat);
  fclose(fp);
  //cerr << mat->size1 << "***" << mat->size2 << endl;
  //gsl_matrix_fprintf(stdout, mat, "%f");
  return mat;
}

/**
 * This is the function to use for reading a matrix for use with CLAPACK
 * routines, it automatically reads in a matrix stored in row-major (C)
 * format into a FORTRAN format....thus avoiding wrapper calls and wasted
 * memory for a copy of the input matrix
 */
gsl_matrix* SSUtil::read_gsl_matrix_fortran(char* fname)
{
  gsl_matrix* tmp = read_gsl_matrix(fname);
  gsl_matrix* mat = 0;

  if (tmp != 0) {
    mat = gsl_matrix_alloc(tmp->size2, tmp->size1);
    gsl_matrix_transpose_memcpy(mat, tmp);
    size_t t = mat->size1;
    mat->size1 = mat->size2;
    mat->size2 = t;
    gsl_matrix_free(tmp);
  }
  return mat;
}

int SSUtil::read_gsl_matrix_float(gsl_matrix_float* mat, const char* fname)
{
  FILE* fp = fopen(fname, "r");
  if (fp == NULL) {
    std::cerr << "read_gsl_matrix: Could not open " << fname << std::endl;
    return -1;
  }

  gsl_matrix_float_fscanf(fp, mat);
  fclose(fp);
  return 0;  
}

int SSUtil::read_gsl_matrix(gsl_matrix* mat, const char* fname)
{
   FILE* fp = fopen(fname, "r");
   if (fp == NULL) {
     std::cerr << "read_gsl_matrix: Could not open " << fname << std::endl;
     return -1;
   }
   
   //int m, n;
   //fscanf(fp, "%d %d", &m, &n);
   //if ((int)mat->size1 != m or (int)mat->size2 != n) {
   //  std::cerr << "read_gsl_matrix: Incorrect matrix size " << m << " x " << n << std::endl;
    // return -1;
   //}
   gsl_matrix_fscanf(fp, mat);
   fclose(fp);
   return 0;
}


/*
 * Read matrix as 1-D array into memory
 */

gsl_matrix* SSUtil::read_gsl_matrix_raw(char* file)
{
   FILE* fp = fopen(file, "r");
   if (fp == NULL) {
     std::cerr << "read_gsl_matrix: Could not open " << file << std::endl;
     return 0;
   }
   
   size_t m, n;
   int st;
   st=fscanf(fp, "%zu %zu", &m, &n);
   gsl_matrix* mat = gsl_matrix_alloc(m, n);

   for (size_t i = 0; i < m*n; i++)
     st=fscanf(fp, "%lf", &mat->data[i]);
   
   fclose(fp);
   return mat;
}

/**
 * Does and fwrite of a GSL matrix.
 * TBD: Include error checking....
 */
int SSUtil::write_gsl_matrix(gsl_matrix* m, char* fname)
{
  FILE* fp = fopen(fname, "wb");
  if (fp == NULL) {
    std::cerr << "write_gsl_matrix: could not open " << fname << std::endl;
    return -1;
  }

  gsl_matrix_fwrite(fp, m);
  return 0;
}

/*
 * TBD: Include error checking....
 * do a binary (fread) of a prealloced gsl matrix
 */
int SSUtil::fread_gsl_matrix(gsl_matrix* m, char* fname)
{
  FILE* fp = fopen(fname, "rb");
  if (fp == NULL) {
    std::cerr << "fread_gsl_matrix: could not open " << fname << std::endl;
    return -1;
  }
  return   gsl_matrix_fread(fp, m);
}

/*
 * TBD: Include error checking....
 */
gsl_matrix_float* SSUtil::fread_gsl_matrix_float(char* file)
{
  FILE* fp = fopen(file, "rb");
  if (!fp) {
    std::cerr << "gsl_matrix_float* fread_gsl_matrix_float(" << file << "): failed\n";
    return 0;
  }
  size_t m, n;
  size_t st;
  st=fread(&m, sizeof(size_t), 1, fp);
  st=fread(&n, sizeof(size_t), 1, fp);
  gsl_matrix_float* mat = gsl_matrix_float_alloc(m, n);
  if (mat == 0) {
    std::cerr << "gsl_matrix_float* fread_gsl_matrix_float(" << file << "): Error allocating matrix\n";
    return 0;
  }
  // gsl_matrix_float_fread fails for large files because the num. of items
  //read is wrong due to overflow and wrap around. So we do an untested
  //fread here instead. Ideally, we could split it into loading step by
  //step, that would be better....TODO later some day....
  //gsl_matrix_float_fread(fp, mat);
  size_t rsz = m*n;
  if (rsz > 320000000) {
    size_t iread = fread(mat->data, sizeof(float), 320000000, fp);
    std::cerr << "gsl_matrix_float_fread: READ " << iread << " ITEMS\n";
    iread = fread(mat->data+320000000, sizeof(float), rsz-320000000, fp); 
    std::cerr << "gsl_matrix_float_fread: READ " << iread << " ITEMS\n";
  } else {
    size_t iread = fread(mat->data, sizeof(float), rsz, fp);
    std::cerr << "gsl_matrix_float_fread: READ " << iread << " ITEMS\n";
  }
  return mat;
}

/*
 * TBD: Include error checking....
 */
gsl_matrix* SSUtil::fread_gsl_matrix(char* file)
{
  FILE* fp = fopen(file, "rb");
  if (!fp) {
    std::cerr << "gsl_matrix* fread_gsl_matrix(" << file << "): failed\n";
    return 0;
  }
  size_t m, n;
  size_t st;
  st=fread(&m, sizeof(size_t), 1, fp);
  st=fread(&n, sizeof(size_t), 1, fp);
  gsl_matrix* mat = gsl_matrix_alloc(m, n);
  if (mat == 0) {
    std::cerr << "gsl_matrix* fread_gsl_matrix(" << file << "): Error allocating matrix\n"; 
    fclose(fp);
    return 0;

  }
  gsl_matrix_fread(fp, mat);
  fclose(fp);
  return mat;
}

// Of course, since it is fread, it is raw....
gsl_matrix* SSUtil::fread_gsl_matrix_raw(char* file)
{
  return fread_gsl_matrix(file);
}

// fortran style fread?
gsl_matrix* SSUtil::fread_gsl_matrix_fortran(char* file)
{
  gsl_matrix* tmp = fread_gsl_matrix(file);
  gsl_matrix* mat = 0;
  if (tmp != 0) {
    mat = gsl_matrix_alloc(tmp->size2, tmp->size1);
    gsl_matrix_transpose_memcpy(mat, tmp);
    size_t t = mat->size1;
    mat->size1 = mat->size2;
    mat->size2 = t;
    gsl_matrix_free(tmp);
  }
  return mat;
}

/*
 * TBD: Include error checking....
 */
/*
 * fwrite a gsl matrix. This function also stores the size of the matrix in
 * the binary file allowing for the fread_gsl_matrix function above...
 */
int SSUtil::fwrite_gsl_matrix(gsl_matrix* mat, const char* file)
{
  FILE* fp = fopen(file, "wb");
  if (!fp) {
    std::cerr << "int fwrite_gsl_matrix(" << file << "): file open failed\n";
    return -1;
  }
  size_t st;
  st=fwrite(&mat->size1, sizeof(size_t), 1, fp);
  st=fwrite(&mat->size2, sizeof(size_t), 1, fp);
  return gsl_matrix_fwrite(fp, mat);
}


/*
 * fwrite a gsl matrix. This function also stores the size of the matrix in
 * the binary file allowing for the fread_gsl_matrix function above...
 */
int SSUtil::fwrite_gsl_matrix_float(gsl_matrix_float* mat, const char* file)
{
  //int sizes[] = {100, 200, 500, 1000, 2000, 5000, 10000, 15000, 20000};

  //char filename[256];

  //for (int i = 0; i < 9; i++) {
  //sprintf(filename, "%s.%d", file, i);
  //std::cerr << "Writing out to: " << filename << "\n";
    FILE* fp = fopen(file, "wb");
    if (!fp) {
      std::cerr << "int fwrite_gsl_matrix(" << file << "): file open failed\n";
      return -1;
    }

    //size_t sz = sizes[i]*1000;
    size_t st;
    st=fwrite(&mat->size1, sizeof(size_t), 1, fp);
    st=fwrite(&mat->size2, sizeof(size_t), 1, fp);
    
    //for (int j = 0; j < sizes[i]/100; j++)
    gsl_matrix_float_fwrite(fp, mat);
    fclose(fp);
    //}
  return 0;
}

int SSUtil::fwrite_gsl_matrix(gsl_matrix* mat, FILE* fp)
{
  if (!fp) {
    std::cerr << "int fwrite_gsl_matrix(): file error\n";
    return -1;
  }
  size_t st;
  st=fwrite(&mat->size1, sizeof(size_t), 1, fp);
  st=fwrite(&mat->size2, sizeof(size_t), 1, fp);
  return gsl_matrix_fwrite(fp, mat);
}

std::string SSUtil::matrix_type(int ty)
{
  std::string s;
  switch (ty) {
  case M_DENSE:
    s = "DENSE";
    break;
  case M_CCS:
    s = "CCS";
    break;
  case M_CRS:
    s = "CRS";
    break;
  case M_COORD_CRS:
    s = "COORD_CRS";
    break;
  case M_COORD_CCS:
    s = "COORD_CCS";
    break;
  case M_HB:
    s = "HARWELL_BOEING";
    break;
  case M_MATLAB:
    s = "MATLAB";
    break;
  case M_ROWCOORD:
    s = "ROW_COORD";
    break;
  case M_COLCOORD:
    s = "COL_COORD";
    break;
  default:
    s = "UNDEFINED";
    break;
  }
  return s;
}

matrix* SSUtil::new_matrix(int mtype)
{
  matrix* m = 0;

  switch (mtype) {
  case M_DENSE:
    m = new dense_matrix();
    break;
  case M_CCS:
    m = new ccs();
    break;
  case M_CRS:
    m = new crs();
    break;
  case M_COORD_CRS:
    m = new coord_crs();
    break;
  case M_COORD_CCS:
    m = new coord_ccs();
    break;
  case M_HB:
    m = new harwell_boeing();
    break;
  case M_MATLAB:
    m = new matlab();
    break;
  case M_ROWCOORD:
    m = new row_coord();
    break;
  case M_COLCOORD:
    m = new col_coord();
    break;
  default:
    m = 0;
    break;
  }
  return m;
}


/**
 * Returns a matrix with random entries. Useful function. 
 * Uses simulation strength random number
 * generator.
 */
gsl_matrix* SSUtil::random_matrix(int m, int n)
{
  gsl_matrix* mat = gsl_matrix_alloc(m, n);
  gsl_rng* r = gsl_rng_alloc(gsl_rng_mt19937);
  gsl_rng_set(r, time(0));
  if (mat != 0) {
    for (int i = 0; i < m*n; i++)
      mat->data[i] = gsl_rng_uniform(r);
  }
  gsl_rng_free(r);
  return mat;

}

gsl_vector* SSUtil::random_vector(int m)
{
  gsl_vector* v = gsl_vector_alloc(m);
  gsl_rng* r = gsl_rng_alloc(gsl_rng_mt19937);
  gsl_rng_set(r, time(0));
  if (v != 0) {
    for (int i = 0; i < m; i++)
      v->data[i] = gsl_rng_uniform(r);
  }
  gsl_rng_free(r);
  return v;
}

void SSUtil::set_rand(double* v, size_t sz)
{
  gsl_rng* r = gsl_rng_alloc(gsl_rng_mt19937);
  gsl_rng_set(r, time(0));
  if (v != 0) {
    for (size_t i = 0; i < sz; i++)
      v[i] = gsl_rng_uniform(r);
  }
  gsl_rng_free(r);
}

void SSUtil::set_randn(double* v, int sz)
{
  gsl_rng* r = gsl_rng_alloc(gsl_rng_mt19937);
  //gsl_rng_set(r, time(0));
  if (v != 0) {
    for (int i = 0; i < sz; i++)
      v[i] = gsl_ran_gaussian_ziggurat(r, 1.0);
  }
  gsl_rng_free(r);
}

int SSUtil::vector_normalize(gsl_vector* v)
{
  double no = gsl_blas_dnrm2(v);
  if (no == 0.0)
    return -1;
  for (size_t i = 0; i < v->size; i++)
    v->data[i] /= no;
  return 0;
}

int SSUtil::vector_float_normalize(gsl_vector_float* v)
{
  double no = 0.0;
  for (size_t i = 0; i < v->size; i ++)
    no += v->data[i]*v->data[i];

  no = sqrt(no);

  if (no == 0.0)
    return 0;
  for (size_t i = 0; i < v->size; i++)
    v->data[i] /= no;
  return 0;
}

void SSUtil::matrix_normalize(gsl_matrix* M)
{
  double sum = 0;
  for (size_t i = 0; i < M->size1 * M->size2; i++) {
    sum += M->data[i];
  }

  for (size_t i = 0; i < M->size1 * M->size2; i++) {
    M->data[i] /= sum;
  }
}

int SSUtil::vector_normalize(double* v, int sz)
{
  double no = euclidean_norm(v, sz);
  if (no == 0.0)
    return 0;
  for (int i = 0; i < sz; i++)
    v[i] /= no;
  return 0;
}

gsl_vector** SSUtil::alloc_array_vectors(size_t M, size_t N)
{
  gsl_vector** ary = new gsl_vector*[M];
  for (size_t i = 0; i < M; i++)
    ary[i] = gsl_vector_alloc(N);

  return ary;
}

void SSUtil::free_array_vectors(gsl_vector** ary, size_t M)
{
  for (size_t i = 0; i < M; i++)
    gsl_vector_free(ary[i]);
  delete[] ary;
}

/**
 * Print out contents of gsl matrix A onto ostream os
 */
int SSUtil::printon(gsl_matrix* A, std::ostream& os)
{
  char buf[255];

  os << A->size1 << " " << A->size2 << std::endl;
  for (size_t i = 0; i < A->size1; i++) {
    for (size_t j = 0; j < A->size2; j++) {
      sprintf(buf, "%.6f", gsl_matrix_get(A, i, j));
      //os << gsl_matrix_get(A, i, j) << " ";
      os << buf << " ";
    }
    os << std::endl;
  }
  return 0;
}

int SSUtil::printon_float(gsl_matrix_float* A, std::ostream& os)
{
  os << A->size1 << " " << A->size2 << std::endl;
  for (size_t i = 0; i < A->size1; i++) {
    for (size_t j = 0; j < A->size2; j++) {
      os << gsl_matrix_float_get(A, i, j) << " ";
    }
    os << std::endl;
  }
  return 0;
}

/**
 * Writes out each column of the input matrix g into files
 */
int SSUtil::gsl_matrix_to_pgms(gsl_matrix* g, char* prefix, int r, int c, int scale)
{
  if (r*c != (int)g->size1) {
    std::cerr << "gsl_to_pgm: " << r << " x " << c << " != " << g->size1 << std::endl;
    return -1;
  }
  
  char filename[255];
  for (int j = 0; j < (int)g->size2; j++) {
    sprintf(filename, "%s.%d.pgm", prefix, j);
    FILE* fp = fopen(filename, "w");
    
    // Write pgm header now
    fprintf(fp, "P5\n");		// pgm magic number
    fprintf(fp, "%d %d\n", c, r);	// cols, rows (width, height)
    fprintf(fp, "255\n");		// maxgray level
    
    for (int h = 0; h < r; h++) {
      for (int w = 0; w < c; w++)
	fprintf(fp, "%c", (unsigned char)(gsl_matrix_get(g, h*c+w, j)*scale));
    }
    fclose(fp);
  }
  return 0;

}

/**
 * Returns number of non-zeros in matrix A
 */
long SSUtil::nnzero(gsl_matrix* A)
{
  long res = 0;
  for (size_t i = 0; i < A->size1*A->size2; i++)
    if (A->data[i] != 0) ++res;
  return res;
}

/**
 * @args: gsl_matrix* A -- Input matrix
 * @args: char* prefix  -- Filename prefix 
 */
int SSUtil::gsl_to_sparse(gsl_matrix* A, char* prefix)
{
  size_t cols = A->size2;
  size_t rows = A->size1;

  std::string dim(prefix);
  std::string rows_file(dim + "_row_ccs");
  std::string cols_file(dim + "_col_ccs");
  std::string nz_file  (dim + "_txx_nz");

  dim += "_dim";

  std::ofstream ofile(dim.data());
  //ofile.open(dim.data());
  if (ofile.fail()) {
    std::cerr << "gsl_to_sparse: Error opening " << dim << std::endl;
    return -1;
  }

  long nz = nnzero(A);

  ofile << rows << " " << cols << " " << nz << std::endl;
  ofile.close();
  ofile.clear();

  ofile.open(rows_file.data());
  std::ofstream colptrs;
  colptrs.open(cols_file.data());

  if (ofile.fail() or colptrs.fail()) {
    std::cerr << "gsl_to_sparse: Error opening file\n";
    return -1;
  }

  std::ofstream nzfile;
  nzfile.open(nz_file.data());
  if (nzfile.fail()) {
    std::cerr << "gsl_to_sparse: Error opening file " << nz_file << "\n";
    return -1;
  }

  // Now we write out the CCS matrix appropriately.
  int colcnt = 0;
  colptrs << colcnt << " "; 	// Start of col_ccs file

  for (size_t i = 0; i < rows; i++) {
    for (size_t j = 0; j < cols; j++) {
      double x = gsl_matrix_get(A, i, j);
      if (x != 0) {
	nzfile << x << "\n";
	ofile << i << "\n";
	++colcnt;
      }
    }
    colptrs << colcnt << "\n";
  }

  colptrs << nz << std::endl;
  ofile.close();
  colptrs.close();
  nzfile.close();
  return 0;
}

/**
 * This function is a wrapper for converting a C-style matrix to a FORTRAN
 * style matrix so that we can pass it to CLAPACK routines. Nothing else
 * prolly needs to be changed before calling a CLAPACK routine!
 **/
gsl_matrix* SSUtil::prepare_for_clapack(gsl_matrix* tmp)
{
  gsl_matrix* mat = gsl_matrix_alloc(tmp->size2, tmp->size1);
  gsl_matrix_transpose_memcpy(mat, tmp);
  // gsl_matrix_free(tmp); // We leave it untouched...
  //size_t s1 = mat->size1;
  //mat->size1 = mat->size2;	// BAD LOW LEVEL changes to a GSL mat!!
  //mat->size2 = s1;
  return mat;
}

double SSUtil::fnorm(gsl_matrix* X)
{

  if (X == 0) {
    std::cerr << "Factorization::fnorm(X) called for unallocated X\n";
    return -1;
  }

  double no = 0.0;
  for (size_t i = 0; i < X->size1 * X->size2; i++) {
    double t = X->data[i];
    no += (t*t);
  }
  return sqrt(no);
}

/**
 * apply
 */

int SSUtil::apply(double (*f)(double), gsl_matrix* src, gsl_matrix* dest)
{
  if (dest->size1 != src->size1 or dest->size2 != src->size2)
    return -1;

  for (size_t i = 0; i < src->size1*src->size2; i++)
    dest->data[i] = f(src->data[i]);
  return 0;
}

double SSUtil::sum (double* v, size_t n)
{
  double s = 0.0;
  for (size_t i = 0; i < n; i++)
    s += v[i];
  return s;
}

double SSUtil::sum (float* v, size_t n)
{
  double s = 0.0;
  for (size_t i = 0; i < n; i++)
    s += v[i];
  return s;
}

double SSUtil::abs_sum (double* v, size_t n)
{
  double s = 0.0;
  for (size_t i = 0; i < n; i++)
    s+= fabs(v[i]);
  return s;
}

double SSUtil::abs_sum (float* v, size_t n)
{
  double s = 0.0;
  for (size_t i = 0; i < n; i++)
    s+= fabs(v[i]);
  return s;
}


double SSUtil::euclidean_norm (double *v, size_t n)
{
  double s = 0.0;
  for (size_t i = 0; i < n; i++)
    s += (v[i] * v[i]);
  return sqrt(s);
}

double SSUtil::euclidean_norm (float *v, size_t n)
{
  double s = 0.0;
  for (size_t i = 0; i < n; i++)
    s += (v[i] * v[i]);
  return sqrt(s);
}

double SSUtil::lp_norm (double* v, size_t n, double p, bool useAbs)
{
  if (p <= 0)
    return 0.0;

  double s = 0.0;
  if (useAbs) {
    for (size_t i = 0; i < n; i++)
      s += pow (fabs(v[i]), p);
  } else {
    for (size_t i = 0; i < n; i++)
      s += pow (v[i], p);
  }
  return pow (s, 1/p);
}

double SSUtil::lp_norm (float* v, size_t n, double p, bool useAbs)
{
  if (p <= 0)
    return 0.0;

  double s = 0.0;
  if (useAbs) {
    for (size_t i = 0; i < n; i++)
      s += pow (fabs(v[i]), p);
  } else {
    for (size_t i = 0; i < n; i++)
      s += pow (v[i], p);
  }
  return pow (s, 1/p);
}

double SSUtil::linf_norm (double* v, size_t n)
{
  double s = 0.0;
  for (size_t i = 0; i < n; i++) {
    double t = fabs(v[i]);
    if (t > s)
      s = t;
  }
  return s;
}

double SSUtil::linf_norm (float* v, size_t n)
{
  double s = 0.0;
  for (size_t i = 0; i < n; i++) {
    double t = fabs(v[i]);
    if (t > s)
      s = t;
  }
  return s;
}

size_t SSUtil::num_nonzeros(double* v, size_t n)
{
  size_t c = 0;
  for (size_t i = 0; i < n; i++)
    if (v[i] != 0)
      ++c;
  return c;
}


size_t SSUtil::num_nonzeros(float* v, size_t n)
{
  size_t c = 0;
  for (size_t i = 0; i < n; i++)
    if (v[i] != 0)
      ++c;
  return c;
}

void SSUtil::gsl_rand_vector(gsl_vector* v)
{

  gsl_rng* r = gsl_rng_alloc(gsl_rng_mt19937);
  gsl_rng_set(r, time(0));
  if (v != 0) {
    for (size_t i = 0; i < v->size; i++)
      v->data[i] = gsl_rng_uniform(r);
  }
  gsl_rng_free(r);
}

void SSUtil::gsl_rand_vector_float(gsl_vector_float* v)
{

  gsl_rng* r = gsl_rng_alloc(gsl_rng_mt19937);
  gsl_rng_set(r, time(0));
  if (v != 0) {
    for (size_t i = 0; i < v->size; i++)
      v->data[i] = (float)gsl_rng_uniform(r);
  }
  gsl_rng_free(r);
}


gsl_vector* SSUtil::gsl_rand_vector(size_t n)
{
  gsl_vector* v = gsl_vector_alloc(n);
  gsl_rng* r = gsl_rng_alloc(gsl_rng_mt19937);
  gsl_rng_set(r, time(0));
  if (v != 0) {
    for (size_t i = 0; i < n; i++)
      v->data[i] = gsl_rng_uniform(r);
  }
  gsl_rng_free(r);
  return v;

}


int SSUtil::binary_search (size_t* v, size_t& key, size_t N)
{
  if (N == 0)
    return -1;
  if (N == 1)
    return (v[0] == key ? 0 : -1);

  size_t mid;
  size_t hi = N;
  size_t low = 0;

  while (low <= hi) {
    // Worry not yet about overflows....might start worrying sometime though...
    mid = (low + hi) /2;
    if (v[mid] == key)
      return mid;
    if (key < v[mid]) {
      hi = mid-1;
    } else {
      low = mid+1;
    }
  }
  return -1;
}

