// File: sparsematrix_compact_float.h -*- c++ -*-
// Time-stamp: <14 Februar 2008 12:26:03  CET --  suvrit>
// Version 1.0

/* This file contains the base class for all the sparse matrix classes that
 * we will eventually support. This file depends on the GSL library for
 * operations dealing with vector, matrix multiply and such operations.
*/

#ifndef _MY_SPARSEMATRIX_COMPACT_FLOAT_H
#define _MY_SPARSEMATRIX_COMPACT_FLOAT_H

#include <string>
#include <iostream>
#include <gsl/gsl_vector_float.h>
#include <gsl/gsl_matrix_float.h>


#include "matrix_compact_float.h"

namespace SSLib {

  /**
   * @class sparsematrix_compact_float This is the base class for all the
   * sparsematrix_compact_float classes in the library. It includes some basic
   * features that all 
   * sparsematrices have. Note that this interface requires a subclass to
   * implement certain basic virtual functions. This class is derived from
   * the matrix base class, permitting one to intermingle sparse matrices
   * with general dense matrices. Of course, to retain correctness and
   * efficiency one should take care to invoke the appropriate methods.
   *
   * Note that this class does not enforce the requirement of too many
   * sparse matrix operations that a derived class should support. That
   * requirement is given by the superclass 'matrix'.
   *
   * TODO: Augment this class in the future.
   */
  class sparsematrix_compact_float : public matrix_compact_float {

  protected:
    /// Creates an empty sparse matrix. Public usage not allowed, because
    /// using matrix is suggested instead of sparsematrix directly.
    sparsematrix_compact_float() : matrix_compact_float (0, 0) {
      NZ = 0; name = "";
    }
    
    /// number of non-zeros is this matrix
    size_t NZ;
    /// A string identifier as the name of this matrix.
    std::string name;

  public:
    /**
     * Constructs a sparse matrix with specified number of rows, columns and
     * nonzeros. No space is allocated at this stage. Each derived class is
     * responsible for its memory management.
     */
    sparsematrix_compact_float(size_t rows, size_t cols, size_t nz) : matrix_compact_float(rows, cols) {
      NZ = nz;
    }
    
    
    virtual ~sparsematrix_compact_float() {}

    /** @name General_Features_Operations
     * Some basic features of the matrix
     */
    /*@{*/
    /// The maximum number of nonzero elements in the matrix (corresponds to
    /// matlab's nzmax
    size_t nonzeros() const { return NZ;}

    /// Just a statistic to indicate how full the matrix is.
    double density() const { return 1.0*NZ/ (nrows()*ncols()*1.0);}

    /// Sets the size and nonzero parameters of the matrix
    void setsize(size_t m, size_t n, size_t nnz) {
      matrix_setsize(m, n);
      NZ = nnz;
    }

    /// Allocate memory for this sparsematrix
    virtual int matrix_alloc (size_t m, size_t n, size_t nz) = 0;

    virtual int matrix_resize (size_t m, size_t n, size_t nz) = 0;
    /*@}*/

  }; // end of class

} // end namespace
#endif
