// File: ccs.h 
// Time-stamp: <06 March 2007 08:11:20 PM CST --  suvrit>
// Impl. of ccs format sparse matrices
/* Author: Suvrit Sra <suvrit@cs.utexas.edu> */
/* (c) 2007  Suvrit Sra */

#ifndef _SAGE_CCS_H
#define _SAGE_CCS_H

#include <gsl/gsl_vector.h>


#define CCSERR_MALLOC -99
#define CCSERR_FREE   -98
#define CCSERR_RESIZE -97
#define CCS_SUCCESS 0

#define LOAD_BIN 1
#define LOAD_TXT 2

typedef gsl_vector vector;

typedef struct _dccs_matrix {
  double* m_val;
  size_t* m_colptrs;
  size_t* m_rowindx;
  size_t nrows;
  size_t ncols;
  size_t nnz;
} dccs_matrix;


/* ############################################################################# */
/** matrix load/save routines */

// Load a dccs matrix in binary mode
dccs_matrix* dccs_load_bin(const char* prefix);
// Load a dccs matrix from a bunch of text files
dccs_matrix* dccs_load_txt(const char* prefix);
// This is the generic procedure to call
dccs_matrix* dccs_load    (const char*,int mode);
// Save a dccs matrix in binary mode
int          dccs_save_bin(dccs_matrix*, const char*);
// Save a dccs matrix in txt mode
int          dccs_save_txt(dccs_matrix*, const char* prefix);
// TODO: Save a dccs matrix in various different formats
int          dccs_save_as (dccs_matrix*, const char*, int typ, int mode);


/* ############################################################################# */
/** Memory alloc/dealloc functions */

// Alloc a basic dccs matrix
dccs_matrix* dccs_alloc (size_t m, size_t n, size_t nz);
// Calloc a basic dccs matrix
dccs_matrix* dccs_calloc(size_t m, size_t n, size_t nz);
// Free the memory being used by a dccs matrix
int          dccs_free  (dccs_matrix* mtx);
// Try to resize (uses realloc)
int          dccs_resize (dccs_matrix* mtx, size_t m, size_t n, size_t nz);
// Use an externally allocated CCS matrix
dccs_matrix* dccs_external_alloc(size_t* col, size_t* row, double* val);
// Return a deep copy
dccs_matrix* dccs_clone();

/* ############################################################################# */
/** Matrix augmentation or truncation functions */

// Augment matrix by adding a row
int dccs_insert_row(dccs_matrix*, vector* r, int aft);
// Remove the said row
int dccs_remove_row(dccs_matrix*, size_t r);
// Insert a column
int dccs_insert_col(dccs_matrix*, vector* c, int aft);
// Remove the said col
int dccs_remove_col(dccs_matrix*, size_t c);

/* ############################################################################# */
/** Accessing the rows/columns or elements of the matrix */

// Return row 'r' of matrix
int dccs_get_row (const dccs_matrix*, size_t r, vector*);
// Return col 'c' of matrix
int dccs_get_col (const dccs_matrix*, size_t c, vector*);
// return the (i, j) entry of the matrix
double  dccs_get (const dccs_matrix*, size_t i, size_t j);
// TODO: set the (i, j) entry of the matrix.
int dccs_set     (dccs_matrix*, size_t i, size_t j, double val);
// TODO: Sets the specified row to the given vector
int dccs_set_row (dccs_matrix*, size_t r, vector*);
// Sets the existing non-zeros, ignores other entries
int dccs_fast_set_row(dccs_matrix*, size_t r, vector*);
// TODO: Sets the specified col to the given vector
int dccs_set_col (dccs_matrix*, size_t c, vector*);
// Sets only the existing non-zeros, ignores others
int dccs_fast_set_col(dccs_matrix*, size_t c, vector*);
// TODO: Returns a submatrix that goes from row i1--i2 and cols j1--j2
int dccs_submatrix(const dccs_matrix* src, size_t i1, size_t j1, size_t i2, size_t j2, dccs_matrix* dest);


/* ############################################################################# */
/** Elementwise functions of the matrix */

// Vector l_p norms for this matrix
double dccs_norm (const dccs_matrix*, double p);
// Apply an arbitrary function elementwise to this matrix
int   dccs_apply (dccs_matrix*, double (* fn)(double));
// Obtain sum of elements of the matrix
double dccs_sum_all(const dccs_matrix*);
// Set all entries to a specified constant
int   dccs_set_all(dccs_matrix*, double c);
// Scale all entries by a specified constant
int   dccs_scale  (dccs_matrix*, double s);
// elementwise mul (Schur or Hadamard product)
int   dccs_mul    (dccs_matrix* a, const dccs_matrix* b);
int   dccs_fastmul(dccs_matrix* a, const dccs_matrix* b);
// elementwise add
int   dccs_add    (dccs_matrix* a, const dccs_matrix* b);
int   dccs_fastadd(dccs_matrix* a, const dccs_matrix* b);
// elementwise sub
int   dccs_sub    (dccs_matrix* a, const dccs_matrix* b);
int   dccs_fastsub(dccs_matrix* a, const dccs_matrix* b);
// elementwise div
int   dccs_div    (dccs_matrix* a, const dccs_matrix* b);
int   dccs_fastdiv(dccs_matrix* a, const dccs_matrix* b);

/* ############################################################################# */
/** Functions on columns of the matrix */

// compute 'p' norm of given column
double dccs_col_norm (const dccs_matrix*, size_t c, double p);
// compute dot product of col_c1 with col_c2
double dccs_col_dot  (const dccs_matrix*, size_t c1, size_t c2);
// compute dot product of col c with vector v
double dccs_col_dotv (const dccs_matrix*, size_t c, vector* v);
// TODO: column c1 <- c1 + c2 (causes potential fill-in)
int    dccs_col_add  (dccs_matrix*, size_t c1, size_t c2);
// TODO: c1 <- c1 - c2
int    dccs_col_sub  (dccs_matrix*, size_t c1, size_t c2);
// c1 <- s * c1
int    dccs_col_scale(dccs_matrix*, size_t c1, double s);
// TODO: c1 <- c1 - v
int    dccs_col_subv  (dccs_matrix*, size_t c1, vector* v);
// TODO: c1 <- c1 + v
int    dccs_col_addv  (dccs_matrix*, size_t c1, vector* v);
// r <- s * c1
int    dccs_col_scalev(dccs_matrix*, size_t c1, double s, vector* r);
// compute sum of each column and store in the vector
int    dccs_col_sum  (dccs_matrix*, vector* r);
// r <- r + a*col_c
int    dccs_col_daxpy(dccs_matrix*, size_t c, double a, vector* r);

/* ############################################################################# */
/** Functions on columns of the matrix */

// compute 'p' norm of given column
double dccs_row_norm (const dccs_matrix*, size_t r, double p);
// compute dot product of row_r1 with row_r2
double dccs_row_dot  (const dccs_matrix*, size_t r1, size_t r2);
// compute dot product of row r with vector v
double dccs_row_dotv (const dccs_matrix*, size_t r, vector* v);
// TODO: row r1 <- r1 + r2 (causes potential fill-in)
int    dccs_row_add  (dccs_matrix*, size_t r1, size_t r2);
// TODO: r1 <- r1 - r2
int    dccs_row_sub  (dccs_matrix*, size_t r1, size_t r2);
// r1 <- s * r1
int    dccs_row_scale(dccs_matrix*, size_t r1, double s);
// TODO: r1 <- r1 - v
int    dccs_row_subv  (dccs_matrix*, size_t r1, vector* v);
// TODO: r1 <- r1 + v
int    dccs_row_addv  (dccs_matrix*, size_t r1, vector* v);
// r <- s * r1
int    dccs_row_scalev(dccs_matrix*, size_t r1, double s, vector* r);
// compute sum of each column and store in the vector
int    dccs_row_sum  (dccs_matrix*, vector* r);
// r <- r + a*row_r
int    dccs_row_daxpy(dccs_matrix*, size_t ro, double a, vector* r);


/* ############################################################################# */
/** Important Matrix Vector operations */

// r <- Trans(matrix)*x
int    dccs_gemv (dccs_matrix*, int transpose, vector* x, vector* r); 


/* ############################################################################# */
/** Matrix-Matrix operations */

// Matrix x Matrix, i.e., C <- A * B
int    dccs_gemm(dccs_matrix* C, const dccs_matrix* A, const dccs_matrix* B);
// More memory efficient  A = A * B
int    dccs_gemm2(dccs_matrix* A, const dccs_matrix* B);
// Transpose operation
int    dccs_transpose_copy(dccs_matrix* At, const dccs_matrix* A);
// TODO: In place transpose
int    dccs_transpose     (dccs_matrix* A);

/* ############################################################################# */
/** Spectral operations */

// Compute the top 'k' eigenvectors, store in a dense matrix (mode determines several things)
int    dccs_eigs(dccs_matrix* A, gsl_matrix* V, vector* d, int k, int mode);
// Compute top 'k' singular values, store left and right sing vectors dep. on mode
int    dccs_svds(dccs_matrix* A, gsl_matrix* U, gsl_matrix* V, int k, int mode);


/* ############################################################################# */
/** Miscellaneous operations */

// Compute dot product of row r with column c
double dccs_row_col_dot (dccs_matrix*, size_t r, size_t c);


#endif // _SAGE_CCS_H
