// File: row_coord.cc
// Author: Suvrit Sra
// Time-stamp: <23 February 2010 10:43:28 AM CET --  suvrit>
// Copyright (C) 2005 Suvrit Sra (suvrit@cs.utexas.edu)
// Copyright The University of Texas at Austin

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

// Implements the row coordinate based class.

#include <iostream>
#include <cstdio>
#include <cstring>
#include "row_coord.h"

/**
 * Assumes file is stored in row coordinate format. That is:
 * row1:val1 row2:val2 .... rowk:valk
 * rowx:valx rowy:valy .... rowj:valj
 * and so on...each line begins a new row of the matrix. If a row is
 * totally empty, then there should be a blank line in the file to
 * indicate that. The loaded matrix is stored internally as a CRS matrix.
 * The indices are expected to begin from '0'
 * @todo  Introduce 'base' for indices, to allow dealing with matlab
 * matrices.
 * @todo To write this function.
 */
int SSLib::row_coord::load(const char* fil)
{

  // NOTE!!!!!!!!!!!!!!!!!!!!!!!!!
  // As of now, just calling load_fast
  //std::cerr << "int SSLib::row_coord::load(const char* fil)\n";

  std::string s = std::string(fil) + ".dim";
  FILE* fp = fopen(s.c_str(), "r");
  if (!fp) {
    error("Could not load " + std::string(fil));
    return 1;
  }
  
  size_t m, n, nz;
  size_t st;
  st=fscanf(fp, "%zu %zu %zu", &m, &n, &nz);
  fclose(fp);
  //  std::cerr << m << " " << n << " " << nz << "\n";
  return load_fast(m,n,nz,fil,1,":");
}


/**
 * See comments above for file format.
 * Default: indices are expected to start at '0' and not '1' like
 * matlab. The parameter 'base' can be used to control that. Base = 1 gives
 * matlab indexing and base = 0 is the 'C' style indexing.
 * @param separator tells which character to use as the separator between a
 * row-index and the associated value. Common values are ':' or ' '
 */
int SSLib::row_coord::load_fast (size_t m, size_t n, size_t nz, 
                                 const char* fil, int base, char* separator)
{
  FILE* fp = fopen(fil, "r");
  
  if (!fp) {
    error("Could not load " + std::string(fil));
    return 1;
  }

  matrix_alloc(m, n, nz);

  size_t idx = 0;
  m_colptrs[0] = 0;             // Init it!
  long col = 0, row = 0;        // Start at 0 or 1
  double val;

  char* line = 0;
  size_t len = 0;
  int read = -1;

  char* fmtstr = "%d%c%lf";
  char jnk;
  char* blk;
  
  while (!feof(fp)) {
    // Read a line == row of the file and process it.
    read = getline(&line, &len, fp);
    if (read > 0) {
      blk = strtok(line, " ");
      if (blk != NULL) {
        sscanf(blk, fmtstr, &row, &jnk, &val);
        //std::cerr << "(" << row << "," << row - base << "," << val << ") ";
        m_rowindx[idx] = row - base;
        m_val[idx] = val;
        ++idx;
      }
      while ( (blk = strtok(0, " ")) != NULL) {
        sscanf(blk, fmtstr, &row, &jnk, &val);
        //std::cerr << "(" << row << "," << row - base<< "," << val << ") ";
        m_rowindx[idx] = row - base;
        m_val[idx] = val;
        ++idx;
      }
      if (line) { free(line); line = 0;}
      len = 0;
      blk = 0; read = -1;
    }
    col++; m_colptrs[col] = idx;
  }

  m_colptrs[m] = nz;
  //fclose(fp);   CHK...there's a memory leak around...
  //std::cerr << "Loaded: " << m << " " << n << " " << nz << "\n";
  return 0;
}

int SSLib::row_coord::load_bin(const char* fil)
{
  return -1;
}

int SSLib::row_coord::load_txt(const char* fil)
{
  return load(fil);
}

int SSLib::row_coord::load(const char* fil, bool asbin)
{
  if (asbin)
    return load_bin(fil);
  return load_txt(fil);
}

void SSLib::row_coord::error (std::string s)
{
  if (s != "")
    std::cerr << "ROWCOORD:"  << s << std::endl;
}

