// File: matrix_compact_float.h        -*- c++ -*-
// Author: Suvrit Sra
// Time-stamp: <14 Februar 2008 02:15:05  CET --  suvrit>
// Base class for both sparse and dense matrices....

#ifndef _MY_MATRIX_COMPACT_FLOAT_BASE_H
#define _MY_MATRIX_COMPACT_FLOAT_BASE_H

#include "matrix_common.h"
#include <sys/types.h>
#include <iostream>
#include <gsl/gsl_vector_float.h>
#include <gsl/gsl_vector_uint.h>

/*! \mainpage SSLib: A libary for manipulating sparse (and even dense) matrices.
 * \section Abstract
 * This document describes SSLib, a library for sparse matrix
 manipulations. Additionally, it includes a description of a sample driver
 program that accompanies this library. The library includes four basic
 sparse matrix types. The Harwell-Boeing and the related Compressed Column
 Storage formats; Compressed Row storage and Matlab sparse matrices
 (internal as well as the $(i,j,a_{ij})$ format). The library offers
 methods to load and save such files, perform conversions between the
 different formats and do basic linear algebra operations.
 *
 * \author Suvrit Sra
 * \section Notes
 * Please see the driver.h and driver.cc files to see how to use the sparse
 * library. The class SSLib::ccs is of central importance, and it forms the
 * backbone of the SSLib::harwell_boeing and SSLib::matlab classes.
 */

#ifndef vector_free
#define vector_free(x) gsl_vector_float_free( (x) );
#define vector_alloc(n)  gsl_vector_float_alloc((n))
#define vector_calloc(n) gsl_vector_float_calloc((n))
#endif

/**
 * @namespace SSLib
 * This namespace contains my matrix class, its derivatives dense_matrix and
 * sparsematrix. Essentially, SSLib was made to hold only sparsematrices,
 * but often in my own programs, it is easier if both dense and sparse
 * matrices are derived from the same 'matrix' base class. Thus SSLib holds
 * 'matrix' and its derived classes.
 */
namespace SSLib {
  typedef gsl_vector_float vector_float;
  typedef unsigned short int index_t;
  class matrix_compact_float;
  
#ifdef _INTERNAL_EIGS_
  struct eigenpairs {
    matrix* eigvectors;
    vector_float* eigvalues;
  };

  struct singular_triples {
    matrix* left_vectors;
    matrix* right_vectors;
    vector_float* singular_values;
  };

#endif 

  /**
   * @class matrix_compact
   * This is the base class for all the matrices that i use in my code. It
   * is an abstract class and it provides the interface that  is needed for
   * working with matrices. Will augment this class as necessary. To save
   * space, might create a template class???
   */
  class matrix_compact_float {
    size_t size1;               /// Num of rows
    size_t size2;               /// Num of colums
    matrix_type mtype;          /// What type of matrix this is.
    storage_type stype;
    std::string myname;         /// Name of the matrix.
    bool nonneg;
    int base;                   // base = 0 or 1, default is 0
  public:

    /// Create an empty (unallocated) matrix of specified size.
    matrix_compact_float (size_t rows, size_t cols) {
      size1 = rows;
      size2 = cols;
      base = 0;
      nonneg = false;
    }
    
    virtual ~matrix_compact_float() { size1 = 0; size2 = 0;}

    /** @name Basic_Matrix_Features
     * Functions discussing basic features of matrices
     */

    /*@{*/

    /// Returns number of rows. 
    size_t nrows() const { return size1;}

    /// Returns number of colums
    size_t ncols() const { return size2;}

    /// Sets the row and column dimensionality of the matrix
    void matrix_setsize(size_t m, size_t n) {
      size1 = m;
      size2 = n;
    }

    /// get base
    int get_base() const { return base;}
    void set_base (int b) { base = b;}

    bool is_non_negative() const { return nonneg;}
    void set_nonnegative(bool n) { nonneg = n;}

    /// Gets the name of the matrix
    std::string get_name() const { return myname;}

    /// Set the matrix name
    void set_name(std::string s) { myname = s;}

    /// Unallocate/free the memory held by the matrix
    virtual void free_data() = 0;

    /// Return the type of the matrix. Default type is unspecified, unless
    /// explicitly set.
    int get_type() { return mtype;}

    virtual float* get_data() = 0;
    int get_storage_type() { return stype;}

    /// Sets the type of the matrix
    void set_matrix_type(matrix_type ty) {  mtype = ty;   }
    void set_storage_type(storage_type ty) { stype = ty;}

    /*@}*/


    /** @name Accessors_Mutators
     * All these functions are pure virtual functions, and should be
     * implemented by any derived class.
     */
    
    /*@{*/
    
    /**
     * Elementwise accessors/mutators.
     * -------------------------------
     */

    /*@{*/

    /// Get the (i,j) entry of the matrix
    virtual float operator()   (size_t i, size_t j            ) = 0;
    /// Get the (i,j) entry of the matrix
    virtual float get          (size_t i, size_t j            ) = 0;

    /// Set the (i,j) entry of the matrix. Not all matrices can support this.
    virtual int set            (size_t i, size_t j, float val) = 0;

    /*@}*/

    /**
     * Vector level accessors/mutators
     * ---------------------------------
     */

    /*@{*/
    /// Returns 'r'-th row as a vector
    virtual int get_row (size_t r, vector_float*& r) = 0;
    /// Returns 'c'-th col as a vector
    virtual int get_col (size_t c, vector_float*& c) = 0;
    /// Returns main or second diagonal (if p == true)
    virtual int get_diag(bool p, vector_float*& d  ) = 0;
    
    /// Sets the specified row to the given vector
    virtual int set_row(size_t r, vector_float*&) = 0;
    /// Sets the specified col to the given vector
    virtual int set_col(size_t c, vector_float*&) = 0;
    /// Sets the specified diagonal to the given vector
    virtual int set_diag(bool p, vector_float*&)  = 0;
    /*@}*/

    /**
     * Matrix level accessors/mutators
     */
    /*@{*/
    /// Returns a submatrix that goes from row i1--i2 and cols j1--j2
    virtual int submatrix(size_t i1, size_t j1, size_t i2, size_t j2, matrix_compact_float*&) = 0;
    /// Returns a submatrix given the row-col index sets
    virtual int submatrix(gsl_vector_uint* I, gsl_vector_uint* J, matrix_compact_float*&)     = 0;
    /*@}*/

    /*@}*/


    /** ****************************************************************
     *  Matrix Operations begin here. They are group separately into
     *  'self (essentially elementwise operations on itself)'
     *  'matrix-vector operations'
     *  'matrix-matrix operations'
     *  ****************************************************************
     */

    /** @name Elementwise_Operations 
     * Elementwise operations on the elements of this matrix.
     */
    /*@{*/
    /// Vector l_p norms for this matrix, p > 0
    virtual float norm (float p) = 0;
    /// Vector l_p norms, p is 'l1', 'l2', 'fro', 'inf'
    virtual float norm (char*  p) = 0;

    /// Apply an arbitrary function elementwise to this matrix. Changes the matrix.
    virtual float apply (float (* fn)(float)) = 0;

    /// Scale the matrix so that x_ij := s * x_ij
    virtual int scale (float s) = 0;

    /// Adds a const 's' so that x_ij := s + x_ij
    virtual int add_const(float s) = 0;
    /*@}*/

    
    /** @name Row_Col_Operations
     *  ****************************************************
     * Operations on rows and columns of this matrix
     *  ****************************************************
     */
    /*@{*/

    /// The column norm
    virtual float col_norm(size_t c) = 0;
    /// Compute the vector l_p norm of column c
    virtual float col_norm (size_t c, float p) = 0;
    /// Compute dot product of column c1 with c2
    virtual float col_dot  (size_t c1, size_t c2)  = 0;
    /// Compute dot product of column c with vector v
    virtual float col_dot  (size_t c, vector_float* v) = 0;
    /// Adds columns c2 to column c1. c1 := c1 + c2
    virtual int    col_add  (size_t c1, size_t c2)  = 0;
    /// Subtracts column c2 from c1. c1 := c1 - c2
    virtual int    col_sub  (size_t c1, size_t c2)  = 0;
    /// Scales column c1. c1 := s * c1
    virtual int    col_scale(size_t c1, float s) = 0;
    /// Subtracts vector v from column c1
    virtual int    col_sub  (size_t c1, vector_float* v) = 0;
    /// Adds vector v to column c1
    virtual int    col_add  (size_t c1, vector_float* v)  = 0;
    
    /// Do colum sums
    virtual int col_sum(vector_float* v) = 0;

    /// Scales column c1 by 's' and stores result in 'r'
    virtual int    col_scale(size_t c1, float s, vector_float* r) = 0;

    /// The l2 norm
    virtual float row_norm(size_t r) = 0;
    /// Compute the vector l_p norm of row r
    virtual float row_norm (size_t r, float p) = 0;
    /// Compute dot product of row r1 with r2
    virtual float row_dot  (size_t r1, size_t r2)  = 0;
    /// Compute dot product of row r with vector v
    virtual float row_dot  (size_t r, vector_float* v) = 0;
    /// Compute dot product using x(idx) components
    virtual float row_dot(size_t r, vector_float* x, size_t* idx, size_t) = 0;
    /// Adds row r2 to row r1. r1 := r1 + r2
    virtual int    row_add  (size_t r1, size_t r2)  = 0;
    /// Subtracts row r2 from r1.     r1 := r1 - r2
    virtual int    row_sub  (size_t r1, size_t r2)  = 0;
    /// Scales column r1.       r1 := s * r1
    virtual int    row_scale(size_t r1, float s) = 0;
    /// Adds vector v to row r1
    virtual int    row_add  (size_t r1, vector_float* r) = 0;
    /// Subtracts vector v from row r1
    virtual int    row_sub  (size_t r1, vector_float* r) = 0;
    /// Scales row r = s*row(r1)

    /// Do row sums
    virtual int row_sum(vector_float* v) = 0;

    virtual int    row_scale(size_t r1, float s, vector_float* r) = 0;

    /// r = a*row(i) + r
    virtual int    row_daxpy(size_t i, float a, vector_float* r) = 0;


    /// dot ( row i, col j)
    virtual float row_col_dot (size_t r, size_t c) = 0;

    /*@}*/

    /** @name Matrix_Vector_Operations
     * Matrix - vector operations
     */
    /*@{*/
    
    /// Let r := this * x or  this^T * x depending on tranA
    virtual int dot (bool, vector_float*, vector_float*) = 0;

    /*@}*/
    
    /** @name Matrix_Matrix_Operations 
     * Matrix - Matrix operations. BLAS3 stuff.
     */

    /*@{*/
    /// this := this + b  where 'b' is a matrix. Note, each derived class
    /// will need to use RTTI to determine type of 'b' while doing the
    /// addition if efficient computation is needed.
    virtual int add (matrix_compact_float* b) = 0;

    /// this <- this - b where 'b' is a matrix.
    virtual int sub (matrix_compact_float* b) = 0;

    /// this <- this .* b  computes the Hadamard product.
    virtual int dot (matrix_compact_float* b) = 0;

    /// this <- this ./ b computes Hadamard division
    virtual int div (matrix_compact_float* b) = 0;
    
    /// this <- this * b
    virtual int mul (matrix_compact_float* b) = 0;

    /// Computes an inplace transpose, destroying the original
    virtual void transpose() = 0;
    /// Creates a transpose without destroying the original
    virtual matrix_compact_float* transpose_copy() = 0;


    /// Normalize the columns of the matrix to have unit L2 norm
    virtual void normalize_columns() = 0;

    /// Print out to file or stdout or whatever
    virtual void print() = 0;

#ifdef _INTERNAL_EIGS_
    /// Returns top 'k' eigenpairs 
    virtual eigenpairs* eigs (size_t k) = 0;
    
    /// Returns singular triplets
    virtual singular_triples* svds (size_t k) = 0;

#endif

    virtual int compute_AtA(matrix_compact_float* result) = 0;
    /*@}*/


    /** @name Utility_and_IO_Functions
     * Utility functions. Most important amongst these are the I/O functions.
     */

    /*@{*/
  
    /// DEPRECATED: Superflous error message function.
    virtual void matrix_error (std::string s) {  std::cerr << s << std::endl;  } 

    virtual void error(std::string s) = 0;

    /// Load matrix from specified file in either text or binary mode
    virtual int    load(const char*, bool asbin) = 0;

    /// Write out matrix to disk as a text or binary file
    virtual int    save(const char*, bool asbin) = 0;
    
    /// Write out matrix in a specified way
    virtual int    save(const char*, bool asbin, int typ) = 0;
    
    /// Check if the dimensions of 'this' equal those of 'b'
    bool dimension_match (matrix_compact_float* b);

    /// Can 'this' be multiplied with 'b' or b.transpose etc.
    bool can_be_multiplied (matrix_compact_float* b, bool tranA, bool tranB);
    
    /*@}*/

  protected:
    /** @name Default_operations
     * Default functions, in case specializations don't exist.
     * All of these functions use the operator(i,j) or the get(i,j)
     * functions to implement the appropriate behavior. Useful for testing,
     * but should not really be used if the more efficient versions can be
     * invoked by doing RTTI.
     */
    /*@{*/
    int default_add (matrix_compact_float* b);
    int default_sub (matrix_compact_float* b);
    int default_mul (matrix_compact_float* b, bool tranA, bool tranB);
    int default_dot (matrix_compact_float* b);
    int default_div (matrix_compact_float* b);
    /*@}*/
  };

}
#endif
