// File: matrix_base.h        -*- c++ -*-
// Author: Suvrit Sra
// Time-stamp: <27 April 2006 02:40:27 PM CDT --  suvrit>
// Base class for both sparse and dense matrices....

#ifndef _MY_MATRIXBASE_BASE_H
#define _MY_MATRIXBASE_BASE_H

#include <sys/types.h>
#include <iostream>

#include "vector_base.h"

namespace SSLib {
  /// Types of matrices. As of now, not really used anywhere in the library
  /// itself, but can be exploited by an external program.
  enum matrix_type {
    ZEROS,
    CONST,
    IDENTITY,
    DIAGONAL,
    SYMMETRIC,
    LOWERTRI,
    UPPERTRI,
    UPPERHESS,
    LOWERHESS,
    TRIDIAG,
    BIDIAG,
    GENERAL,
    NONNEG
  };

  class matrix_base;

  /**
   * @class matrix_base
   * This is the base class for all the matrices that i use in my code. It
   * is an abstract class and it provides the interface that  is needed for
   * working with matrices. Will augment this class as necessary. To save
   * space, might create a template class??? but i don't like c++ templates...
   */
  class matrix_base {
  private:
    bool allocated;
    size_t size1;               /// Num of rows
    size_t size2;               /// Num of colums
    matrix_type mtype;          /// What type of matrix this is.
    std::string myname;         /// Name of the matrix.
  public:
    /// Create an empty (unallocated) matrix of specified size.
    matrix_base (size_t rows, size_t cols) {
      size1 = rows;
      size2 = cols;
      allocated = false;
      myname = "Empty Matrix";
    }
    
    virtual ~matrix_base() { allocated = false; myname = "";}

    /** @name Basic_Matrix_Features
     * Functions discussing basic features of matrices
     */

    /*@{*/

    /// Returns number of rows. 
    size_t nrows() const { return size1;}

    /// Returns number of colums
    size_t ncols() const { return size2;}

    /// Sets the row and column dimensionality of the matrix
    void matrix_setsize(size_t m, size_t n) {
      size1 = m;
      size2 = n;
    }

    /// Gets the name of the matrix
    std::string get_name() const { return myname;}

    /// Set the matrix name
    void set_name(std::string s) { myname = s;}

    /// Alloc/Unallocate the memory held by the matrix
    virtual matrix_base* alloc_matrix () = 0;
    virtual void free_matrix() = 0;

    /// Return the type of the matrix. Default type is unspecified, unless
    /// explicitly set.
    int get_type() { return mtype;}

    /// Sets the type of the matrix
    void set_matrix_type(matrix_type ty) {  mtype = ty;   }

    /// Used commonly, hence has a separate function here
    bool is_non_negative() { return mtype == NONNEG;}

    /*@}*/

    /**
     * Vector level accessors/mutators
     */
    /*@{*/
    /// Returns 'r'-th row as a vector
    virtual vector_base* get_row (size_t r) = 0;
    /// Returns 'c'-th col as a vector
    virtual vector_base* get_col (size_t c) = 0;
    /// Returns main or second diagonal (if p == true)
    virtual vector_base* get_diag(bool p  ) = 0;
    
    /// Sets the specified row to the given vector
    virtual int set_row(size_t r, vector_base*) = 0;
    /// Sets the specified col to the given vector
    virtual int set_col(size_t c, vector_base*) = 0;
    /// Sets the specified diagonal to the given vector
    virtual int set_diag(bool p, vector_base*)  = 0;
    /*@}*/

    /**
     * Matrix level accessors/mutators
     */
    /*@{*/
    /// Returns a submatrix that goes from row i1--i2 and cols j1--j2
    virtual matrix_base* submatrix(size_t i1, size_t j1, 
                                   size_t i2, size_t j2) = 0;
    /*@}*/

    /*@}*/


    /** ****************************************************************
     *  Matrix Operations begin here. They are group separately into
     *  'self (essentially elementwise operations on itself)'
     *  'matrix-vector operations'
     *  'matrix-matrix operations'
     *  ****************************************************************
     */

    /** @name Elementwise_Operations 
     * Elementwise operations on the elements of this matrix.
     */
    /*@{*/
    /// Vector l_p norms for this matrix, p > 0
    virtual double norm (double p) = 0;
    /// Vector l_p norms, p is 'l1', 'l2', 'fro', 'inf'
    virtual double norm (char*  p) = 0;

    /*@}*/

    
    /** @name Row_Col_Operations
     *  ****************************************************
     * Operations on rows and columns of this matrix
     *  ****************************************************
     */
    /*@{*/
    /// Compute the vector l_p norm of column c
    virtual double col_norm (uint c, double p) = 0;
    /// Compute dot product of column c1 with c2
    virtual double col_dot  (uint c1, uint c2)  = 0;
    /// Compute dot product of column c with vector v
    virtual double col_dot  (uint c, vector_base* v) = 0;
    /// Adds columns c2 to column c1. c1 := c1 + c2
    virtual int    col_add  (uint c1, uint c2)  = 0;
    /// Subtracts column c2 from c1. c1 := c1 - c2
    virtual int    col_sub  (uint c1, uint c2)  = 0;
    /// Subtracts vector v from column c1
    virtual int    col_sub  (uint c1, vector_base* v) = 0;
    /// Adds vector v to column c1
    virtual int    col_add  (uint c1, vector_base* v)  = 0;

    /// Compute the vector l_p norm of row r
    virtual double row_norm (uint r, double p) = 0;
    /// Compute dot product of row r1 with r2
    virtual double row_dot  (uint r1, uint r2)  = 0;
    /// Compute dot product of row r with vector v
    virtual double row_dot  (uint r, vector* v) = 0;
    /// Adds row r2 to row r1. r1 := r1 + r2
    virtual int    row_add  (uint r1, uint r2)  = 0;
    /// Subtracts row r2 from r1.     r1 := r1 - r2
    virtual int    row_sub  (uint r1, uint r2)  = 0;
    /// Scales column r1.       r1 := s * r1
    virtual int    row_scale(uint r1, double s) = 0;
    /// Adds vector v to row r1
    virtual int    row_add  (uint r1, vector* r) = 0;
    /// Subtracts vector v from row r1
    virtual int    row_sub  (uint r1, vector* r) = 0;
    /// Scales row r = s*row(r1)
    virtual int    row_scale(uint r1, double s, vector* r) = 0;

    /// r = a*row(i) + r
    virtual int    row_daxpy(uint i, double a, vector* r) = 0;

    /*@}*/

    /** @name Matrix_Vector_Operations
     * Matrix - vector operations
     */
    /*@{*/
    
    /// Let r := this*x or this^T * x
    ///virtual int dot(bool tranA, double* x, double* r) = 0;

    /// Let r := this * x or  this^T * x depending on tranA
    virtual int dot (bool tranA, vector* x, vector* r) = 0;
    /// Let r := this*x + y  or  this^T*x + y
    /*@}*/
    
    /** @name Matrix_Matrix_Operations 
     * Matrix - Matrix operations. BLAS3 stuff.
     */

    /*@{*/
    /// this := this + b  where 'b' is a matrix. Note, each derived class
    /// will need to use RTTI to determine type of 'b' while doing the
    /// addition if efficient computation is needed.
    virtual int add (matrix* b) = 0;

    /// this <- this - b where 'b' is a matrix.
    virtual int sub (matrix* b) = 0;

    /// this <- this .* b  computes the Hadamard product.
    virtual int dot (matrix* b) = 0;

    /// this <- this ./ b computes Hadamard division
    virtual int div (matrix* b) = 0;
    
    /// Computes an inplace transpose, destroying the original
    virtual void transpose() = 0;
    /// Creates a transpose without destroying the original
    virtual matrix* transpose_copy() = 0;

    /// Print out to file or stdout or whatever
    virtual void print() = 0;

#ifdef _INTERNAL_EIGS_
    /// Returns top 'k' eigenpairs 
    virtual eigenpairs* eigs (uint k) = 0;
    
    /// Returns singular triplets
    virtual singular_triples* svds (uint k) = 0;

#endif

    /*@}*/


    /** @name Utility_and_IO_Functions
     * Utility functions. Most important amongst these are the I/O functions.
     */

    /*@{*/
  
    /// DEPRECATED: Superflous error message function.
    void matrix_error (std::string s) {  std::cerr << s << std::endl;  }

    /// Load matrix from specified file
    virtual int    load(const char*) = 0;

    /// Load matrix from specified file in either text or binary mode
    virtual int    load(const char*, bool asbin) = 0;

    /// Write out matrix to disk as a text or binary file
    virtual int    save(const char*, bool asbin) = 0;

    /// Write out matrix to disk in CCS, CRS, etc. formats
    virtual int save_as_ccs(const char*) = 0;
    virtual int save_as_crs(const char*) = 0;
    virtual int save_as_coord(const char*) = 0;
    virtual int save_as_col_coord(const char*) = 0;
    virtual int save_as_matlab(const char*) = 0;
    virtual int save_as_hb(const char*) = 0;
    virtual int save_as_dense(const char*) = 0;
    
    /// Check if the dimensions of 'this' equal those of 'b'
    bool dimension_match (matrix* b);

    /// Can 'this' be multiplied with 'b' or b.transpose etc.
    bool can_be_multiplied (matrix* b, bool tranA, bool tranB);
    
    /*@}*/

  protected:
    /** @name Default_operations
     * Default functions, in case specializations don't exist.
     * All of these functions use the operator(i,j) or the get(i,j)
     * functions to implement the appropriate behavior. Useful for testing,
     * but should not really be used if the more efficient versions can be
     * invoked by doing RTTI.
     */
    /*@{*/
    int default_add (matrix* b);
    int default_sub (matrix* b);
    int default_mul (matrix* b, bool tranA, bool tranB);
    int default_dot (matrix* b);
    int default_div (matrix* b);
    /*@}*/
  };

}
#endif
