// File: eigs.cc
// Author: Suvrit Sra
// Time-stamp: <23 February 2010 05:58:48 PM CET --  suvrit>
// Shows how to use the sparselib with ARPACK to compute eigs of a symmetric 
// CCS sparse matrix
// Copyright (C) 2005 Suvrit Sra (suvrit@cs.utexas.edu)

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


#include "util.h"
#include "ccs.h"

extern "C" {
#include <f2c.h>
#include "carpack.h"

  /// Junk needed for ARPACK
  struct {
    integer logfil, ndigit, mgetv0, msaupd, msaup2, msaitr, mseigt, msapps,
      msgets, mseupd, mnaupd, mnaup2, mnaitr, mneigh, mnapps, mngets,
      mneupd, mcaupd, mcaup2, mcaitr, mceigh, mcapps, mcgets, mceupd;
  } debug_;
  
#define debug_1 debug_

}

#include <iostream>
#include <cstdlib>
#include <gsl/gsl_vector.h>
#include <gsl/gsl_matrix.h>

int compute_eigs (int lenAF, SSLib::ccs* mat, int k, gsl_vector*&, gsl_matrix*&, gsl_matrix_view&);

int main (int argc, char** argv) 
{

  gsl_vector* eigen_values;
  gsl_matrix* eigen_vectors;
  gsl_matrix_view view;
  int lenAF;
  FILE* fp;
  std::string fname;

  if (argc != 5) {
    std::cerr << "Usage: eigs ccsPrefix ccsType K lenAF\n";
    return -1;
  }

  int st;
  SSLib::ccs* A = new SSLib::ccs();
  A->set_type(std::string(argv[2]));
  if (A->get_type()=="bin") {
    std::cerr << "Loading " << argv[1] << " as binary CCS matrix\n";
    st=A->load(argv[1], true);}
  else
    st=A->load(argv[1], false);
  
  if (st) {
    std::cerr << "Error loading " << argv[1] << " quitting\n";
    return -1;
  }

  lenAF=atoi(argv[4]);
  fname=std::string(argv[1]);
  fname = fname + ".arpackVals";
  std::cerr << "Beginning the eigendecomposition" << std::endl;

  st=compute_eigs (lenAF, A, atoi(argv[3]), eigen_values, eigen_vectors, view);
  if (st >= 0) {
    SSUtil::print_vector(eigen_values);
    fp = fopen(fname.c_str(), "w");
    for (size_t i = 0; i < eigen_values->size; i++)
      fprintf(fp, "%.14lf\n", eigen_values->data[i]);
    fclose(fp);
  }
  fname=std::string(argv[1]);
  fname = fname + ".arpackVecs";
  //SSUtil::write_gsl_matrix(eigen_vectors, const_cast<char*>(fname.c_str()));
  return 0;

}

int compute_eigs (int lenAF, SSLib::ccs* mat, int k, 
                  gsl_vector*& ew, gsl_matrix*& ev, gsl_matrix_view& view)
{

  std::string errorList[16] = {
    "Normal exit.",
    "N must be positive.",
    "NEV must be positive.",
    "NCV must be greater than NEV and less than or equal to N.",
    "WHICH must be one of 'LM', 'SM', 'LA', 'SA' or 'BE'.",
    "BMAT must be one of 'I' or 'G'.",
    "Length of private work WORKL array is not sufficient.",
    "Error return from trid. eigenvalue calculation"
    "Information error from LAPACK routine dsteqr.",
    "Starting vector is zero.",
    "IPARAM(7) must be 1,2,3,4,5.",
    "IPARAM(7) = 1 and BMAT = 'G' are incompatible.", 
    "NEV and WHICH = 'BE' are incompatible.", 
    "DSAUPD did not find any eigenvalues to sufficient accuracy.",
    "HOWMNY must be one of 'A' or 'S' if RVEC = .true.",
    "HOWMNY = 'S' not yet implemented"
  };

  integer nev = k;                  // Number of eigenvalues
  integer ncv = min(mat->nrows(), lenAF);             // Length of Arnoldi factorization
  integer maxncv = min(mat->nrows(), ncv+2);         // max
  char bmat = 'I';              // Standard eigenvalue problem
  char* which = "LM";           // K eigs of largest magnitude

  /// Do some more error checking here....
  integer n = mat->nrows();
  integer maxn = n;

  doublereal* workl = new doublereal[3*maxncv*maxncv + 6*maxncv];
  doublereal* d     = new doublereal[3*maxncv];
  doublereal* resid = new doublereal[maxn];
  doublereal* v     = new doublereal[maxn*maxncv];
  doublereal* workd  = new doublereal[3*maxn];

  gsl_vector v1, v2;

  doublereal tol = 1.0e-1;
  integer ido = 0;
  integer info = 0;
  integer ierr;
  integer ishfts = 1;
  integer maxitr = 300;
  integer model = 1;
  integer iparam[11];
  integer ipntr[14];
  logical select[maxncv];
  integer ldv = maxn;
  integer lworkl = ncv * (ncv + 8);
  //integer nx = n;
  doublereal sigma;

  iparam[0] = ishfts;
  iparam[2] = maxitr;
  iparam[6] = model;
  

  /*     %-------------------------------------------% */
  /*     | M A I N   L O O P (Reverse communication) | */
  /*     %-------------------------------------------% */

  debug_1.ndigit = -3;
  debug_1.logfil = 6;
  debug_1.msgets = 0;
  debug_1.msaitr = 0;
  debug_1.msapps = 0;
  debug_1.msaupd = 1;
  debug_1.msaup2 = 0;
  debug_1.mseigt = 0;
  debug_1.mseupd = 0;
  
 L10:

  /*        %---------------------------------------------% */
  /*        | Repeatedly call the routine DNAUPD and take | */
  /*        | actions indicated by parameter IDO until    | */
  /*        | either convergence is indicated or maxitr   | */
  /*        | has been exceeded.                          | */
  /*        %---------------------------------------------% */

  dsaupd_(&ido, &bmat, &n, which, &nev, &tol, resid, &ncv, v, &ldv,
          iparam, ipntr, workd, workl, &lworkl, &info, (ftnlen)1, (ftnlen)2);

  if (ido == -1 || ido == 1) {

    /*           %-------------------------------------------% */
    /*           | Perform matrix vector multiplication      | */
    /*           |                y <--- Op*x                | */
    /*           | The user should supply his/her own        | */
    /*           | matrix vector multiplication routine here | */
    /*           | that takes workd(ipntr(1)) as the input   | */
    /*           | vector, and return the matrix vector      | */
    /*           | product to workd(ipntr(2)).               | */
    /*           %-------------------------------------------% */
    
    v1.data = &workd[ipntr[0] - 1];
    v2.data = &workd[ipntr[1] - 1];
    v1.size = n;
    v2.size = n;

    // This does v2 := A.v1 
    mat->dot(true, &v1, &v2);
    
    /*           %-----------------------------------------% */
    /*           | L O O P   B A C K to call DNAUPD again. | */
    /*           %-----------------------------------------% */
    
    goto L10;
    
  }

  /*     %----------------------------------------% */
  /*     | Either we have convergence or there is | */
  /*     | an error.                              | */
  /*     %----------------------------------------% */
  
    if (info < 0) {
      
      /*        %--------------------------% */
      /*        | Error message, check the | */
      /*        | documentation in DNAUPD. | */
      /*        %--------------------------% */
      
      std::cerr << "ARPACK ERROR: " << errorList[-info] << std::endl;
      /// TODO: Clean up the allocated memory
      delete[] workl;
      delete[] workd;
      delete[] resid;
      delete[] v;
      delete[] d;
      return info;
    } else {
      logical rvec = TRUE_; ierr=0;
      dseupd_(&rvec, "All", select, d, v, &ldv, &sigma, &bmat, &n, which, &nev, &tol, resid, &ncv, v, &ldv, iparam, ipntr, workd, workl, &lworkl, &ierr, (ftnlen)1, (ftnlen)1, (ftnlen)2);
    }
    
    /*         %----------------------------------------------% */
    /*         | Eigenvalues are returned in the first column | */
    /*         | of the two dimensional array D and the       | */
    /*         | corresponding eigenvectors are returned in   | */
    /*         | the first NCONV (=IPARAM(5)) columns of the  | */
    /*         | two dimensional array V if requested.        | */
    /*         | Otherwise, an orthogonal basis for the       | */
    /*         | invariant subspace corresponding to the      | */
    /*         | eigenvalues in D is returned in V.           | */
    /*         %----------------------------------------------% */
    
    
    if (ierr != 0) {
      
      /*            %------------------------------------% */
      /*            | Error condition:                   | */
      /*            | Check the documentation of DSEUPD. | */
      /*            %------------------------------------% */

      std::cerr << "Got back IERR = " << ierr << std::endl;
      /// Cleanup
      return -1;
    } else {
      ew = gsl_vector_alloc(nev);
      for (int i = 0; i < nev; i++)
        ew->data[i] = d[i];

      delete[] d;
      delete[] workl;
      delete[] workd;
      delete[] resid;
      
      view = gsl_matrix_view_array(v, iparam[4], n);
      ev = &view.matrix;
    }

    return 0;
}

extern "C" int MAIN__()
{ return 0;
}
