// File: driver.cc
// Author: Suvrit Sra
// Time-stamp: <23 February 2010 11:10:09 AM CET --  suvrit>
// Driver file for sparse matrix stuff

#include <ctime>
#include <cstdlib>
#include <string>
#include <iostream>
#include <cstdio>
#include <readline/readline.h>
#include <readline/history.h>
#include "driver.h"
#include "dense_matrix.h"
#include "util.h"
#include "ccs_float.h"
#include "crs_float.h"
#include "harwell_boeing_float.h"
#include "col_coord_float.h"
#include "coord_float.h"
#include "coord_ccs.h"
#include "coord_crs_float.h"
#include "coord_crs.h"
#include "row_coord.h"
#include "row_coord_float.h"
#include "col_coord.h"
#include "col_coord_float.h"


extern "C" {
#include <f2c.h>

int dsaupd_ (integer *ido, char *bmat, integer *n, char *which, integer *nev,
             doublereal *tol, doublereal *resid, integer *ncv, doublereal *v, 
             integer *ldv, integer *iparam, integer *ipntr, 
             doublereal *workd, doublereal *workl, integer *lworkl, 
             integer *info, ftnlen bmat_len, ftnlen which_len );

int dseupd_ ( logical *rvec, char *howmny, logical *select, 
                     doublereal *d__, doublereal *z__, integer *ldz, 
                     doublereal *sigma, char *bmat, integer *n, char *which,
                     integer *nev, doublereal *tol, doublereal *resid, 
                     integer *ncv, doublereal *v, integer *ldv, integer *iparam,
                     integer *ipntr, doublereal *workd, doublereal *workl, 
                     integer *lworkl, integer*info, ftnlen howmny_len, 
                     ftnlen bmat_len, ftnlen which_len );

/* Common Block Declarations */

struct {
  integer logfil, ndigit, mgetv0, msaupd, msaup2, msaitr, mseigt, msapps,
    msgets, mseupd, mnaupd, mnaup2, mnaitr, mneigh, mnapps, mngets,
    mneupd, mcaupd, mcaup2, mcaitr, mceigh, mcapps, mcgets, mceupd;
} debug_;

#define debug_1 debug_

}

#ifndef _dbg
#define _dbg(x)   std::cout << "*" << x.c_str() << "*\n"
#endif


driver::driver (int argc, char** argv) 
{
  mat1 = mat2 = mat3 = 0;
  smat1 = smat2 = smat3 = 0;
  input_line = 0;
  curpos = 0;
  singp = false;
  ew = 0;
  ev = 0;
  vec  = 0;
  r = 0;
  base = 0;
  sew = 0; sev = 0; svec = 0; sr =0;
  bin = false;
  version = "0.7";
  prompt = "sparse> ";

  cmdtable = new std::string[NCOMMANDS];
  cmdtable[LOAD]     = std::string("load");
  cmdtable[UNLOAD]   = std::string("unload");
  cmdtable[SAVE]     = std::string("save");
  cmdtable[PRINT]    = std::string("print");
  cmdtable[LOADVEC]  = std::string("loadvec");
  cmdtable[SET]      = std::string("set");
  cmdtable[HELP]     = std::string("help");
  cmdtable[EIGS]     = std::string("eigs");
  cmdtable[ADD]      = std::string("add");
  cmdtable[SUB]      = std::string("sub");
  cmdtable[MUL]      = std::string("mul");
  cmdtable[DOAX]     = std::string("doax");
  cmdtable[DOATX]    = std::string("doatx");
  cmdtable[SHELLCMD] = std::string("sh");
  cmdtable[TRAN]     = std::string("tran");
  cmdtable[SIZE]     = std::string("size");
  cmdtable[SETP]    = std::string("setp");
  cmdtable[SETBIN]  = std::string("setb");
  cmdtable[DOATA]   = std::string("doata");
  cmdtable[DOAAT]   = std::string("doaat");
  cmdtable[TESTC]    = std::string("test");
  cmdtable[RANDMAT] = std::string("randmat");
  cmdtable[RANDVEC] = std::string("randvec");
  cmdtable[SETBASE] = std::string("setbase");
  init_functions();
}

void driver::init_functions()
{
  function_table.insert(afunc(cmdtable[LOAD], &driver::load));
  function_table.insert(afunc(cmdtable[UNLOAD], &driver::unload));
  function_table.insert(afunc(cmdtable[SAVE], &driver::save));
  function_table.insert(afunc(cmdtable[PRINT], &driver::print));
  function_table.insert(afunc(cmdtable[LOADVEC], &driver::loadvec));
  function_table.insert(afunc(cmdtable[SET], &driver::set));
  function_table.insert(afunc(cmdtable[HELP], &driver::help));
  function_table.insert(afunc(cmdtable[EIGS], &driver::eigs));
  function_table.insert(afunc(cmdtable[ADD], &driver::add));
  function_table.insert(afunc(cmdtable[SUB], &driver::sub));
  function_table.insert(afunc(cmdtable[MUL], &driver::mul));
  function_table.insert(afunc(cmdtable[DOAX], &driver::doax));
  function_table.insert(afunc(cmdtable[DOATX], &driver::doatx));
  function_table.insert(afunc(cmdtable[SHELLCMD], &driver::sh));
  function_table.insert(afunc(cmdtable[TRAN], &driver::tran));
  function_table.insert(afunc(cmdtable[SIZE], &driver::size));
  function_table.insert(afunc(cmdtable[SETP], &driver::setp));
  function_table.insert(afunc(cmdtable[SETBIN], &driver::setb));
  function_table.insert(afunc(cmdtable[DOATA], &driver::doata));
  function_table.insert(afunc(cmdtable[DOATA], &driver::doaat));
  function_table.insert(afunc(cmdtable[TESTC], &driver::test));
  function_table.insert(afunc(cmdtable[RANDMAT], &driver::randmat));
  function_table.insert(afunc(cmdtable[RANDVEC], &driver::randvec));
  function_table.insert(afunc(cmdtable[SETBASE], &driver::setbase));
}


int driver::setbase()
{
  std::string m = next_word();
  if (m == "") return 0;
  base = atoi(m.c_str());
  return 0;
}

int driver::setb()
{
  std::string m = next_word();
  if (m == "") return 0;
  if (m == "true") bin = true;
  if (m == "false") bin = false;
  return 0;
}

int driver::size()
{
  matrix* mat = 0;
  matrix_float* smat = 0;

  if (singp) {
    smat = get_matrix_float();
  } else
    mat = get_matrix();

  if (!mat && !smat) {
    ERROR("No such matrix");
    return -1;
  }
  
  if (singp)
    std::cout << smat->get_name() << ": (" << smat->nrows() 
              << "," << smat->ncols() << ")\n";
  else 
    std::cout << mat->get_name() << ": (" << mat->nrows() 
              << "," << mat->ncols() << ")\n";
  return 0;
}

std::string driver::next_word(bool tf)
{
  //std::cerr << "CURPOS: ##" << curpos << "##\n";
  if (*curpos == '\0' || *curpos == '\n') {
    if (tf)
      ERROR("\t !!Command requires argument!!");
    return "";
  }
  
  char* s = curpos;
  while (*s && (*s != '\n') && !isspace(*s)) {
    s++;
  }

  char* rpos = curpos;

  //std::cerr << "RPOS: ##"  << rpos << "##\n";
  curpos = s;
  // Eat whitespace
  while (*curpos && (*curpos != '\n') && isspace(*curpos)) ++curpos;

  char* ns = (char*) malloc( (s-rpos+1)*sizeof(char));
  for (int i = 0; i < s-rpos; i++) {
    ns[i] = rpos[i];
  }
  ns[s-rpos] = '\0';
  std::string str = std::string(ns);
  free(ns);
  
  //std::cerr << "Returning: ##" << str << "##\n";
  return str;
}

int driver::setp()
{
  std::string m = next_word();
  if (m == "float") {
    set_single_precision(true);
    return 0;
  } else if (m == "double") {
    set_single_precision(false);
    return 0;
  } else
    return -1;
  
}

matrix* driver::get_matrix()
{
  matrix* mat;
  std::string m = next_word();
  if (m == "") return 0;
  if (m == "m1" && mat1 != 0) { 
    mat = mat1; mat->set_name(std::string("m1"));
  } else if (m == "m2" && mat2 != 0) {
     mat = mat2; mat->set_name(std::string("m2"));
  } else if (m == "m3" && mat3 != 0) {
     mat = mat3; mat->set_name(std::string("m3"));
  } else 
    mat = 0;
  return mat;
}

matrix_float* driver::get_matrix_float()
{
  matrix_float* smat;
  std::string m = next_word();
  if (m == "") return 0;
  if (m == "m1" && smat1 != 0) { 
    smat = smat1; smat->set_name(std::string("m1"));
  } else if (m == "m2" && smat2 != 0) {
     smat = smat2; smat->set_name(std::string("m2"));
  } else if (m == "m3" && smat3 != 0) {
     smat = smat3; smat->set_name(std::string("m3"));
  } else 
    smat = 0;
  return smat;
}

int driver::unload()
{
  if (singp)
    return unload_float();

  std::string m = next_word(true);
  if (m == "")
    return -1;

  if (m == "m1") {
    if (mat1) {
      mat1->free_data();
      delete mat1;
      mat1 = 0;
    }
  } else if (m == "m2") {
    if (mat2) {
      mat2->free_data();
      delete mat2;
      mat2 = 0;
    }
  } else if (m == "m3") {
    if (mat3) {
      mat3->free_data();
      delete mat3;
      mat3 = 0;
    }
  } else if (m == "r") {
    if (r) {
      gsl_vector_free(r);
      r = 0;
    }
  } else if (m == "vec") {
    if (vec) {
      gsl_vector_free(vec);
      vec = 0;
    }
  } else {
    ERROR("Invalid matrix unload requested");
    return -2;
  }

  return 0;
}


int driver::unload_float()
{
  std::string m = next_word(true);
  if (m == "")
    return -1;

  if (m == "m1") {
    if (smat1) {
      smat1->free_data();
      delete smat1;
      smat1 = 0;
    }
  } else if (m == "m2") {
    if (smat2) {
      smat2->free_data();
      delete smat2;
      smat2 = 0;
    }
  } else if (m == "m3") {
    if (smat3) {
      smat3->free_data();
      delete smat3;
      smat3 = 0;
    }
  } else if (m == "r") {
    if (sr) {
      gsl_vector_float_free(sr);
      sr = 0;
    }
  } else if (m == "vec") {
    if (svec) {
      gsl_vector_float_free(svec);
      svec = 0;
    }
  } else {
    ERROR("Invalid matrix unload requested");
    return -2;
  }

  return 0;
}

int driver::test()
{
  matrix* mat = get_matrix();
  if (mat == 0) {
    return -1;
  }
  std::cout << "DOT = " << mat->row_col_dot(0, 0) << "\n";
  return 0;
}

/**
 * @todo
 */
int driver::add()
{
  // Here one faces the multiple dispatch problem
  // Assume mat1 is ccs and mat2 is dense, how to make sure things go
  // smooth...easy, as of now, do not permit adding diff types of matrices.
  
  return -1;
}

/**
 * @todo
 */
int driver::sub()
{
  return -1;
}

/**
 * @todo
 */
int driver::mul()
{
  return -1;
}

/**
 * @todo Needs to be cleaned out.
 */
int driver::doax()
{
  if (singp)
    return doax_float();

  matrix* mat = get_matrix();

  if (mat && vec) {
    if (r) { gsl_vector_free(r); r = 0;}
    if (!r) r = gsl_vector_calloc(mat->nrows());
    mat->dot(false, vec, r);
  }
  return 0;
}


int driver::doata()
{
  if (singp)
    return doata_float();

  matrix* mat = get_matrix();
  if (!mat) {
    ERROR("Matrix not specified!, Say doata m1 or doata m2 etc..");
    return -1;
  }

  std::string fname = next_word();
  if (fname == "") {
    ERROR ("You need to specify output file to hold A'A");
    return -2;
  }

  return do_ata_helper(mat, fname.c_str());
}


int driver::do_ata_helper(matrix* mat, const char* f)
{
  FILE* fp = fopen(f, "w");
  if (!fp) {
    ERROR("File opening error in doata()");
    return -3;
  }

  vector* dp = gsl_vector_calloc(mat->ncols());
  vector* c = gsl_vector_calloc(mat->nrows());
  for (uint i = 0; i < mat->ncols(); i++) {
    mat->get_col(i, c);
    mat->dot(true, c, dp);
    for (uint j = 0; j < mat->ncols(); j++) 
      fprintf(fp, "%f ", dp->data[j]);
    fprintf(fp, "\n");
  }
  gsl_vector_free(c);
  gsl_vector_free(dp);
  fclose(fp);
  return 0;
}

int driver::doata_float()
{
  return -1;
}

int driver::doaat()
{
  if (singp)
    return doaat_float();

  matrix* mat = get_matrix();
  if (!mat) {
    ERROR("Matrix not specified!, Say doaat m1 or doaat m2 etc..");
    return -1;
  }

  std::string fname = next_word();
  if (fname == "") {
    ERROR ("You need to specify output file to hold AA'");
    return -2;
  }

  return do_aat_helper(mat, fname.c_str());
}

int driver::do_aat_helper(matrix* mat, const char* f)
{
  FILE* fp = fopen(f, "w");
  if (!fp) {
    ERROR("File opening error in doata()");
    return -3;
  }

  vector* dp = gsl_vector_calloc(mat->ncols());
  vector* c = gsl_vector_calloc(mat->nrows());

  for (uint i = 0; i < mat->ncols(); i++) {
    mat->get_col(i, c);
    mat->dot(true, c, dp);
    for (uint j = 0; j < mat->ncols(); j++) 
      fprintf(fp, "%f ", dp->data[j]);
    fprintf(fp, "\n");
  }
  gsl_vector_free(c);
  gsl_vector_free(dp);
  fclose(fp);
  return 0;
}

int driver::doaat_float()
{
  return -1;
}

/**
 * @todo Needs to be cleaned out.
 */
int driver::doax_float()
{
  matrix_float* smat = get_matrix_float();

  if (smat && svec) {
    if (sr) { gsl_vector_float_free(sr); sr = 0;}
    if (!sr) sr = gsl_vector_float_calloc(smat->nrows());
    smat->dot(false, svec, sr);
  }
  return 0;
}

int driver::doatx()
{
  if (singp)
    return doatx_float();

  matrix* mat = get_matrix();

  if (mat && vec) {
    if (r) { gsl_vector_free(r); r = 0;}
    if (!r) r = gsl_vector_calloc(mat->ncols());
    mat->dot(true, vec, r);
  }
  return 0;
}

int driver::doatx_float()
{
  matrix_float* smat = get_matrix_float();

  if (smat && svec) {
    if (sr) { gsl_vector_float_free(sr); sr = 0;}
    if (!sr) sr  = gsl_vector_float_calloc(smat->ncols());
    smat->dot(true, svec, sr);
  }
  return 0;
}

int driver::sh()
{
  std::string w = std::string(curpos);
  if (w == "") return -1;
  return system(w.c_str());
}

int driver::loadvec()
{
  if (singp)
    return loadvec_float();

  std::string w = next_word();
  if (w == "") return -1;
  vector* oldvec = vec;
  vec = SSUtil::read_gsl_vector(w.data());
  if (!vec) {
    ERROR("Could not read vector from file: " << w.c_str());
    return -1;
  }
  if (oldvec)
    gsl_vector_free(oldvec);
  return 0;
}


int driver::loadvec_float()
{
  std::string w = next_word();
  if (w == "") return -1;
  vector_float* oldvec = svec;
  svec = SSUtil::read_gsl_vector_float(w.data());
  if (!svec) {
    ERROR("Could not read vector from file: " << w.c_str());
    return -1;
  }
  if (oldvec)
    gsl_vector_float_free(oldvec);
  return 0;
}

int driver::set ()
{
  return 0;
}

int driver::help()
{
  std::string s = next_word();
  if (s == "") {
    show_help();
    return 0;
  } else {
    for (int i = 0; i < NCOMMANDS; i++) {
      if (s == cmdtable[i]) {
        show_help(i);
        return 0;
      }
    }
  }
  return -1;
}

/**
 * Uses ARPACK to compute the eigenvecs and values
 */
int driver::eigs()
{
  if (singp)
    eigs_float();

  matrix* mat = get_matrix();
  if (mat == 0) {
    ERROR ("Invalid matrix requested");
    return -1;
  }

  if (mat->nrows() != mat->ncols()) {
    ERROR ("Matrix must be square, and also symmetric");
  }

  uint k;
  // Now find out how many eigs the user wants....
  std::string m = next_word();
  if (m == "")
    k = min((size_t)2, mat->nrows());
  else
    k = min((size_t)atoi(m.c_str()), mat->nrows());

  std::cout << "Computing " << k << " eigenpairs\n";
  if (k == 0) 
    return 0;

  /// Here we start the ARPACK stuff. This function puts stuff in ew and ev
  /// as it should.
  return compute_eigenpairs(mat, k);
}

int driver::eigs_float()
{
  ERROR("NOT YET IMPLEMENTED!");
  return -1;
}

int driver::compute_eigenpairs(matrix* mat, uint k)
{
  integer nev = k;                  // Number of eigenvalues
  integer ncv = min(mat->nrows(), k + 2);             // Length of Arnoldi factorization
  integer maxncv = min(mat->nrows(), ncv+2);         // max
  char bmat = 'I';              // Standard eigenvalue problem
  char* which = "LM";           // K eigs of largest magnitude

  /// Do some more error checking here....
  integer n = mat->nrows();
  integer maxn = n;

  doublereal* workl = new doublereal[3*maxncv*maxncv + 6*maxncv];
  doublereal* d     = new doublereal[3*maxncv];
  doublereal* resid = new doublereal[maxn];
  doublereal* v     = new doublereal[maxn*maxncv];
  doublereal* workd  = new doublereal[3*maxn];

  doublereal tol = 0.0;
  integer ido = 0;
  integer info = 0;
  integer ierr;
  integer ishfts = 1;
  integer maxitr = 300;
  integer model = 1;
  integer iparam[11];
  integer ipntr[14];
  logical select[maxncv];
  integer ldv = maxn;
  integer lworkl = ncv * (ncv + 8);
  integer nx = n;
  doublereal sigma;

  iparam[0] = ishfts;
  iparam[2] = maxitr;
  iparam[6] = model;
  vector v1, v2;
  /*     %-------------------------------------------% */
  /*     | M A I N   L O O P (Reverse communication) | */
  /*     %-------------------------------------------% */

  debug_1.ndigit = -3;
  debug_1.logfil = 6;
  debug_1.msgets = 0;
  debug_1.msaitr = 0;
  debug_1.msapps = 0;
  debug_1.msaupd = 0;
  debug_1.msaup2 = 0;
  debug_1.mseigt = 0;
  debug_1.mseupd = 0;
  
 L10:

  /*        %---------------------------------------------% */
  /*        | Repeatedly call the routine DNAUPD and take | */
  /*        | actions indicated by parameter IDO until    | */
  /*        | either convergence is indicated or maxitr   | */
  /*        | has been exceeded.                          | */
  /*        %---------------------------------------------% */

  dsaupd_(&ido, &bmat, &n, which, &nev, &tol, resid, &ncv, v, &ldv,
          iparam, ipntr, workd, workl, &lworkl, &info, (ftnlen)1, (ftnlen)2);

  if (ido == -1 || ido == 1) {

    /*           %-------------------------------------------% */
    /*           | Perform matrix vector multiplication      | */
    /*           |                y <--- Op*x                | */
    /*           | The user should supply his/her own        | */
    /*           | matrix vector multiplication routine here | */
    /*           | that takes workd(ipntr(1)) as the input   | */
    /*           | vector, and return the matrix vector      | */
    /*           | product to workd(ipntr(2)).               | */
    /*           %-------------------------------------------% */
    
    v1.data = &workd[ipntr[0] - 1];
    v2.data = &workd[ipntr[1] - 1];
    v1.size = nx;
    v2.size = nx;

    mat->dot(true, &v1, &v2);
    
    /*           %-----------------------------------------% */
    /*           | L O O P   B A C K to call DNAUPD again. | */
    /*           %-----------------------------------------% */
    
    goto L10;
    
  }

  /*     %----------------------------------------% */
  /*     | Either we have convergence or there is | */
  /*     | an error.                              | */
  /*     %----------------------------------------% */
  
    if (info < 0) {

      /*        %--------------------------% */
      /*        | Error message, check the | */
      /*        | documentation in DNAUPD. | */
      /*        %--------------------------% */
      
      ERROR("Error with eigenvalue ARPACK routine: INFO == " << info);
      /// TODO: Clean up the allocated memory
      delete[] workl;
      delete[] workd;
      delete[] resid;
      delete[] v;
      delete[] d;
      return -1;
    } else {
      logical rvec = TRUE_;
      dseupd_(&rvec, "All", select, d, v, &ldv, &sigma, 
              &bmat, &n, which, &nev, &tol, resid, &ncv, v, &ldv,
              iparam, ipntr, workd, workl, &lworkl, &ierr, 
              (ftnlen)1, (ftnlen)1, (ftnlen)2);
    }
    
    /*         %----------------------------------------------% */
    /*         | Eigenvalues are returned in the first column | */
    /*         | of the two dimensional array D and the       | */
    /*         | corresponding eigenvectors are returned in   | */
    /*         | the first NCONV (=IPARAM(5)) columns of the  | */
    /*         | two dimensional array V if requested.        | */
    /*         | Otherwise, an orthogonal basis for the       | */
    /*         | invariant subspace corresponding to the      | */
    /*         | eigenvalues in D is returned in V.           | */
    /*         %----------------------------------------------% */
    
    
    if (ierr != 0) {
      
      /*            %------------------------------------% */
      /*            | Error condition:                   | */
      /*            | Check the documentation of DSEUPD. | */
      /*            %------------------------------------% */

      ERROR("Got back IERR = " << ierr);
      /// Cleanup
      return -1;
    } else {
      if (ew) {
        gsl_vector_free(ew); ew = 0;
      }
      ew = gsl_vector_alloc(nev);
      
      for (int i = 0; i < nev; i++)
        ew->data[i] = d[i];

      delete[] d;
      delete[] workl;
      delete[] workd;
      delete[] resid;
      
      view = gsl_matrix_view_array(v, iparam[4], n);
      
      if (ev) {
        delete[] (ev->get_matrix())->data;
        delete ev;
      } 
      
      std::cout << "(" << view.matrix.size1 << " , " << view.matrix.size2 << ")\n";
      ev = new dense_matrix(&view.matrix, view.matrix.size1, view.matrix.size2);
    }

    return 0;
}

// sparse> load filename [as ccs|crs|...]
int driver::load ()
{
  if (singp)
    return load_float();
  
  // Get filename
  std::string fname = next_word(true);
  if (fname == "") {
    return -1;
  }

  MatType loadtype;

  // See if the 'as something' string is also given in
  std::string loadas = next_word();
  if (loadas == "") {
    loadtype = M_CCS;
  } else if (loadas == "as") {
    // See what word is following the 'as' part
    loadas = next_word(true);
    if (loadas == "") {
      return -2;
    }
    if (loadas == "ccs") 
      loadtype = M_CCS;
    else if (loadas == "crs")
      loadtype = M_CRS;
    else if (loadas == "mat")
      loadtype = M_MATLAB;
    else if (loadas == "hb")
      loadtype = M_HB;
    else if (loadas == "dense")
      loadtype = M_DENSE;
    else if (loadas == "coordccs")
      loadtype = M_COORD_CCS;
    else if (loadas == "coordcrs")
      loadtype = M_COORD_CRS;
    else if (loadas == "colcoord")
      loadtype = M_COLCOORD;
    else if (loadas == "rowcoord")
      loadtype = M_ROWCOORD;
    else {
      ERROR(loadas << ": Invalid/Unsupported matrix type");
      return -4;
    }
  } else {
    ERROR("load: malformed command");
    return -2;
  }

  std::cout << "Trying to load `" 
            << fname.c_str() << "' as `" << loadas.c_str() << "'...";
  
  matrix* loadmat;
  switch (loadtype) {
  case M_CCS:
    loadmat = new ccs();
    break;
  case M_CRS:
    loadmat = new crs();
    break;
  case M_MATLAB:
    loadmat = new matlab();
    break;
  case M_HB:
    loadmat = new harwell_boeing();
    break;
  case M_DENSE:
    loadmat = new dense_matrix();
    break;
  case M_COORD_CCS:
    loadmat = new coord_ccs();
    break;
  case M_COORD_CRS:
    loadmat = new coord_crs();
    break;
  case M_COLCOORD:
    loadmat = new col_coord();
    break;
  case M_ROWCOORD:
    loadmat = new row_coord();
    break;
  default:
    loadmat = 0;
    break;
  }

  loadmat->set_base(base);
  if (mat1 != 0 && mat2 != 0 && mat3 != 0) {
    ERROR("All 3 matrices already loaded or in use. Please unload one of them");
    return -5;
  }

  
  if (loadmat->load(fname.c_str(), bin) != 0) {
    std::cout << "Failed\n";
    delete loadmat;
    return 1;
  }
  
  std::cout << "Done\n";
  if (mat1 == 0)
    mat1 = loadmat;
  else if (mat2 == 0)
    mat2 = loadmat;
  else if (mat3 == 0)
    mat3 = loadmat;
  else {
    ;
  }
 
  return 0;
}


int driver::load_float ()
{
  // Get filename
  std::string fname = next_word(true);
  if (fname == "") {
    return -1;
  }
  
  std::cerr << "Attempting to LOAD: " << fname << "\n";
  MatType loadtype;

  // See if the 'as something' string is also given in
  std::string loadas = next_word();
  if (loadas == "") {
    loadtype = M_CCS_FLOAT;
  } else if (loadas == "as") {
    // See what word is following the 'as' part
    loadas = next_word(true);
    if (loadas == "") {
      return -2;
    }
    if (loadas == "ccs") 
      loadtype = M_CCS_FLOAT;
    else if (loadas == "crs")
      loadtype = M_CRS_FLOAT;
    else if (loadas == "mat")
      loadtype = M_MATLAB_FLOAT;
    else if (loadas == "hb")
      loadtype = M_HB_FLOAT;
    else if (loadas == "dense")
      loadtype = M_DENSE_FLOAT;
    else if (loadas == "coordccs")
      loadtype = M_COORD_CCS_FLOAT;
    else if (loadas == "coordcrs")
      loadtype = M_COORD_CRS_FLOAT;
    else if (loadas == "colcoord")
      loadtype = M_COLCOORD_FLOAT;
    else if (loadas == "rowcoord")
      loadtype = M_ROWCOORD_FLOAT;
    else {
      ERROR(loadas << ": Invalid/Unsupported matrix type");
      return -4;
    }
  } else {
    ERROR("load: malformed command");
    return -2;
  }

  std::cout << "Trying to load `" 
            << fname.c_str() << "' as `" << loadas.c_str() << "'...";
  
  matrix_float* loadmat;
  switch (loadtype) {
  case M_CCS_FLOAT:
    loadmat = new ccs_float();
    break;
  case M_CRS_FLOAT:
    loadmat = new crs_float();
    break;
  case M_HB_FLOAT:
    loadmat = new harwell_boeing_float();
    break;
  case M_DENSE_FLOAT:
    loadmat = new dense_matrix_float();
    break;
  case M_COORD_CCS_FLOAT:
    loadmat = new coord_float();
    break;
  case M_COORD_CRS_FLOAT:
    loadmat = new coord_crs_float();
    break;
  case M_COLCOORD_FLOAT:
    loadmat = new col_coord_float();
    break;
  case M_ROWCOORD_FLOAT:
    loadmat = new row_coord_float();
    break;
  default:
    loadmat = 0;
    break;
  }

  loadmat->set_base(base);
  if (smat1 != 0 && smat2 != 0 && smat3 != 0) {
    ERROR("All 3 matrices already loaded or in use. Please unload one of them");
    return -5;
  }
  
  if (loadmat->load(fname.c_str(), bin) != 0) {
    std::cout << "Failed\n";
    delete loadmat;
    return 1;
  }
  
  std::cout << "Done\n";
  if (smat1 == 0)
    smat1 = loadmat;
  else if (smat2 == 0)
    smat2 = loadmat;
  else if (smat3 == 0)
    smat3 = loadmat;
  else {
    ;
  }
  return 0;
}

int driver::save ()
{
  if (singp)
    return save_float();
  
  matrix* mat = 0;

  std::string  w = next_word(true);
  if (w == "") {
    ERROR("Invalid command, missing argument or matrix not loaded!");
    return -1;
  }
  
  if (w == "m1") {
    mat = mat1;
  }  else if (w == "m2")
    mat = mat2;
  else if (w == "m3")
    mat = mat3;
  else if (w == "ew")
    mat = ev;
  else if (w == "r") {
    w = next_word(true);
    if (w == "") return -2;
    return SSUtil::write_gsl_vector(r, w.c_str());
  } else {
    ERROR("Invalid command or matrix requested");
    return -1;
  }

  w = next_word(true);
  if (w == "") {
    ERROR("Filename not given");
    return -3;
  } else    {
    ERROR("trying to save to ... " + w);
  }
    

  MatType savetype;

  std::string saveas = next_word();
  if (saveas == "") { 
    savetype = M_SELF;
  } else if (saveas == "as") { 
    // find the type and save as that type
    saveas = next_word(true);
    if (saveas == "")
      return -3;
    if (saveas == "ccs") 
      savetype = M_CCS;
    else if (saveas == "crs")
      savetype = M_CRS;
    else if (saveas == "mat")
      savetype = M_MATLAB;
    else if (saveas == "hb")
      savetype = M_HB;
    else if (saveas == "dense")
      savetype = M_DENSE;
    else if (saveas == "coordcrs")
      savetype = M_COORD_CRS;
    else if (saveas == "coordccs")
      savetype = M_COORD_CCS;
    else if (saveas == "colcoord")
      savetype = M_COLCOORD;
    else {
      ERROR(saveas << ": Invalid/Unsupported matrix type");
      return -4;
    }
  } else {
    ERROR("save: invalid or malformed command");
    return -2;
  }

  return mat->save(w.c_str(),bin, savetype);
}

int driver::save_float ()
{
  matrix_float* smat;
  std::string  w = next_word(true);
  if (w == "" || smat1 == 0) {
    ERROR("Invalid command, missing argument or matrix does not exist");
    return -1;
  }

  if (w == "m1") {
    smat = smat1;
  }  else if (w == "m2")
    smat = smat2;
  else if (w == "m3")
    smat = smat3;
  else if (w == "ew")
    smat = sev;
  else if (w == "r") {
    w = next_word(true);
    if (w == "") return -2;
    return SSUtil::write_gsl_vector_float(sr, w.c_str());
  } else {
    ERROR("Invalid command or matrix requested");
    return -1;
  }

  w = next_word(true);
  if (w == "") {
    ERROR("Filename not given");
    return -3;
  } else    {
    ERROR("trying to save to ... " + w);
  }
   

  MatType savetype;

  std::string saveas = next_word();
  if (saveas == "") { 
    savetype = M_SELF;
  } else if (saveas == "as") { 
    // find the type and save as that type
    saveas = next_word(true);
    if (saveas == "")
      return -3;
    if (saveas == "ccs") 
      savetype = M_CCS_FLOAT;
    else if (saveas == "crs")
      savetype = M_CRS_FLOAT;
    else if (saveas == "mat")
      savetype = M_MATLAB_FLOAT;
    else if (saveas == "hb")
      savetype = M_HB_FLOAT;
    else if (saveas == "dense")
      savetype = M_DENSE_FLOAT;
    else if (saveas == "coordccs")
      savetype = M_COORD_CCS_FLOAT;
    else if (saveas == "coordcrs")
      savetype = M_COORD_CRS_FLOAT;
    else if (saveas == "colcoord")
      savetype = M_COLCOORD_FLOAT;
    else if (saveas == "rowcoord")
      savetype = M_ROWCOORD_FLOAT;
    else {
      ERROR(saveas << ": Invalid/Unsupported matrix type");
      return -4;
    }
  } else {
    ERROR("save_float: invalid or malformed command: " + saveas);
    return -2;
  }

  
  return smat->save(w.c_str(),bin, savetype);
}

int driver::print ()
{
  if (singp)
    return print_float();

  std::string m = next_word();
  if (m == "" && mat1 != 0)
    mat1->print();
  else {
    if (m == "m1" && mat1 != 0)
      mat1->print();
    else if (m == "m2" && mat2 != 0)
      mat2->print();
    else if (m == "m3" && mat3 != 0)
      mat3->print();
    else if (m == "r" && r != 0)
      SSUtil::print_vector(r);
    else if (m == "vec" && vec != 0)
      SSUtil::print_vector(vec);
    else if (m == "ew" && ew != 0)
      SSUtil::print_vector(ew);
    else if (m == "ev" && ev != 0)
      ev->print();
    else {
      ERROR("Invalid matrix requested");
      return -1;
    }
  }
  return 0;
}

int driver::print_float ()
{
  std::string m = next_word();
  if (m == "" && smat1 != 0)
    smat1->print();
  else {
    if (m == "m1" && smat1 != 0)
      smat1->print();
    else if (m == "m2" && smat2 != 0)
      smat2->print();
    else if (m == "m3" && smat3 != 0)
      smat3->print();
    else if (m == "r" && sr != 0)
      SSUtil::print_vector_float(sr);
    else if (m == "vec" && svec != 0)
      SSUtil::print_vector_float(svec);
    else if (m == "ew" && sew != 0)
      SSUtil::print_vector_float(sew);
    else if (m == "ev" && sev != 0)
      sev->print();
    else {
      ERROR("Invalid matrix requested");
      return -1;
    }
  }
  return 0;
}

int driver::tran()
{
  if (singp) {
    ERROR("TRAN not yet implemented yet for single precision!");
    return -1;
  }
    
  std::string m = next_word();
  if (m != "") {
    ERROR ("No arguments entertained by the TRAN command");
    return -1;
  }
  if (mat2 != 0 && mat3 != 0) {
    ERROR("Both m2 and m3 in use. Can't transpose. Unload one of them first");
    return -2;
  }
  matrix* mat;
  if (mat1 != 0)
    mat = mat1->transpose_copy();
  else {
    ERROR("I can only transpose m1, which currently is empty");
    return -4;
  }
  if (mat2 == 0)
    mat2 = mat;
  else
    mat3 = mat;
  return 0;
}

int driver::randmat() 
{
  std::string m = next_word();
  if (m == "") {
    ERROR ("Missing number of rows");
    return -1;
  }

  std::string n = next_word();
  if (n == "") {
    ERROR ("Missing number of columns");
    return -1;
  }
  
  uint ro = atoi(m.c_str());
  uint co = atoi(n.c_str());

  gsl_matrix* M = SSUtil::random_matrix(ro, co);
  matrix* mat = new dense_matrix(M, ro, co);

  if (mat1 == 0)
    mat1 = mat;
  else if (mat2 == 0)
    mat2 = mat;
  else if (mat3 == 0)
    mat3 = mat;
  else {
    ERROR ("Unload m1, m2 or m3 first");
    delete mat;
    return -1;
  }
  return 0;
}

int driver::randvec()
{
  std::string m = next_word();
  if (m == "") {
    ERROR("Missing size of vector");
    return -1;
  }

  vector *v = SSUtil::random_vector(atoi(m.c_str()));

  if (r == 0)
    r = v;
  else {
    vector_free(v);
    ERROR ("First unload vector 'r'");
    return -1;
  }
  
  return 0;
}

char* driver::readline_getline()
{
  /* If the buffer has already been allocated,
     return the memory to the free pool. */
  if (line) {
    free (line);
    line = 0;
  }

  /* Get a line from the user. */
  line = readline (prompt);

  /* If the line has any text in it,   save it on the history. */
  if (line && *line)
    add_history (line);

  return line;
}

int driver::start() 
{
  line = 0;
  using_history();

  do {
    line =      readline_getline();
    if (input_line) {
      free(input_line);
      input_line = 0;
    }
    input_line = strdup(line);
    curpos = input_line;
    
    std::string s = next_word();
    if (s == "exit" || s == "bye" || s == "quit")
      break;
    if (s == "") continue;
    std::map<std::string, _intfn>::iterator p = function_table.find(s);

    if (p == function_table.end()) {
      std::cerr << "Command " << s << " not understood. Type 'help' for list of commmands\n";
      continue;
    } else {
      if ((this->*p->second)() < 0) {
        std::cerr << "Error invoking function for processing `" << s << "'\n";
        continue;
      }
    }

  } while (line != 0);
  return 0;
}

void driver::show_help()
{
  std::cerr << "Commands supported\n"
	    << "\tload  -- Loads a sparse matrix files\n"
	    << "\tunload -- Unloads specified matrix\n"
	    << "\tsave  -- Saves a sparse matrix\n" 
	    << "\tprint  -- Prints out current matrix\n"
	    << "\thelp  -- Prints this help message\n"
	    << "\tloadvec -- Loads a vector\n"
	    << "\tset  -- Sets various options\n"
	    << "\teigs -- Do a simple power iteration\n"
	    << "\tadd  -- Adds two loaded matrices\n"
	    << "\tsub  -- Subtracts two loaded matrices\n"
	    << "\tmul  -- Multiplies two sparse matrices to yield dense\n"
	    << "\tdoax -- Does A*x using specified matrix\n"
	    << "\tdoatx -- Does A'x\n"
        << "\ttran -- Computes A', stores in mat2 or mat3\n"
	    << "\tsh    -- Run a shell command\n"
            << "\tsetp -- Set single or double precision\n"
            << "\tsetb -- Set load mode to binary or text\n"
	    << "\tquit, exit, bye -- Exit the program\n"
	    << "Type 'help' followed by one of the above commands for more detailed help\n"; 
  
}

void driver::show_help(int c)
{
  switch (c) {
  case LOAD:
    std::cout
      << "load <matrix> [as [mat|ccs|crs|hb|dense]]\n"
      << "Loads the file given by <matrix>. Defaults to using"
      << " <matrix> as a prefix for ccs files. Using the optional"
      << " argument 'as <matrixtype>' allows you to load matrices"
      << " other than CCS matrices. For MATLAB and Harwell-Boeing sparse"
      << " matrices, <matrix> should give the full filename. For all others"
      << " it gives the filename prefix. NOTE that if the setb true was"
      << "executed <matrix> is the full filename, and is loaded in binary"
      << "(non-text) format for all types of matrices. Once the matrix is "
      << "loaded it is available as m1 or m2 or m3. Three different matrices"
      << "can be loaded at the same time.\n";
    break;
  case UNLOAD:
    std::cout << "Unload <m1|m2|m3>\n"
	      << "Unloads (i.e., frees) the specified matrix so that a new one may be loaded instead\n";
    break;
  case SAVE:
    std::cout << "save <m1|m2|m3> <matrix> [as [mat|ccs|crs|hb|dense]]\n"
	      << "Saves the specified matrix to filename or filename prefix given by"
	      << " <matrix>. Defaults to using ccs files. Using the optional"
	      << " argument 'as <matrixtype>' allows you to load matrices"
	      << " other than CCS matrices. For MATLAB and Harwell-Boeing sparse"
	      << " matrices, <matrix> should give the full filename. For all others"
	      << " it gives the filename prefix. NOTE that if the option 'filetype'"
	      << " is set to binary (by using the command 'set filetype bin') then"
	      << " <matrix> is the full filename, and is saved in binary (non-text)"
	      << " format for all types of matrices\n";
    break;
  case PRINT:
    std::cout << "print <m1|m2|m3>\n"
	      << "Prints out the specified matrix to stdout\n";
    break;
  case LOADVEC:
    std::cout << "loadvec <filename>\n"
	      << "Loads a given vector from the specified file\n";
    break;
  case SET:
    std::cout << "set <option> <value>\n"
	      << "The various options and possible values are:\n"
	      << "filetype ascii|bin (default is ascii)\n"
	      << "maxiter num  Specifies the max no. of iterations for eigs\n";
    break;
  case HELP:
    std::cout << "help <cmdname>\n"
	      << "Typing 'help' by itself displays the list of avail commands\n";
    break;
  case EIGS:
    std::cout << "eigs <m1|m2|m3> <k>\n"
              << "Calls ARPACK routine to compute top <k> eigenvalues and vectors"
              << " of the specified matrix. The eigenvectors are left in the "
              << " variable <ev> and the values in <ew>.\n";
    break;
  case ADD:
    std::cout << "add\n"
	      << "Does m3 =  m1 + m2. Currently m1 and m2"
	      << " *must* be matrices of the same type and sparsity pattern\n";
    break;
  case SUB:
    std::cout << "sub\n"
	      << "Does m3 = m1 - m2. Currently m1 and m2"
	      << " *must* be matrices of the same type and sparsity pattern\n";
    break;
  case MUL:
    std::cout << "mul\n"
	      << "Does m3 = m1*m2. The resulting m3 is a dense matrix, so use"
	      << " this function with caution.\n";
    break;
  case DOAX:
    std::cout << "doax <m1|m2>\n"
	      << "Does m1*x or m2*x, where 'x' is a vector loaded using loadvec\n";
    break;
  case DOATX:
    std::cout << "doatx <m1|m2>\n"
	      << "Does m1'*x or m2'*x, where 'x' is a vector loaded using loadvec\n";
    break;
  case TRAN:
    std::cout << "tran\n"
              << "Computes Transpose(m1) by default and stores it in m2 or m3\n";
    break;
  case SHELLCMD:
    std::cout << "sh <command>\n"
	      << "Runs the <command> using the shell. E.g.\n"
	      << "sparse> sh ls  will get you the directory listing\n";
    break;
  case SETP:
    std::cout << "setp <float | double>\n";
      break;
  case SETBIN:
    std::cout << "setb <true | false>\n";
    break;
  default:
    std::cerr << "Invalid command requested\n";
    break;
  }
}

