/* File: dense_matrix_float.h        -*- c++ -*-         */
// Author: Suvrit Sra
// Time-stamp: <13 June 2006 09:56:35 PM CDT --  suvrit>
// Single precision dense matrix


#ifndef _MY_DENSEMATRIX_FLOAT_H
#define _MY_DENSEMATRIX_FLOAT_H

#include <gsl/gsl_matrix_float.h>
#include <gsl/gsl_blas.h>
#include "matrix_float.h"
#include "exceptions.h"

namespace SSLib {

  /**
   * @class dense_matrix_float to save space.
   * Just putting this in here to allow a more
   * complete 
   * solution, where all types of matrices are just subclasses of the
   * top-level matrix class. This particular class is just a wrapper around
   * a gsl_matrix class, to be able to inherit the complete power of the GSL.
   */
  class dense_matrix_float : public matrix_float {
    gsl_matrix_float* M;
    gsl_vector_float_view row;
    gsl_vector_float_view col;
  public:
    dense_matrix_float()  : matrix_float(0, 0)
    {
      M = 0;
    }
    dense_matrix_float(size_t rows, size_t cols, bool zeros = false)
      : matrix_float(rows, cols) {
      if (zeros)
        M = gsl_matrix_float_calloc(rows, cols);
      else 
        M = gsl_matrix_float_alloc(rows, cols);
      if (M == 0) {
        throw matrix_alloc("Memory allocation error in dense_matrix_float()"); 
      }
    }

    dense_matrix_float(gsl_matrix_float* m, size_t rows, size_t cols) : matrix_float(rows, cols) {
      M = m;
    }

    ~dense_matrix_float()
    {
      free_data();
    }
    virtual void free_data() { if (M) gsl_matrix_float_free(M); M = 0;}
    virtual int fload(const char*, size_t m, size_t n);
    virtual int load(const char*);
    virtual int load(const char*, bool);
    virtual int save(const char*, bool);
    /// Write out matrix in a specified way
    virtual int    save(const char*, bool asbin, int typ);
    /// Write out matrix to disk in CCS, CRS, etc. formats
    virtual int save_as_ccs(const char*);
    virtual int save_as_crs(const char*);
    virtual int save_as_coord(const char*);
    virtual int save_as_col_coord(const char*);
    virtual int save_as_matlab(const char*);
    virtual int save_as_hb(const char*);
    virtual int save_as_dense(const char*);
    

    float operator()(size_t i, size_t j) {
      return gsl_matrix_float_get(M, i, j);
    }

    float get(size_t i, size_t j) {
      return gsl_matrix_float_get(M, i, j);
    }

    int set(size_t i, size_t j, float v) {
      gsl_matrix_float_set(M, i, j, v); return 0;
    }
    
    // Am exposing the data to enable speed in some algorithms. One has
    // to sacrifice cleanliness of code for speed at times.
    gsl_matrix_float* get_matrix() {
      return M;
    }
    
    gsl_vector_float* get_row (size_t r) {
      row = gsl_matrix_float_row(M, r);
      return &row.vector;
    }

    gsl_vector_float* get_col (size_t c) {
      col = gsl_matrix_float_column(M, c);
      return &col.vector;
    }
    
    float* get_data() {
      return M->data;
    }
    
    // TODO:
    gsl_vector_float* get_diag(bool) { return 0;}
    // TODO:
    int set_row(size_t, gsl_vector_float* v) { return -1;}
    int set_col(size_t, gsl_vector_float* v) { return -1;}
    int set_diag(bool, gsl_vector_float* v) { return -1;}
    matrix_float* submatrix (size_t i1, size_t j1, 
                             size_t i2, size_t j2) { return 0;}
        /** ****************************************************************
     *  Matrix Operations begin here. They are group separately into
     *  'self (essentially elementwise operations on itself)'
     *  'matrix-vector operations'
     *  'matrix-matrix operations'
     *  ****************************************************************
     */

    /** @name Elementwise
     * Operations on elements of this matrix
     */

    /*@{*/

    /// Vector l_p norms for this matrix
    virtual double norm (double p);
    virtual double norm (char*  p);

    /// Apply an arbitrary function elementwise to this matrix
    virtual float apply (float (* fn)(float));

    /// x_ij <- s * x_ij
    virtual int scale (float s);

    /// x_ij <- s + x_ij
    virtual int add_const(float s);

    /*@}*/

    
    /** @name RowColOps
     *  ****************************************************
     * Operations on columns of this matrix
     *  ****************************************************
     */

    /*@{*/

    virtual double col_norm (size_t c, double p);
    virtual double col_dot  (size_t c1, size_t c2);
    virtual double col_dot  (size_t c, vector_float* v);
    virtual int    col_add  (size_t c1, size_t c2);
    virtual int    col_sub  (size_t c1, size_t c2);
    virtual int    col_scale(size_t c1, float s);
    virtual int    col_sub  (size_t c1, vector_float* v);
    virtual int    col_add  (size_t c1, vector_float* v);
    virtual int    col_scale(size_t c1, float s, vector_float* r);

    virtual double row_norm (size_t r, double p);
    virtual double row_dot  (size_t r1, size_t r2);
    virtual double row_dot  (size_t r, vector_float* v);
    virtual int    row_add  (size_t r1, size_t r2);
    virtual int    row_sub  (size_t r1, size_t r2);
    virtual int    row_scale(size_t c1, float s);
    virtual int    row_add  (size_t r1, vector_float* r);
    virtual int    row_sub  (size_t r1, vector_float* r);
    virtual int    row_scale(size_t c1, float s, vector_float* r);
    virtual int    row_saxpy(size_t i, float a, vector_float* r);
    /*@}*/

    /**
     * Matrix - vector operations
     */
    virtual int dot (bool tranA, vector_float* x, vector_float* r);

    /*@{*/
    
    /** 
     * Matrix - Matrix operations
     */
    // this <- this + b
    virtual int add (matrix_float* b);
    // this <- this - b
    virtual int sub (matrix_float* b);
    // this <- this . b
    virtual int dot (matrix_float* b);
    // this <- this ./ b
    virtual int div (matrix_float* b);
    
    // offensichtlich
    virtual void transpose();
    virtual matrix_float* transpose_copy();
    // Print out to file or stdout or whatever
    virtual void print();

    /*@}*/




  };
}

#endif 
