// File: dense_matrix.cc 
// Author: Suvrit Sra
// Time-stamp: <23 February 2010 10:41:51 AM CET --  suvrit>
// Implments the dense matrix using gsl_matrix


#include "dense_matrix.h"
#include "util.h"

#include <typeinfo>
extern "C" {
#include <f2c.h>
#include "clapack.h"
}


void SSLib::dense_matrix::error(std::string s)
{
  std::cerr << "dense_matrix: " << s << std::endl;
}

int SSLib::dense_matrix::add (matrix* b)
{
  if (!dimension_match(b))
    return -1;
  if (typeid(b) == typeid(this)) {
    gsl_matrix_add(M, ((dense_matrix*)b)->get_matrix());
    return 0;
  } else
    return default_add (b);
}

int SSLib::dense_matrix::sub (matrix* b)
{
  if (!dimension_match(b))
    return -1;
  if (typeid(b) == typeid(this)) {
    gsl_matrix_sub(M, ((dense_matrix*)b)->get_matrix());
    return 0;
  } else
    return default_sub (b);
}


int SSLib::dense_matrix::dot (matrix* b)
{
  if (!dimension_match(b))
    return -1;
  if (typeid(b) == typeid(this)) {
    gsl_matrix_mul_elements(M, ((dense_matrix*)b)->get_matrix());
    return 0;
  } else
    return default_dot (b);
}


int SSLib::dense_matrix::div (matrix* b)
{
  if (!dimension_match(b))
    return -1;
  if (typeid(b) == typeid(this)) {
    gsl_matrix_div_elements(M, ((dense_matrix*)b)->get_matrix());
    return 0;
  } else
    return default_div (b);
}

int SSLib::dense_matrix::load(const char* s)
{
  gsl_matrix* t = M;
  M = SSUtil::read_gsl_matrix(const_cast<char*>(s));
  if (M) {
    if (t != 0)
      gsl_matrix_free (t);
    matrix_setsize(M->size1, M->size2);
    return 0;
  } else {
    M = t;
    return 1;
  }
}

int SSLib::dense_matrix::load(const char* s, bool asbin)
{
  gsl_matrix* t = M;
  if (asbin)
    M = SSUtil::fread_gsl_matrix(const_cast<char*>(s));
  else 
    M = SSUtil::read_gsl_matrix(const_cast<char*>(s));

  if (M) {
    if (t != 0)
      gsl_matrix_free(t);
    matrix_setsize(M->size1, M->size2);
    return 0;
  } else {
    M = t;
    return 1;
  }
}

int SSLib::dense_matrix::save(const char* s, bool asbin, int typ)
{
  switch (typ) {
  case M_SELF:
  case M_DENSE:
    return save(s, asbin); 
  case M_CCS:
    return save_as_ccs(s, asbin);
  case M_CRS:
    return save_as_crs(s, asbin);
  case M_HB:
    return save_as_hb(s, asbin);
  case M_COORD_CCS:
    return save_as_coord(s, asbin);
  case M_MATLAB:
    return save_as_matlab(s, asbin);
  case M_ROWCOORD:
    return save_as_rowcoord(s, asbin);
  case M_COLCOORD:
    return save_as_colcoord(s, asbin);
  default:
    matrix_error("DENSE_MATRIX: INVALID MATRIX TYPE REQUESTED in SAVE(...)");
    return -2;
  }
}

int SSLib::dense_matrix::save(const char* s, bool asbin)
{
  if (asbin)
    return  SSUtil::fwrite_gsl_matrix(M, const_cast<char*>(s));
  else {
    return SSUtil::write_gsl_matrix(M, const_cast<char*>(s));
  }
}				

/// Write out matrix to disk in CCS, CRS, etc. formats
int SSLib::dense_matrix::save_as_ccs(const char*s, bool asbin)
{
  if (asbin)
    return save_ccs_binary(s);
  else
    return save_ccs_text(s);
}

int SSLib::dense_matrix::save_ccs_binary(const char* s)
{
  FILE* fp = fopen(s, "wb");
  if (!fp) {
    std::cerr << "DENSE_MATRIX: could not open " << s << " for writing (bin)\n";
    return -1;
  }
  // Write the dimension info
  size_t m = nrows();
  size_t n = ncols();
  size_t nz = m*n;

  for (size_t i = 0; i < m*n; i++)
    if (M->data[i] == 0) --nz;

  if (fwrite(&m,  sizeof(size_t), 1, fp) != 1) {
    error("Error writing M (numrows) to file");
    fclose(fp);
    return -2;
  }

  if (fwrite(&n,  sizeof(size_t), 1, fp) != 1) {
    error("Error writing N (numcols) to file");
    fclose(fp);
    return -2;
  }

  
  if (fwrite(&nz, sizeof(size_t), 1, fp) != 1) {
    error ("Error writing NZ to file"); 
    fclose(fp);
    return -2;
  }
  
  // Write out the colptrs array
  // First build it temporarily
  size_t* cp = (size_t*) calloc ( n+1, sizeof(size_t));
  size_t* ri = (size_t*) malloc ( sizeof(size_t) * nz);
  double* va = (double*) malloc ( sizeof(double) * nz);
  
  size_t ctr = 0;
  for (size_t c = 0; c < n; c++) {
    cp[c+1]=cp[c];
    for (size_t r = 0; r < m; r++) {
      double t = gsl_matrix_get(M, r, c);
      if (t != 0) {
        ++cp[c+1];
        ri[ctr] = r;
        va[ctr] = t;
        ++ctr;
      }
    }
  }

  cp[n] = nz;
  // write the colptrs
  if (fwrite(cp, sizeof(size_t), n+1, fp) != n+1) {
    error ("Error writing colptrs");
    free(cp); free(ri); free(va);
    fclose(fp);
    return -3;
  }

  // write the row indices
  if (fwrite(ri, sizeof(size_t), nz, fp) != nz) {
    error ("Error writing rowindices");
    free(cp); free(ri); free(va);
    fclose(fp);
    return -3;
  }
  // write the nonzeros themselves
  if (fwrite(va, sizeof(double), nz, fp) != nz) {
    error ("Error writing nonzeros");
    free(cp); free(ri); free(va);
    return -3;
  }

  free(cp); free(ri); free(va);
  fclose(fp);
  return 0;
}

int SSLib::dense_matrix::save_ccs_text(const char* s)
{
  std::string pfix = std::string(s);

  std::string fname = pfix + "_dim";

  FILE* fp = fopen(fname.c_str(), "w");
  if (!fp) {
    std::cerr << "DENSE_MATRIX: could not open " << fname 
              << " for writing (txt)\n"; 
    return -1;
  }
  // Write the dimension info
  size_t m = nrows();
  size_t n = ncols();
  size_t nz = m*n;

  for (size_t i = 0; i < m*n; i++)
    if (M->data[i] == 0) --nz;

  fprintf(fp, "%zu %zu %zu\n", m, n, nz);
  fclose(fp);

  
  
  // Write out the colptrs array
  // First build it temporarily
  size_t* cp = (size_t*) calloc ( n+1, sizeof(size_t));
  size_t* ri = (size_t*) malloc ( sizeof(size_t) * nz);
  double* va = (double*) malloc ( sizeof(double) * nz);
  
  size_t ctr = 0;
  for (size_t c = 0; c < n; c++) {
    cp[c+1]=cp[c];
    for (size_t r = 0; r < m; r++) {
      double t = gsl_matrix_get(M, r, c);
      if (t != 0) {
        ++cp[c+1];
        ri[ctr] = r;
        va[ctr] = t;
        ++ctr;
      }
    }
  }

  cp[n] = nz;
  // write the colptrs
  fname = pfix + "_col_ccs";
  fp = fopen(fname.c_str(), "w");
  if (!fp) {
    std::cerr << "DENSE_MATRIX: could not open " << fname 
              << " for writing (txt)\n"; 
    return -1;
  }
  for (size_t i = 0; i < n; i++)
    fprintf(fp, "%zu\n", cp[i]);
  fprintf(fp, "%zu", nz);
  fclose(fp);

  // Write the row indices
  fname = pfix + "_row_ccs";
  fp = fopen(fname.c_str(), "w");
  if (!fp) {
    std::cerr << "DENSE_MATRIX: could not open " << fname 
              << " for writing (txt)\n"; 
    return -1;
  }
  for (size_t i = 0; i < nz -1 ; i++)
    fprintf(fp, "%zu\n", ri[i]);
  fprintf(fp, "%zu", ri[nz-1]);
  fclose(fp);

  // write the nonzeros themselves
  // Write the row indices
  fname = pfix + "_txx_nz";
  fp = fopen(fname.c_str(), "w");
  if (!fp) {
    std::cerr << "DENSE_MATRIX: could not open " << fname 
              << " for writing (txt)\n"; 
    return -1;
  }
  for (size_t i = 0; i < nz -1 ; i++)
    fprintf(fp, "%lf\n", va[i]);
  fprintf(fp, "%lf", va[nz-1]);
  fclose(fp);
  
  free(cp); free(ri); free(va);

  return 0;
}

int SSLib::dense_matrix::save_as_crs(const char* s, bool asbin)
{
  if (asbin)
    return save_crs_binary(s);
  else
    return save_crs_text(s);

}

int SSLib::dense_matrix::save_crs_binary(const char* s)
{
  FILE* fp = fopen(s, "wb");
  if (!fp) {
    std::cerr << "DENSE_MATRIX: could not open " << s << " for writing (bin)\n";
    return -1;
  }
  // Write the dimension info
  size_t m = nrows();
  size_t n = ncols();
  size_t nz = m*n;

  for (size_t i = 0; i < m*n; i++)
    if (M->data[i] == 0) --nz;

  if (fwrite(&m,  sizeof(size_t), 1, fp) != 1) {
    error("Error writing M (numrows) to file");
    fclose(fp);
    return -2;
  }

  if (fwrite(&n,  sizeof(size_t), 1, fp) != 1) {
    error("Error writing N (numcols) to file");
    fclose(fp);
    return -2;
  }

  
  if (fwrite(&nz, sizeof(size_t), 1, fp) != 1) {
    error ("Error writing NZ to file"); 
    fclose(fp);
    return -2;
  }
  
  // First build CRS data temporarily
  size_t* rp = (size_t*) calloc ((m+1), sizeof(size_t));
  size_t* ci = (size_t*) malloc ( sizeof(size_t) * nz);
  double* va = (double*) malloc ( sizeof(double) * nz);
  
  size_t ctr = 0;
  for (size_t r = 0; r < m; r++) {
    rp[r+1]=rp[r];
    for (size_t c = 0; c < n; c++) {
      double t = gsl_matrix_get(M, r, c);
      if (t != 0) {
        ++rp[r+1];
        ci[ctr] = c;
        va[ctr] = t;
        ++ctr;
      }
    }
  }
  
  rp[n] = nz;
  // write the colptrs
  if (fwrite(rp, sizeof(size_t), m+1, fp) != m+1) {
    error ("Error writing colptrs");
    free(rp); free(ci); free(va);
    fclose(fp);
    return -3;
  }

  // write the row indices
  if (fwrite(ci, sizeof(size_t), nz, fp) != nz) {
    error ("Error writing rowindices");
    free(rp); free(ci); free(va);
    fclose(fp);
    return -3;
  }
  // write the nonzeros themselves
  if (fwrite(va, sizeof(double), nz, fp) != nz) {
    error ("Error writing nonzeros");
    free(rp); free(ci); free(va);
    return -3;
  }

  free(rp); free(ci); free(va);
  fclose(fp);
  return 0;
}

int SSLib::dense_matrix::save_crs_text(const char* s)
{
  std::string pfix = std::string(s);

  std::string fname = pfix + "_dim";

  FILE* fp = fopen(fname.c_str(), "w");
  if (!fp) {
    std::cerr << "DENSE_MATRIX: could not open " << fname 
              << " for writing (txt)\n"; 
    return -1;
  }
  // Write the dimension info
  size_t m = nrows();
  size_t n = ncols();
  size_t nz = m*n;

  for (size_t i = 0; i < m*n; i++)
    if (M->data[i] == 0) --nz;

  fprintf(fp, "%zu %zu %zu\n", m, n, nz);
  fclose(fp);

  // First build CRS temporarily
  size_t* rp = (size_t*) calloc (m+1, sizeof(size_t));
  size_t* ci = (size_t*) malloc ( sizeof(size_t) * nz);
  double* va = (double*) malloc ( sizeof(double) * nz);
  
  size_t ctr = 0;
  for (size_t r = 0; r < m; r++) {
    rp[r+1]=rp[r];
    for (size_t c = 0; c < n; c++) {
      double t = gsl_matrix_get(M, r, c);
      if (t != 0) {
        ++rp[r+1];
        ci[ctr] = c;
        va[ctr] = t;
        ++ctr;
      }
    }
  }

  rp[n] = nz;
  // write the colptrs
  fname = pfix + "_row_ccs";
  fp = fopen(fname.c_str(), "w");
  if (!fp) {
    std::cerr << "DENSE_MATRIX: could not open " << fname 
              << " for writing (txt)\n"; 
    return -1;
  }
  for (size_t i = 0; i < m; i++)
    fprintf(fp, "%zu\n", rp[i]);
  fprintf(fp, "%zu", nz);
  fclose(fp);
  
  // Write the row indices
  fname = pfix + "_col_ccs";
  fp = fopen(fname.c_str(), "w");
  if (!fp) {
    std::cerr << "DENSE_MATRIX: could not open " << fname 
              << " for writing (txt)\n"; 
    return -1;
  }
  for (size_t i = 0; i < nz -1 ; i++)
    fprintf(fp, "%zu\n", ci[i]);
  fprintf(fp, "%zu", ci[nz-1]);
  fclose(fp);

  // write the nonzeros themselves
  // Write the row indices
  fname = pfix + "_txx_nz";
  fp = fopen(fname.c_str(), "w");
  if (!fp) {
    std::cerr << "DENSE_MATRIX: could not open " << fname 
              << " for writing (txt)\n"; 
    return -1;
  }
  for (size_t i = 0; i < nz -1 ; i++)
    fprintf(fp, "%f\n", va[i]);
  fprintf(fp, "%f", va[nz-1]);
  fclose(fp);
  
  free(rp); free(ci); free(va);

  return 0;
}


int SSLib::dense_matrix::save_as_coord(const char* s, bool asbin)
{
  return -1;
}

int SSLib::dense_matrix::save_as_rowcoord(const char* s, bool asbin)
{
  return -1;
}

int SSLib::dense_matrix::save_as_colcoord(const char* s, bool asbin)
{
  size_t nz = 0;
  FILE* fp = fopen(s, "w");
  for (size_t i = 0; i < nrows(); i++) {
    for (size_t j = 0; j < ncols(); j++) {
      double t = get(i, j);
      if (t != 0) {
        fprintf(fp, "%zu:%lf ", j+1, t);
        ++nz;
      }
    }
    fprintf(fp, "\n");
  }
  fclose(fp);
  std::string dim(s);
  dim += ".dim";
  fp = fopen(dim.data(), "w");
  if (!fp) {
    fprintf(stderr, "Failed to open dim file\n");
    return -1;
  }
  fprintf(fp, "%zu %zu %zu", nrows(), ncols(), nz);
  fclose(fp);
  return 0;
  
}

int SSLib::dense_matrix::save_as_matlab(const char* s, bool asbin)
{
  return -1;
}

int SSLib::dense_matrix::save_as_hb(const char* s, bool asbin)
{
  return -1;
}

int SSLib::dense_matrix::save_as_dense(const char* s, bool asbin)
{
  return save(s, asbin);
}


void SSLib::dense_matrix::normalize_columns()
{
  for (size_t c = 0; c < ncols(); c++) {
    double cn = col_norm(c, 2);
    col_scale(c, 1.0/cn);
  }
}

/* Elementwise functions on this matrix */


/// Vector l_p norms for this matrix
/// @todo Faster exponents
/// @param 'p', for lp norms.
double SSLib::dense_matrix::norm (double p)
{
  double nrm = 0.0;
  // we take advantage of non-negativity while computing the matrix norms. 
  // Also we compute exponents faster TODO
  if (is_non_negative()) {
    if (p == 1) {
      nrm = SSUtil::sum(M->data, nrows()*ncols());
    } else if (p == 2) {
      nrm = SSUtil::euclidean_norm(M->data, nrows()*ncols());
    } else {
      nrm = SSUtil::lp_norm(M->data, nrows()*ncols(), p);
    }
  } else {
    if (p == 1) {
      nrm = SSUtil::abs_sum(M->data, nrows()*ncols());
    } else if (p == 2) {
      nrm = SSUtil::euclidean_norm(M->data, nrows()*ncols());
    } else {
      nrm = SSUtil::lp_norm(M->data, nrows()*ncols(), p, true);
    }
  }
  return nrm;
}

/**
 * @param p specifies 'fro', 'l1', 'matrix-l1', 'inf', 'matrix-inf'
 * @return the specified norm
 * or return -1 if could not compute specified norm...
 */
double SSLib::dense_matrix::norm(char* p) 
{
  std::string s(p);
  if (s == "fro") {
    return norm(2);
  } else if (s == "l1") {
    return norm(1);
  } else if (s == "inf") {
    return SSUtil::linf_norm(M->data, nrows()*ncols());
  } else if (s == "matrix-l1") {
    matrix_error("Not yet implemented!");
  } else if (s == "matrix-inf") {
    matrix_error ("Not yet implemented");
  } else {
    matrix_error ("Invalid or unsupported norm requested");
  }
  return -1;
}

// Apply an arbitrary function elementwise to this matrix
double SSLib::dense_matrix::apply (double (* fn)(double))
{
  for (size_t i = 0; i < nrows()*ncols(); i++) {
    M->data[i] = fn(M->data[i]);
  }
  return 0.0;
}

/* Operations on columns of this matrix */

double SSLib::dense_matrix::col_norm(size_t c)
{
  return col_norm(c, 2);
}

/**
 * @param c      the column whose norm to computer
 * @param p      compute l_p norm
 * @return
 *    -   -1 if invalid 'p' norm requested
 *    -   -2 if invalid column specified
 *    -   double value as the requested norm
 */
double SSLib::dense_matrix::col_norm (size_t c, double p)
{
  gsl_vector_view col = gsl_matrix_column(M,c);
  if (p == 1) {
    return gsl_blas_dasum(&col.vector);
  }
  if (p == 2) {
    return gsl_blas_dnrm2(&col.vector);
  }
  return -1;
}

double SSLib::dense_matrix::col_dot  (size_t c1, size_t c2)
{
  double r;
  gsl_vector_view col1 = gsl_matrix_column(M, c1);
  gsl_vector_view col2 = gsl_matrix_column(M, c2);
  gsl_blas_ddot(&col1.vector, &col2.vector, &r);
  return r;
}

double SSLib::dense_matrix::col_dot  (size_t c, vector* v)
{
  double r;
  gsl_vector_view col = gsl_matrix_column(M, c);
  gsl_blas_ddot(&col.vector, v, &r);
  return r;
}

int    SSLib::dense_matrix::col_add  (size_t c1, size_t c2)
{

  return 0;
}

int    SSLib::dense_matrix::col_sub  (size_t c1, size_t c2)
{
  return 0;
}

int SSLib::dense_matrix::col_sum(vector* r)
{
  gsl_vector* v = gsl_vector_alloc(nrows());
  gsl_vector_set_all(v, 1.0);
  dot(true, v, r);
  gsl_vector_free(v);
  return 0;
}

int    SSLib::dense_matrix::col_scale(size_t c1, double s)
{
  gsl_vector_view col = gsl_matrix_column(M,c1);
  gsl_blas_dscal(s, &col.vector);
  return 0;
}

/**
 * @todo
 */
int SSLib::dense_matrix::col_sub (size_t c1, vector* c2)
{
  error("NOT IMPLEMENTED");
  return -1;
}

/**
 * @todo
 */
int SSLib::dense_matrix::col_add (size_t c1, vector* c2)
{
  gsl_vector_view col = gsl_matrix_column(M, c1);
  gsl_vector_add(c2, &col.vector);
  return 0;
}

/**
 * Scaled col c1, and stores result in a preallocated vector r
 * r <--- r + s*column
 * @return
 *    -    0 if computtaion was successful
 *    -    1 if dimension mismatch
 *    -    2 if invalid column requested
 */
int SSLib::dense_matrix::col_scale(size_t c1, double s, vector* r)
{
  if (c1 < 0 || c1 > ncols())
    return -2;

  if (r->size != nrows())
    return -1;

  for (size_t i = 0; i < nrows(); i++)
    r->data[i] += s * gsl_matrix_get(M, i, c1);

  return 0;
}

double SSLib::dense_matrix::row_norm(size_t r)
{
  return row_norm(r, 2);
}

/* Operations on rows of this matrix */
double SSLib::dense_matrix::row_norm (size_t r, double p)
{
  gsl_vector_view row = gsl_matrix_row(M,r);
  if (p == 1) {
    return gsl_blas_dasum(&row.vector);
  }
  if (p == 2) {
    return gsl_blas_dnrm2(&row.vector);
  }
  return 0.0;
}

double SSLib::dense_matrix::row_dot  (size_t r1, size_t r2)
{
  gsl_vector_view row1 = gsl_matrix_row(M, r1);
  gsl_vector_view row2 = gsl_matrix_row(M, r2);
  double r;
  gsl_blas_ddot(&row1.vector, &row2.vector, &r);
  return r;
}

double SSLib::dense_matrix::row_dot(size_t r, vector* x, size_t* ix, size_t ixn)
{
  return row_dot(r, x);
}

double SSLib::dense_matrix::row_dot  (size_t r, vector* v)
{
  gsl_vector_view row = gsl_matrix_row(M, r);
  double res;
  gsl_blas_ddot(&row.vector, v, &res);
  return res;
}

double SSLib::dense_matrix::row_col_dot(size_t r, size_t c)
{
  if (nrows() != ncols()) {
    error("Dimension mismatch in row_col_dot");
    return 0.0;
  }
  double res = 0;
  gsl_vector_view row = gsl_matrix_row(M, r);
  gsl_vector_view col = gsl_matrix_column(M, c);
  gsl_blas_ddot(&row.vector, &col.vector, &res);
  return res;
}

int    SSLib::dense_matrix::row_add  (size_t r1, size_t r2)
{
  gsl_vector_view row1 = gsl_matrix_row(M, r1);
  gsl_vector_view row2 = gsl_matrix_row(M, r2);
  gsl_vector_add(&row1.vector, &row2.vector);
  return 0;
}

int    SSLib::dense_matrix::row_sub  (size_t r1, size_t r2)
{
  gsl_vector_view row1 = gsl_matrix_row(M, r1);
  gsl_vector_view row2 = gsl_matrix_row(M, r2);
  gsl_vector_sub(&row1.vector, &row2.vector);
  return 0;
}

int    SSLib::dense_matrix::row_scale(size_t c1, double s)
{
  gsl_vector_view row = gsl_matrix_row(M,c1);
  gsl_vector_scale(&row.vector, s);
  return 0;
}
 
int SSLib::dense_matrix::row_add (size_t r1, vector* v)
{
  gsl_vector_view row = gsl_matrix_row(M, r1);
  gsl_vector_add(&row.vector, v);
  return 0;
}

int SSLib::dense_matrix::row_sub (size_t r1, vector* v)
{
  gsl_vector_view row = gsl_matrix_row(M, r1);
  gsl_vector_sub(&row.vector, v);
  return 0;
}

int SSLib::dense_matrix::row_sum(vector* r)
{
  gsl_vector* v = gsl_vector_alloc(ncols());
  gsl_vector_set_all(v, 1.0);
  dot(false, v, r);
  gsl_vector_free(v);
  return 0;
}

int SSLib::dense_matrix::row_scale(size_t c1, double s, vector* r)
{
  gsl_vector_view row = gsl_matrix_row(M, c1);
  gsl_vector_set_zero(r);
  gsl_blas_daxpy(s, &row.vector, r);
  return 0;
}

int SSLib::dense_matrix::row_daxpy(size_t i, double a, vector* r)
{
  gsl_vector_view row = gsl_matrix_row(M,i);
  gsl_blas_daxpy(a, &row.vector, r);
  return 0;
}

/** @name MatrixFunctions
 *
 * Functions or operations involving entire matrix
 */
/*@{*/

/**
 * Function does result <- Ax
 * @param x The vector to be multiplied by this,
 * @param result the *UNALLOCATED* pointer result, which  will be allocated
 * to hold the result if the computation can be carried out.
 * @return
 *    -    0 if computation was ok
 *    -    1 if computation could not be carried otu (dimension mismatch)
 */
int SSLib::dense_matrix::dot (bool tranA, vector* x, vector* result)
{
  if (tranA)
    gsl_blas_dgemv(CblasTrans, 1.0, M, x, 0.0, result);
  else
    gsl_blas_dgemv(CblasNoTrans, 1.0, M, x, 0.0, result);
  return 0;
}

int SSLib::dense_matrix::mul(matrix* b)
{
  // Call the appropriate version of DGEMM_ for doing the mult....
#ifdef _HAVE_GOTO_BLAS_
  // call goto's dgemm_
#elif _HAVE_BLAS_
  // 
#else // use default gsl mul routine

#endif
  return -1;
}

void SSLib::dense_matrix::transpose()
{
  return;
}

SSLib::matrix* SSLib::dense_matrix::transpose_copy()
{
  return 0;
}

int SSLib::dense_matrix::scale(double s)
{
  for (size_t i = 0; i < nrows()*ncols(); i++)
    M->data[i] *= s;
  return 0;
}


void SSLib::dense_matrix::print()
{
  for (size_t i = 0; i < nrows(); i++) {
    for (size_t j = 0; j < ncols(); j++)
        std::cout << get(i, j) << " ";
    std::cout << std::endl;
  }
	
  return;
}

int SSLib::dense_matrix::add_const(double s)
{
  for (size_t i = 0; i < nrows()*ncols(); i++)
    M->data[i] += s;
  return 0;
}

int SSLib::dense_matrix::compute_AtA(SSLib::matrix* result)
{
  // Should throw an exception if 'result' is not a dense matrix
  // But will ignore that as of now....
  SSLib::dense_matrix* dresult = static_cast<SSLib::dense_matrix*>(result);
  gsl_blas_dgemm(CblasTrans, CblasNoTrans, 1.0, M, M, 0.0, dresult->M);
  return 0;
}

int SSLib::dense_matrix::solve_psd_linear_system(vector* b, vector* x)
{
  char uplo = 'L';
  integer k = ncols();
  integer nrhs = 1;
  integer info;
  gsl_vector_memcpy(x, b);
  dposv_(&uplo, &k, &nrhs, M->data, &k, x->data, &k, &info);
  int rv = (int) info;
  return rv;
}

int SSLib::dense_matrix::solve_least_squares(vector* b, vector* x)
{
  char trans = 'N';
  integer m = M->size1;
  integer n = M->size2;
  integer nrhs = 1;
  integer lda = M->size1;
  integer ldb = M->size1;
  doublereal work[1];
  doublereal* actualwork;
  integer lwork = -1;
  integer info = 0;
  //gsl_vector_memcpy(x, b);
  gsl_matrix* copyA = SSUtil::prepare_for_clapack(M);
  
  dgels_(&trans, &m, &n, &nrhs, copyA->data, &lda, b->data, &ldb, 
         work,  &lwork, &info);
  actualwork = new doublereal[(int)work[0]];
  lwork = (int) work[0];
  dgels_(&trans, &m, &n, &nrhs, copyA->data, &lda, b->data, &ldb, 
         actualwork,  &lwork, &info);

  for (size_t i = 0; i < x->size; i++)
    x->data[i] = b->data[i];
  return (int) info;
}

#ifdef _INTERNAL_EIGS_
/// Returns top 'k' eigenpairs 
SSLib::eigenpairs* SSLib::dense_matrix::eigs (size_t k)
{
  std::cerr << "NOT YET IMPLMENTED!\n";
  return 0;
}
    
/// Returns singular triplets
SSLib::singular_triples* SSLib::dense_matrix::svds (size_t k)
{
  std::cerr << "NOT YET IMPLMENTED!\n";
  return 0;
}


#endif
