// File: crs_float.cc
// Time-stamp: <23 February 2010 10:51:02 AM CET --  suvrit>
// Implements the single precision compressed column storage class

#include "crs_float.h"
#include "exceptions.h"
#include "ccs_float.h"
#include <cmath>
#include <cstring>
#include <algorithm>
#include "util.h"

#include <fstream>

using namespace SSLib;

/**
 * @param m num of rows in matrix
 * @param n num of cols in matrix
 * @param nz num of nonzeros
 * @throws sparsematrix_float::exception_malloc if an invalid matrix size is
 * requested or the new operator fails to allocate storage.
 */
int crs_float::matrix_alloc (size_t m, size_t n, size_t nz)
{
  if (m == 0 || n == 0) {
    sparsematrix_exception* e = new 
      exception_malloc("Invalid matrix size requested...", m, n);
    throw e;
  }
  
  setsize(m, n, nz);

  // Allocate the arrays
  m_rowptrs = (size_t*) malloc ( sizeof (size_t) * (m+1));
  m_colindx = (size_t*) malloc ( sizeof (size_t) * nz);
  m_val     = (float*) malloc ( sizeof (float) * nz);
  m_rowptrs[m] = nz;
  if (m_rowptrs == 0 || m_colindx == 0 || m_val == 0) {
    sparsematrix_exception* e = new 
      exception_malloc("CRS_FLOAT: Memory allocation error"); 
    throw e;
  }

  m_isExternal = false;
  return 0;
}


/**
 * @param m num of rows in matrix
 * @param n num of cols in matrix
 * @param nz num of nonzeros
 * @throws sparsematrix::exception_malloc if an invalid matrix size is
 * requested or the new operator fails to allocate storage.
 */
int crs_float::matrix_calloc (size_t m, size_t n, size_t nz)
{
  if (m == 0 || n == 0) {
    sparsematrix_exception* e = new exception_malloc("Invalid matrix size requested...", m, n);
    throw e;
  }
  
  setsize(m, n, nz);

  // Allocate the arrays
  m_rowptrs = (size_t*) calloc (m+1, sizeof (size_t));
  m_colindx = (size_t*) calloc (nz,  sizeof (size_t));
  m_val     = (float*) malloc ( sizeof (double) * nz);
  m_rowptrs[m] = nz;
  if (m_rowptrs == 0 || m_colindx == 0 || m_val == 0) {
    sparsematrix_exception* e = new exception_malloc("CRS_FLOAT: Memory allocation error"); 
    throw e;
  }

  m_isExternal = false;
  return 0;
}

int crs_float::matrix_resize (size_t m, size_t n, size_t nz)
{
  if (m == 0 || n == 0) {
    sparsematrix_exception* e = new 
      exception_malloc("Invalid matrix size requested...", m, n);
    throw e;
  }

  // Try to resize the arrays
  m_rowptrs = (size_t*) realloc (m_rowptrs, sizeof(size_t) * (m+1));
  if (m_rowptrs == NULL) {
    std::cerr << "ROWPTRS: Could not resize matrix\n";
    return -1;
  }

  m_colindx = (size_t*) realloc (m_colindx, sizeof(size_t) * nz);
  if (m_colindx == NULL) {
    std::cerr << "ROWINDX: Could not resize matrix\n";
    return -1;
  }

  m_val = (float*) realloc (m_val, sizeof(float) * nz);
  if (m_val == NULL) {
    std::cerr << "VAL: Could not resize matrix\n";
    return -1;
  }
  
  setsize(m, n, nz);
  return 0;
}
 
/**
* @param prefix -- the file name prefix that is attached to the 4 ccs format
* files.
* @see m_filename
* @return
*   -     0 if file loaded successfully
*   -     1 if file loading error
*/
int crs_float::load (const char* prefix)
{
  std::cerr << "int crs_float::load (const char* prefix)\n";
  m_filename = std::string(prefix);
  return load_text(prefix);
}

int crs_float::load(const char* fp, bool asbin)
{
  if (asbin) {
    return load_binary(fp);
  } else {
    return load_text(fp);
  }
  return 0;
}

int crs_float::load_binary(const char* fname)
{
  m_filename = std::string(fname);
  FILE* fp = fopen (fname, "rb");
  if (fp == 0) {
    std::string s = "Could not open file " + m_filename;
    error(s);
    return 1;
  }

  size_t m;
  size_t n;
  size_t nz;
  int mt;
  if (SSLib::read_file_header_bin(fp, &m, &n, &nz, &mt, M_CRS_FLOAT) < 0) {
    fclose(fp);
    return -2;
  }
  
  setsize(m, n, nz);
  // Allocate the arrays
  m_rowptrs = (size_t*) malloc (sizeof (size_t) * (m+1));
  m_colindx = (size_t*) malloc (sizeof (size_t) * nz);
  m_val     = (float*) malloc (sizeof (float) * nz);
   
  // REad the colptrs
  if (fread(m_rowptrs, sizeof(size_t), m+1, fp) != m+1) {
    error ("Error reading rowptrs");
    return 3;
  }

  // REad the row indices
  if (fread(m_colindx, sizeof(size_t), nz, fp) != nz) {
    error ("Error reading column indices");
    return 3;
  }
  // REad the nonzeros themselves
  if (fread(m_val, sizeof(float), nz, fp) != nz) {
    error ("Error reading nonzeros");
    return 3;
  }
  fclose(fp);
  return 0;
}


/**
 * Read n a CRS file
 */
int crs_float::load_text(const char* prefix)
{
  // This function assumes that m_type is already set
  m_filename = std::string(prefix);
  // Set up the file names that we are gonna open
  std::string dim(m_filename);
  std::string rows_file(dim + "_row_ccs");
  std::string cols_file(dim + "_col_ccs");
  std::string nz_file (dim + "_"+ m_type + "_nz");
  
  dim +=  "_dim";
  
  std::ifstream infile;
  infile.open(dim.data());
  if (infile.fail()) {
    std::string s = "Error: could not open " + dim;
    error(s);
    return 1;
  }
  
  size_t m, n, nz;

  infile >> m >> n >> nz;
  infile.close();
  infile.clear();

  // Allocate the arrays
  m_rowptrs = (size_t*) malloc (sizeof (size_t) * (m+1));
  m_colindx = (size_t*) malloc (sizeof (size_t) * nz);
  m_val     = (float*) malloc (sizeof (float) * nz);
  m_rowptrs[m] = nz; // Add the convenience value
    
  setsize (m, n, nz);

  // Now read in all the columnpointers
  infile.open(cols_file.data());
  if (infile.fail()) {
    std::string s = "Error: could not open " + cols_file;
    error(s);
    free_data();
    return 2;
  }
  // Make sure the UB on i is correct.
  for (size_t i = 0; i < nz; i++) 
    infile >> m_colindx[i];
  infile.close();
  infile.clear();
  
  infile.open(rows_file.data());
  if (infile.fail()) {
    std::string s = "Error: could not open " + rows_file;
    error(s);
    free_data();
    return 3;
  }

  for (size_t i = 0; i < m; i++)
    infile >> m_rowptrs[i];
  infile.close();
  infile.clear();
  
  // Now read in the actual m_values
  infile.open(nz_file.data());
  if (infile.fail()) {
    std::string s = "Error: could not open " + nz_file;
    error(s);
    free_data();
    return 4;
  }
  
  for (size_t i = 0; i < nz; i++)
    infile >> m_val[i];
  infile.close();

  return 0;
  
}

// -----------------------------------------------------------------


/// Write out matrix to disk in CCS, CRS, etc. formats
int crs_float::save(const char* s, bool asbin, int typ)
{
  switch (typ) {
  case M_SELF:
  case M_CRS_FLOAT:
    return save(s, asbin); 
  case M_DENSE_FLOAT:
    return save_as_dense(s, asbin);
  case M_CCS_FLOAT:
    return save_as_ccs(s, asbin);
  case M_HB_FLOAT:
    return save_as_hb(s, asbin);
  case M_COORD_CRS_FLOAT:
    return save_as_coord(s, asbin);
  case M_MATLAB_FLOAT:
    return save_as_matlab(s, asbin);
  case M_ROWCOORD_FLOAT:
    return save_as_rowcoord(s, asbin);
  case M_COLCOORD_FLOAT:
    return save_as_colcoord(s, asbin);
  default:
    matrix_error("CRS_FLOAT: INVALID MATRIX TYPE REQUESTED in SAVE(...)");
    return -2;
  }
}
/**
 * @param file, the name of the file to which the CCS matrix should be saved
 * in binary format. This matrix can be later read by using load with the
 * asbin argument set to true.
 * @return
 *  -   0 if file saved successfully
 *  -   1 if file saving error
 */
int crs_float::save (const char* s, bool asbin) 
{
  if (asbin)
    return save_binary(s);
  return save_text(s);
}

int crs_float::save_as_ccs(const char* fil, bool asbin)
{
  ccs_float* tmp = to_ccs();
  if (!tmp) 
    return -1;
  tmp->save(fil, asbin);
  tmp->free_data();
  delete tmp;
  return 0;
}

/**
 *
 */
int crs_float::save_as_colcoord(const char* fil, bool asbin)
{
  if (asbin)
    return save_colcoord_bin(fil);
  return save_colcoord_txt(fil);
}

int crs_float::save_colcoord_bin(const char* fil)
{
  return -1;
}
int crs_float::save_colcoord_txt(const char* fil)
{
  return -1;
}
// -----------------------------------------------------------------

int crs_float::save_as_rowcoord(const char* fil, bool asbin)
{
  ccs_float* tmp = to_ccs();
  if (!tmp) 
    return -1;
  tmp->save_as_rowcoord(fil, asbin);
  tmp->free_data();
  delete tmp;
  return 0;
}
/**
 *
 */
int crs_float::save_as_matlab(const char* fil, bool asbin)
{
  if (asbin)
    return save_matlab_bin(fil);
  return save_matlab_txt(fil);
}
int crs_float::save_matlab_bin(const char* fil)
{
  return -1;
}
int crs_float::save_matlab_txt(const char* fil)
{
  FILE* fp = fopen(fil, "w");
  if (!fp) {
    std::string s = "Error: could not open " + std::string(fil) + " for writing";
    error(s);
    return -1;
  }

  for (size_t r = 0; r < nrows(); r++) {
    for (size_t c = m_rowptrs[r]; c < m_rowptrs[r+1]; c++) {
      fprintf(fp, "%zu %zu %f\n", r+1, m_colindx[c]+1, m_val[c]);
    }
  }
  fclose(fp);
  return 0;
}
// -----------------------------------------------------------------

/**
 *
 */
int crs_float::save_as_hb(const char* fil, bool asbin)
{
  ccs_float* tmp = to_ccs();
  if (!tmp) 
    return -1;
  tmp->save_as_hb(fil, asbin);
  tmp->free_data();
  delete tmp;
  return 0;
}

/**
 *
 */
int crs_float::save_as_dense(const char* fil, bool asbin)
{
  if (asbin)
    return save_dense_bin(fil);
  else
    return save_dense_txt(fil);
}
int crs_float::save_dense_bin(const char* s)
{
  FILE* fp = fopen(s, "wb");
  if (!fp) {
    std::cerr << "CRS_FLOAT: could not open " << s << " for writing (bin)\n";
    return -1;
  }
  size_t m = nrows();
  size_t n = ncols();
  size_t st;
  st=fwrite(&m, sizeof(size_t), 1, fp);
  st=fwrite(&n, sizeof(size_t), 1, fp);

  // FIXXXXXXXXXXXXXXXXXXXXX!!!
  for (size_t i = 0; i < nrows(); i++)
    for (size_t j = 0; j < ncols(); j++) {
      double t = get(i, j);
      st=fwrite(&t, sizeof(float), 1, fp);
    }

  fclose(fp);
  return 0;
}

int crs_float::save_dense_txt(const char* s)
{
  FILE* fp = fopen(s, "w");
  if (!fp) {
    std::cerr << "CRS_FLOAT: could not open " << s << " for writing (txt)\n";
    return -1;
  }
  fprintf(fp, "%zu %zu\n", nrows(), ncols());
  for (size_t i = 0; i < nrows(); i++)
    for (size_t j = 0; j < ncols(); j++) {
      double t = get(i, j);
      fprintf(fp, "%f\n", t);
    }
  fclose(fp);
  return 0;
}


/**
 *
 */
int crs_float::save_as_coord (const char* fil, bool asbin)
{
  if (asbin)
    return save_coord_bin(fil);
  else
    return save_coord_txt(fil);
}
int crs_float::save_coord_bin(const char* fil)
{
  FILE* fp = fopen(fil, "wb");
  if (!fp) {
    std::string s = "Error: could not open " + std::string(fil) + " for writing";
    error(s);
    return -1;
  }

  size_t m = nrows();
  size_t n = ncols();
  size_t nz = nonzeros();
  int mt = M_COORD_CRS_FLOAT;
  size_t st;
  if (SSLib::write_file_header_bin(fp, &m, &n, &nz, &mt) < 0) 
    return -2;

  for (size_t r = 0; r < nrows(); r++) {
    for (size_t c = m_rowptrs[r]; c < m_rowptrs[r+1]; c++) {
      size_t ri = m_colindx[r];
      size_t cp = r;
      double v = m_val[r];

      st=fwrite(&cp, sizeof(size_t), 1, fp);
      st=fwrite(&ri, sizeof(size_t), 1, fp);
      st=fwrite(&v, sizeof(double), 1, fp);
    }
  }
  fclose(fp);
  return 0;
}
int crs_float::save_coord_txt(const char* fil)
{
  FILE* fp = fopen(fil, "w");
  if (!fp) {
    std::string s = "Error: could not open " + std::string(fil) + " for writing";
    error(s);
    return -1;
  }

  size_t m = nrows();
  size_t n = ncols();
  size_t nz = nonzeros();
  int mt = M_COORD_CRS_FLOAT;

  if (SSLib::write_file_header_txt(fp, m, n, nz, mt) < 0) 
    return -2;

  for (size_t r = 0; r < nrows(); r++) {
    for (size_t c = m_rowptrs[r]; c < m_rowptrs[r+1]; c++) {
      fprintf(fp, "%zu %zu %f\n", r, m_colindx[c], m_val[c]);
    }
  }
  fclose(fp);
  return 0;
}
// 


int crs_float::save_text(const char* s)
{
  // Set up the file names that we are gonna open
  std::string dim(s);
  std::string rows_file(dim + "_row_ccs");
  std::string cols_file(dim + "_col_ccs");
  std::string nz_file (dim + "_"+ m_type + "_nz");
  
  dim +=  "_dim";
  
  std::ofstream ofile;
  ofile.open(dim.data());
  if (ofile.fail()) {
    std::string s = "Error: could not open for writing " + dim;
    error(s);
    return -1;
  }
  
  ofile << nrows()
        << " " << ncols()
        << " " << nonzeros();
  
  ofile.close();
  ofile.clear();
  
  // Now write out all the row pointers
  ofile.open(rows_file.data());
  if (ofile.fail()) {
    std::string s = "Error: could not open for writing: " + rows_file;
    error(s);
    return -2;
  }
  // Make sure the UB on i is correct.
  for (size_t i = 0; i < nrows() + 1; i++) 
    ofile << m_rowptrs[i] << std::endl;
  ofile.close();
  ofile.clear();
  
  ofile.open(cols_file.data());
  if (ofile.fail()) {
    std::string s = "Error: could not open for writing: " + cols_file;
    error(s);
    return -3;
  }

  for (size_t i = 0; i < nonzeros(); i++)
    ofile << m_colindx[i] << std::endl;
  ofile.close();
  ofile.clear();
  
  // Now write out the actual m_values
  ofile.open(nz_file.data());
  if (ofile.fail()) {
    std::string s = "Error: could not open " + nz_file;
    error(s);
    return -4;
  }
  
  for (size_t i = 0; i < nonzeros(); i++)
    ofile << m_val[i] << std::endl;
  ofile.close();

  return 0;
}

/**
 * @param file, the name of the file to which the CCS matrix should be saved
 * in binary format. This matrix can be later read by using load with the
 * asbin argument set to true.
 * @return
 *  -   0 if file saved successfully
 *  -   1 if file saving error
 */
int crs_float::save_binary(const char* file)
{
  FILE* fp = fopen (file, "wb");
  if (fp == 0) {

    return -1;
  }

  size_t m = nrows();
  size_t n = ncols();
  size_t nz = nonzeros();
  int mt = M_CRS_FLOAT;

  
  if (SSLib::write_file_header_bin(fp, &m, &n, &nz, &mt) < 0)
    return -2;
  
  // Write the rowptrs
  if (fwrite(m_rowptrs, sizeof(size_t), m+1, fp) != m+1) {
    error ("Error saving colptrs");
    return -3;
  }

  // Write the col indices
  if (fwrite(m_colindx, sizeof(size_t), nz, fp) != nz) {
    error ("Error saving rowindices");
    return -3;
  }
  // Write the nonzeros themselves
  if (fwrite(m_val, sizeof(float), nz, fp) != nz) {
    error ("Error saving nonzeros");
    return -3;
  }
  fclose(fp);
  return 0;
}


void crs_float::externally_alloc (long* col, long* row, float* val)
{


}

// Return a deep copy
crs_float* crs_float::clone() 
{
  return 0;
}

// Augment matrix by adding a row
int crs_float::matrix_add_row(vector_float* r)
{
  return 0;
}

// Augment by adding a column
int crs_float::matrix_add_col(vector_float* c)
{
  return 0;
}

// Augment by adding a sparsematrix_float
int crs_float::matrix_add_mat(sparsematrix_float* m)
{
  return 0;
}

/* Accessing the matrix elements */

// Return row 'r' of matrix
vector_float* crs_float::get_row(size_t r)
{
  vector_float* row = vector_float_calloc(ncols());
  // We go thru each column, and see if row 'r' exists, if so we pick up the
  // element and put it in the vector, else we carry on...
  for (size_t c = m_rowptrs[r]; c < m_rowptrs[r+1]; c++) {
    row->data[ m_colindx[c] ] = m_val[c];
  }
  return row;
}
 
// Return col 'c' of matrix
vector_float* crs_float::get_col(size_t c) 
{
  
  return 0;
}

// return the (i, j) entry of the matrix
float  crs_float::get (size_t i, size_t j)
{
  size_t sz = m_rowptrs[i+1]-m_rowptrs[i];
  int idx = SSUtil::binary_search(m_colindx + m_rowptrs[i], j, sz);
  if (idx != -1) 
    return (m_val + m_rowptrs[i])[idx];
  return 0.0;
}

/// Sets the specified row to the given vector
int crs_float::set_row(size_t r, vector_float*)
{
  return 0;
}

/// Sets the specified col to the given vector
int crs_float::set_col(size_t c, vector_float*)
{
  return 0;
}

/// Sets the specified diagonal to the given vector
int crs_float::set_diag(bool p, vector_float*)
{
  return 0;
}


/// Returns main or second diagonal (if p == true)
vector_float* crs_float::get_diag(bool p  )
{
  return 0;
}

/// Returns a submatrix that goes from row i1--i2 and cols j1--j2
matrix_float* crs_float::submatrix(size_t i1, size_t j1, size_t i2, size_t j2)
{
  return 0;
}


/* Elementwise functions on this matrix */


// Vector l_p norms for this matrix
double crs_float::norm (double p)
{
  return 0;
}

// Apply an arbitrary function elementwise to this matrix
float crs_float::apply (float (* fn)(float))
{
  
  return 0;
}

double crs_float::col_dot  (size_t c1, size_t c2)
{
  return 0.0;
}

double crs_float::col_dot  (size_t c, vector_float* v)
{
  return 0.0;

}

int    crs_float::col_add  (size_t c1, size_t c2)
{
  return 0;
}

int    crs_float::col_sub  (size_t c1, size_t c2)
{
  return 0;
}

int    crs_float::col_scale(size_t c1, float s)
{
  return 0;
}

int crs_float::col_sub (size_t c1, vector_float* c2)
{
  return 0;
}

int crs_float::col_add (size_t c1, vector_float* c2)
{
  return 0;
}

int crs_float::col_scale(size_t c1, float s, vector_float* r)
{
  return 0;
}

double crs_float::col_norm(size_t c, double p)
{
  return 0;
}

/* Operations on rows of this matrix */
double crs_float::row_norm (size_t r, double p)
{
  double nrm = 0.0;

  if (p == 1) {
    for (size_t c = m_rowptrs[r]; c < m_rowptrs[r+1]; c++)
      nrm += fabs(m_val[c]);
    return nrm;
  } else if (p == 2) {
    for (size_t c = m_rowptrs[r]; c < m_rowptrs[r+1]; c++)
      nrm += m_val[c] * m_val[c];
    return sqrt(nrm);
  } else
    return 0.0;
}

double crs_float::row_dot  (size_t r1, size_t r2)
{
  double dp = 0.0;
  vector_float* row = get_row(r2);
  dp = row_dot(r1, row);
  vector_float_free(row);
  return dp;
}

int crs_float::row_avg(vector_float* res)
{
  double s = 0.0;
  double sz;
  for (size_t r = 0; r < nrows(); r++) {
    s = 0.0; sz = m_rowptrs[r+1]-m_rowptrs[r];
    for (size_t j = m_rowptrs[r]; j < m_rowptrs[r+1]; j++) {
      s += m_val[j];
    }
    res->data[r] = s/sz;
  }
  return 0;
}

int crs_float::row_num_nonzeros(vector_float* res)
{
  for (size_t r = 0; r < nrows(); r++)
    res->data[r] = m_rowptrs[r+1]-m_rowptrs[r];

  return 0;
}

int crs_float::row_sum(vector_float* res)
{
  double s = 0.0;
  double sz;
  for (size_t r = 0; r < nrows(); r++) {
    s = 0.0; sz = 0.0;
    for (size_t j = m_rowptrs[r]; j < m_rowptrs[r+1]; j++) {
      s += m_val[j];
      ++sz;
    }
    res->data[r] = s;
  }
  return 0;
}

double crs_float::row_dot  (size_t r, vector_float* v)
{
  double dp = 0.0;

  for (size_t c = m_rowptrs[r]; c < m_rowptrs[r+1]; c++)
    dp += m_val[c] * v->data [ m_colindx [c] ];

  return dp;
}

int    crs_float::row_add  (size_t r1, size_t r2)
{
  std::cerr << "****** NOT IMPLEMENTED YET *******\n";
  return 0;
}

int    crs_float::row_sub  (size_t r1, size_t r2)
{
  std::cerr << "****** NOT IMPLEMENTED YET *******\n";
  return 0;
}

int    crs_float::row_scale(size_t r, float s)
{
  for (size_t c = m_rowptrs[r]; c < m_rowptrs[r+1]; c++)
    m_val[c] *= s;
  return 0;
}
 
int crs_float::leading_svd(vector_float* u, vector_float* v, int m)
{
  // Init u
  // v <- A'u, u <- Av
  gsl_vector_float_set_all(u,1.0);
  gsl_vector_float_set_all(v, 1.0);
  for (int i = 0; i < m; i++) {
    dot(true, u, v);
    dot(false, v, u);
  }
  return 0;
}


/**
 * This function computes y := A*x or y := A'*x
 * @param tranA, whether to use A or A' (true means use A')
 * @param x      A dense vector
 * @param y      Output (must be a preallocated vector)
 * @return
 *   -   0 If computation was ok
 *   -   1 Computation failed due to dimension mismatch
 */
int crs_float::dot(bool tranA, vector_float*x, vector_float* y)
{
  if (tranA)
    return matrix_tran_vec(x, y);
  else
    return matrix_vec(x, y);
}

int crs_float::row_add (size_t r1, vector_float* v)
{
  std::cerr << "****** NOT IMPLEMENTED YET *******\n";
  return 0;
}

int crs_float::row_sub (size_t r1, vector_float* v)
{
  std::cerr << "****** NOT IMPLEMENTED YET *******\n";
  return 0;
}

int crs_float::row_scale(size_t c1, float s, vector_float* r)
{
  std::cerr << "****** NOT IMPLEMENTED YET *******\n";
  return 0;
}

int crs_float::row_saxpy(size_t i, float a, vector_float* r)
{
  for (size_t c = m_rowptrs[i]; c < m_rowptrs[i+1]; c++) {
    r->data[ m_colindx[c] ] += a*m_val[c];
  }
  return 0;
}

/* Functions or operations involving entire matrix */
// Matrix vec mpy, result = Ax
int crs_float::matrix_vec (vector_float* x, vector_float* result)
{
  std::cerr << "****** NOT IMPLEMENTED YET *******\n";
  return -1;
}

// Transpose[Matrix].vec = A'x
int crs_float::matrix_tran_vec (vector_float* x, vector_float* result)
{
  std::cerr << "****** NOT IMPLEMENTED YET *******\n";
  return -1;
}

// Multiplication of two sparse matrices this = this * m or m * this
int crs_float::mult_sparse (sparsematrix_float* m, int leftOrRight)
{
  return 0;
}

// Multiplication of sparse matrix with dense matrix
int crs_float::mult_dense (matrix_float* m, int leftOrRight)
{
  return 0;
}

// Add two sparse matrices
int crs_float::add_sparse (sparsematrix_float* m)
{
  return 0;
}

// Subtract two sparse matrices
int crs_float::sub_sparse (sparsematrix_float* m)
{
  return 0;
}

// Add dense to sparse
int crs_float::add (matrix_float* m)
{
  return 0;
}

// Sub dense 
int crs_float::sub (matrix_float* m)
{
  return 0;
}

// Hadamard product 
int crs_float::dot (matrix_float* m)
{
  return 0;
}

int crs_float::matrix_dot (sparsematrix_float* m)
{
  return 0;
}


void crs_float::print()
{

  std::cerr << nrows() << " x " << ncols() << std::endl;
  for (size_t i = 0; i < nrows(); i++) {
    for (size_t j = 0; j < ncols(); j++)
      std::cout << get(i,j) << " ";
    std::cout << std::endl;
  }

  // std::cout << nrows() << " x " << ncols() << std::endl;
//   size_t p = 0;
//   for (size_t r = 0; r < nrows(); ++r){
//     size_t colnr = 0; 
//     if (m_rowptrs[r+1] == p) {
//       for ( ; colnr < ncols(); ++colnr){
//         std::cout << 0 << " ";
//       }
//     } else {
//       for (; p < m_rowptrs[r+1]; ++p){
//         while (colnr < m_colindx[p]){
//           std::cout << 0 << " ";
//           ++colnr;
//         }
//         std::cout << m_val[p] << " ";
//         ++colnr;
//       }
//       while (colnr < ncols()){
//         std::cout << 0 << " ";
//         ++colnr;
//       }
//     }
//     std::cout << std::endl;
//   }
}


/**
 * @author  stefje
 * Converts a matrix in CRS format to a matrix in CCS format.
 */
ccs_float* crs_float::to_ccs()
{
  ccs_float* newm = new SSLib::ccs_float();

  size_t m = nrows();
  size_t n = ncols();
  size_t nz = nonzeros();

  std::cerr << "Current matrix is: " << m << " x " 
            << n << " # " << nz << "\n";

  size_t* ncp, *nrind;
  float* nval;

  newm->matrix_alloc(m, n, nz);

  size_t* colind = (size_t*) malloc( sizeof(size_t)* (n+1));

  if (colind == 0) {
    newm->free_data();
    delete newm;
    return 0;
  }

  memset(colind, 0, (n + 1)*sizeof(size_t));

  ncp   = newm->get_colptrs();
  nrind = newm->get_rowindx();
  nval  = newm->get_data();

  memset(ncp, 0, (n+1)*sizeof(size_t));

  //count the nonzeros in each column
  for (size_t i = 0; i < nz; ++i) {
    ncp[ m_colindx[i] + 1 ]++;
  }

  //add for column pointers
  for (size_t i = 1; i< n; ++i)  {
    ncp[i+1] += ncp[i];
  }

  //go through the rows and copy the values into the proper locations in newm
  //at the same time, insert the row indices
  for (size_t r = 0; r < m; r++) { //row r
    for (size_t i = m_rowptrs[r]; i < m_rowptrs[r+1]; ++i){
      size_t ind = ncp[ m_colindx[i] ] + colind[ m_colindx[i] ];
      ++colind[m_colindx[i]];

      nval[ind] = m_val[i];
      nrind[ind] = r;
    }
  }


  free(colind);
  return newm;
}

void crs_float::free_data()
{
  if (m_val) free(m_val); m_val = 0;
  if (m_rowptrs) free(m_rowptrs); m_rowptrs = 0;
  if (m_colindx) free(m_colindx); m_colindx = 0;
}

void crs_float::error (std::string s)
{
  std::cerr << "CRS_FLOAT:"  << s << std::endl;
}

/**
 * @param s  a scalar by which all the lements of the matrix are scaled.
 * @return 0
 */
int crs_float::scale(float s)
{
  for (size_t i = 0; i < NZ; i++)
    m_val[i] *= s;
  return 0;
}

/**
 * @param s  A scalar that is added to all the nonzeros of the matrix
 */
int crs_float::add_const(float s)
{
  for (size_t i = 0; i < NZ; i++)
    m_val[i] += s;
  return 0;
}
