// File: crs.h  -*- c++ -*-
// Time-stamp: <08 July 2007 04:45:40 PM CDT --  suvrit>
// Impl. of crs format sparse matrices

#ifndef _SPARSE_CRS_H
#define _SPARSE_CRS_H

#include "sparsematrix.h"

namespace SSLib {
  class ccs;

  /**
   * @class crs, contains an implementation of the Compressed Row Storage
   * (CRS) sparse matrices. This class is symmetric to CCS, and is more
   * suitable for use with sparse row action methods. It contains
   * conversions to the CCS format, in case needed.
   */
  class crs : public sparsematrix {
    /**
     * The actual non-zeroes themselves.
     */
  protected:
    double* m_val;

    /**
     * Rowptrs for CRS structure. These hold indices into the
     * m_colindx and val. The i-th row starts at
     * m_rowptrs[i] and ends at m_rowptrs[i+1]-1
     */
    size_t* m_rowptrs;
  

    /**
     * Col indices for CRS structure. This array contain the col indices of
     * the matrix elements. For example, m_colindx[ m_rowptrs[i] ] gives the
     * col index of the 1st non-zero element in the i-th row.
     */
    size_t* m_colindx;
    
  protected:
    /**
     * Is the data in externally allocated arrays. That is, if the m_val, 
     * m_rowptrs and m_colindx arrays are externally allocated (useful for
     * creating a crs sparsematrix from another matrix of similar structure
     * etc. 
     */
    bool m_isExternal;

    /**
     * Filename prefix. A CRS matrix is loaded from a collection of 4
     * files. For example, if m_filename="mat", then the code that loads
     * this matrix will search for the files 'mat_dim', 'mat_row_crs',
     * 'mat_col_crs' and 'mat_<m_type>_nz' depending on the value of
     * <code>m_type</code>      
     */
    std::string m_filename;

    /* matrix type, 'txx', 'tfn', etc. */
    std::string m_type;

  protected:
    int load_binary(const char* fname);
    int load_text  (const char* prefx);
    int save_binary(const char* fname);
    int save_text  (const char* prefx);
  public:
    crs () : sparsematrix() {
      m_isExternal = false;
      m_filename = "";
      m_type = "txx";
      m_val = 0;
      m_rowptrs = 0;
      m_colindx = 0;
    }

    virtual ~crs () {    }

    void free_data();

    /** @name MatrixCreation
     * Creating and saving crs matrix functions. These functions allow the
     * user to load a CRS matrix from file, or from already available
     * arrays. They also allow the user to save the current matrix to file.
     *
     */

    /*@{*/


    // Allocate the matrix
    virtual int matrix_alloc (size_t m, size_t n, size_t nz);
    virtual int matrix_calloc (size_t m, size_t n, size_t nz);
    virtual int matrix_resize (size_t m, size_t n, size_t nz);

    virtual int load(const char*, bool asbin);
    


    /// Write out matrix in a specified way
    virtual int    save(const char*, bool asbin, int typ);
    virtual int save (const char*, bool asbin);
    /// Write out matrix to disk in CCS, CRS, etc. formats
    virtual int save_as_ccs (const char*, bool);

    virtual int save_as_coord (const char*, bool);
    virtual int save_coord_bin(const char*);
    virtual int save_coord_txt(const char*);

    virtual int save_as_colcoord (const char*, bool);
    virtual int save_colcoord_bin(const char*);
    virtual int save_colcoord_txt(const char*);

    virtual int save_as_rowcoord (const char*, bool);

    virtual int save_as_matlab (const char*, bool);
    virtual int save_matlab_bin(const char* fil);
    virtual int save_matlab_txt(const char*);

    virtual int save_as_hb (const char*, bool);

    virtual int save_as_dense (const char*, bool);
    virtual int save_dense_bin(const char*);
    virtual int save_dense_txt(const char*);

    void externally_alloc (long* col, long* row, double* val);
    // Return a deep copy
    crs* clone();

    // Augment matrix by adding a row
    int matrix_add_row(vector* r);

    // Augment by adding a column
    int matrix_add_col(vector* c);

    // Augment by adding a sparsematrix
    int matrix_add_mat(sparsematrix* m);
    
    /*@}*/

    /** @name Accessors
     * Accessing the matrix elements 
     */
    /*@{*/

    /// For efficiency i had to do this nasty thing.
    size_t* get_rowptrs() { return m_rowptrs;}
    size_t* get_colindx() { return m_colindx;}
    double* get_data() { return m_val;}

    /// Return row 'r' of matrix
    virtual int get_row(size_t r, vector*& row);
    /// Return col 'c' of matrix
    virtual int get_col(size_t c, vector*& col);
    /// return the (i, j) entry of the matrix

    double  get (size_t i, size_t j);
    double  operator()(size_t i, size_t j){ return 0.0; }
    /// set the (i, j) entry of the matrix.
    int     set (size_t i, size_t j, double val) {return 1;}
    
    /// Sets the specified row to the given vector
    virtual int set_row(size_t r, vector*&);
    /// Sets the specified col to the given vector
    virtual int set_col(size_t c, vector*&);
    /// Returns main or second diagonal (if p == true)
    virtual int get_diag(bool p, vector*&);
    /// Sets the specified diagonal to the given vector
    virtual int set_diag(bool p, vector*&);
    /// Returns a submatrix that goes from row i1--i2 and cols j1--j2
    virtual int submatrix(size_t i1, size_t j1, size_t i2, size_t j2, matrix*&);
    
    /// Returns a submatrix given the row-col index sets
    virtual int submatrix(gsl_vector_uint* I, gsl_vector_uint* J, matrix*& m) {
      return -1;
    }

    /*@}*/

    /** @name Elementwise
     * Elementwise functions/operations on this matrix.
     */
    /*@{*/

    // Vector l_p norms for this matrix
    virtual double norm (double p);
    
    virtual double norm(char* p){ return 0;}

    // Apply an arbitrary function elementwise to this matrix
    virtual double apply (double (* fn)(double));

    virtual int scale(double d){return 0;}
    virtual int add_const(double d){ return 0;}

    /*@}*/

    /*@{*/

    virtual double col_norm (size_t c);
    virtual double col_dot  (size_t c1, size_t c2) ;
    virtual double col_dot  (size_t c, vector* v);
    virtual int    col_add  (size_t c1, size_t c2) ;
    virtual int    col_sub  (size_t c1, size_t c2) ;
    virtual int    col_scale(size_t c1, double s);
    virtual int    col_sub  (size_t c1, vector* v);
    virtual int    col_add  (size_t c1, vector* v) ;
    virtual int    col_scale(size_t c1, double s, vector* r);
    virtual int    col_sum  (vector*);

    virtual double col_norm(size_t, double);
    virtual double row_norm(size_t, double);
    /*@}*/

    
    /** @name
     * Operations on rows of this matrix
     */
    /*@{*/
    virtual double row_norm (size_t r);
    virtual double row_dot  (size_t r1, size_t r2) ;
    virtual double row_dot  (size_t r, vector* v);
    virtual double row_dot  (size_t r, vector* x, size_t* ix,size_t);
    virtual int    row_add  (size_t r1, size_t r2) ;
    virtual int    row_sub  (size_t r1, size_t r2) ;
    virtual int    row_scale(size_t c1, double s);
    virtual int    row_add  (size_t r1, vector* r);
    virtual int    row_sub  (size_t r1, vector* r);
    virtual int row_scale(size_t c1, double s, vector* r);
    virtual int row_daxpy(size_t i, double a, vector* r);
    virtual int    row_sum  (vector*);

    /// dot ( row i, col j)
    virtual double row_col_dot (size_t r, size_t c);
    /*@}*/

    /*@{*/

    // Matrix vec mpy, result = Ax
    virtual int matrix_vec (vector* x, vector* result);


    // Transpose[Matrix].vec = A'x
    virtual int matrix_tran_vec (vector* x, vector* result);


    /**
     * Subtract a sparse matrix from this matrix. The operation implemented is 
     * this := this + m.
     * @param m is a sparsematrix* arg
     * @return 
     *   -     0 if subtraction succeeded
     *   -     1 if size mismatch
     *   -     2 if some other error
     * @warning This function can cause fill-in by destroying non-zeros, and
     * can be slow.
     */
    virtual int add_sparse (sparsematrix* m);

    /**
     * Subtract a sparse matrix from this matrix. The operation implemented is 
     * this := this - m.
     * @param m is a sparsematrix* arg
     * @return 
     *   -     0 if subtraction succeeded
     *   -     1 if size mismatch
     *   -     2 if some other error
     * @warning This function can cause fill-in by destroying non-zeros, and
     * can be slow.
     */  
    virtual int sub_sparse (sparsematrix* m);

    /**
     * Add a dense matrix from this matrix. The operation implemented is 
     * this := this + m
     * @param m is a gsl_matrix* arg
     * @return
     *   -     0 if subtraction succeeded
     *   -     1 if size mismatch
     *   -     2 if some other error
     * @warning This function can cause fill-in by destroying non-zeros, and
     * can be slow.
     */
    virtual int add (matrix* m);

    /**
     * Subtract a dense matrix from this matrix. The operation implemented is 
     * this := this - m
     * @param m is a gsl_matrix* arg
     * @return
     *   -     0 if subtraction succeeded
     *   -     1 if size mismatch
     *   -     2 if some other error
     * @warning This function can cause fill-in by destroying non-zeros, and
     * can be slow.
     */ 
    virtual int sub (matrix* m);

    /**
     * Hadamard product. After the execution of this function our current
     * matrix is: this := this .* m (in matlab notation)
     * @param m is a gsl_matrix* argument
     * @return 
     *  -  0 if the dot product was carried out
     *  -  1 if matrix size mismatch
     *  -  2 if any other error
     */
    virtual int dot (matrix* m);

    virtual int mul (matrix* m);

    virtual int dot(bool, vector*, vector*);

    /**
     * Hadamard product. After the execution of this function our current
     * matrix is: this := this .* m (in matlab notation). This function does
     * a runtime type check to determine the type of 'm' to optimize the dot
     * product operation (since two ccs matrices or ccs * harwell boeing can
     * be dotted much faster without using any get(i, j) operations).
     * @param m is a sparsematrix* argument
     * @return 
     *  -  0 if the dot product was carried out
     *  -  1 if matrix size mismatch
     *  -  2 if any other error
     */
    virtual int matrix_dot (sparsematrix* m);

    virtual int div (matrix* m){ return -1; }
    
    virtual int matrix_div (sparsematrix* m) { return -1;}

    /*@}*/

    /** @name Misc 
     * Miscellanous functions
     */
    /*@{*/

    /**
     * @return true if this matrix is external, i.e., the rows, cols
     * etc. arrays are directly initialized from outside, and not allocated
     * by us.
     */
    bool isExternal() const { return m_isExternal;}
    virtual int compute_AtA(matrix* result);
    virtual void transpose() {}

    /// Normalize the columns of the matrix to have unit L2 norm
    virtual void normalize_columns() {}

    matrix* transpose_copy() { return 0; }
    void print();

    ccs* to_ccs();

    virtual void error (std::string s);
    /*@}*/
  };
}

#endif // _SPARSE_CRS_H
