// File: crs.cc
// Time-stamp: <23 February 2010 10:39:53 AM CET --  suvrit>
// Implements the compressed column storage class

#include "crs.h"
#include "exceptions.h"
#include "ccs.h"
#include <cstring>
#include <cmath>
#include <fstream>

#include "coord_struct.h"

#include "util.h"
using namespace SSLib;

/**
 * @param m num of rows in matrix
 * @param n num of cols in matrix
 * @param nz num of nonzeros
 * @throws sparsematrix::exception_malloc if an invalid matrix size is
 * requested or the new operator fails to allocate storage.
 */
int crs::matrix_alloc (size_t m, size_t n, size_t nz)
{
  if (m == 0 || n == 0) {
    sparsematrix_exception* e = new exception_malloc("Invalid matrix size requested...", m, n);
    throw e;
  }
  
  setsize(m, n, nz);

  // Allocate the arrays
  m_rowptrs = (size_t*) malloc ( sizeof (size_t) * (m+1));
  m_colindx = (size_t*) malloc ( sizeof (size_t) * nz);
  m_val     = (double*) malloc ( sizeof (double) * nz);
  m_rowptrs[m] = nz;
  if (m_rowptrs == 0 || m_colindx == 0 || m_val == 0) {
    sparsematrix_exception* e = new exception_malloc("CRS: Memory allocation error"); 
    throw e;
  }

  m_isExternal = false;
  return 0;
}


/**
 * @param m num of rows in matrix
 * @param n num of cols in matrix
 * @param nz num of nonzeros
 * @throws sparsematrix::exception_malloc if an invalid matrix size is
 * requested or the new operator fails to allocate storage.
 */
int crs::matrix_calloc (size_t m, size_t n, size_t nz)
{
  if (m == 0 || n == 0) {
    sparsematrix_exception* e = new exception_malloc("Invalid matrix size requested...", m, n);
    throw e;
  }
  
  setsize(m, n, nz);

  // Allocate the arrays
  m_rowptrs = (size_t*) calloc (m+1, sizeof (size_t));
  m_colindx = (size_t*) calloc (nz,  sizeof (size_t));
  m_val     = (double*) malloc ( sizeof (double) * nz);
  m_rowptrs[m] = nz;
  if (m_rowptrs == 0 || m_colindx == 0 || m_val == 0) {
    sparsematrix_exception* e = new exception_malloc("CRS: Memory allocation error"); 
    throw e;
  }

  m_isExternal = false;
  return 0;
}

int crs::matrix_resize (size_t m, size_t n, size_t nz)
{
  if (m == 0 || n == 0) {
    sparsematrix_exception* e = new exception_malloc("Invalid matrix size requested...", m, n);
    throw e;
  }

  // Try to resize the arrays
  m_rowptrs = (size_t*) realloc (m_rowptrs, sizeof(size_t) * (m+1));
  if (m_rowptrs == NULL) {
    std::cerr << "ROWPTRS: Could not resize matrix\n";
    return -1;
  }

  m_colindx = (size_t*) realloc (m_colindx, sizeof(size_t) * nz);
  if (m_colindx == NULL) {
    std::cerr << "ROWINDX: Could not resize matrix\n";
    return -1;
  }

  m_val = (double*) realloc (m_val, sizeof(double) * nz);
  if (m_val == NULL) {
    std::cerr << "VAL: Could not resize matrix\n";
    return -1;
  }
  
  setsize(m, n, nz);
  return 0;
}
 
int crs::load(const char* fp, bool asbin)
{
  if (asbin) {
    return load_binary(fp);
  } else {
    return load_text(fp);
  }
  return 0;
}
int crs::load_binary(const char* fname)
{
  m_filename = std::string(fname);
  FILE* fp = fopen (fname, "rb");
  if (fp == 0) {
    std::string s = "Could not open file " + m_filename;
    error(s);
    return 1;
  }

  size_t m;
  size_t n;
  size_t nz;
  int mt;
  if (SSLib::read_file_header_bin(fp, &m, &n, &nz, &mt, M_CRS) < 0) {
    fclose(fp);
    return -2;
  }

  setsize(m, n, nz);
  // Allocate the arrays
  m_rowptrs = (size_t*) malloc (sizeof (size_t) * (m+1));
  m_colindx = (size_t*) malloc (sizeof (size_t) * nz);
  m_val     = (double*) malloc (sizeof (double) * nz);
   
  // REad the colptrs
  if (fread(m_rowptrs, sizeof(size_t), m+1, fp) != m+1) {
    error ("Error reading rowptrs");
    return 3;
  }

  // REad the row indices
  if (fread(m_colindx, sizeof(size_t), nz, fp) != nz) {
    error ("Error reading column indices");
    return 3;
  }
  // REad the nonzeros themselves
  if (fread(m_val, sizeof(double), nz, fp) != nz) {
    error ("Error reading nonzeros");
    return 3;
  }
  fclose(fp);
  return 0;
}
int crs::load_text(const char* prefix)
{
  // This function assumes that m_type is already set
  m_filename = std::string(prefix);
  // Set up the file names that we are gonna open
  std::string dim(m_filename);
  std::string rows_file(dim + "_row_ccs");
  std::string cols_file(dim + "_col_ccs");
  std::string nz_file (dim + "_"+ m_type + "_nz");
  
  dim +=  "_dim";
  
  std::ifstream infile;
  infile.open(dim.data());
  if (infile.fail()) {
    std::string s = "Error: could not open " + dim;
    error(s);
    return 1;
  }
  
  size_t m, n, nz;

  infile >> m >> n >> nz;
  infile.close();
  infile.clear();

  // Allocate the arrays
  m_rowptrs = (size_t*) malloc (sizeof (size_t) * (m+1));
  m_colindx = (size_t*) malloc (sizeof (size_t) * nz);
  m_val     = (double*) malloc (sizeof (double) * nz);
  m_rowptrs[m] = nz; // Add the convenience value
    
  setsize (m, n, nz);

  // Now read in all the columnpointers
  infile.open(cols_file.data());
  if (infile.fail()) {
    std::string s = "Error: could not open " + cols_file;
    error(s);
    free_data();
    return 2;
  }
  // Make sure the UB on i is correct.
  for (size_t i = 0; i < nz; i++) 
    infile >> m_colindx[i];
  infile.close();
  infile.clear();
  
  infile.open(rows_file.data());
  if (infile.fail()) {
    std::string s = "Error: could not open " + rows_file;
    error(s);
    free_data();
    return 3;
  }

  for (size_t i = 0; i < m; i++)
    infile >> m_rowptrs[i];
  infile.close();
  infile.clear();
  
  // Now read in the actual m_values
  infile.open(nz_file.data());
  if (infile.fail()) {
    std::string s = "Error: could not open " + nz_file;
    error(s);
    free_data();
    return 4;
  }
  
  for (size_t i = 0; i < nz; i++)
    infile >> m_val[i];
  infile.close();

  return 0;
  
}


// -----------------------------------------------------------------


/// Write out matrix to disk in CCS, CRS, etc. formats
int crs::save(const char* s, bool asbin, int typ)
{
  switch (typ) {
  case M_SELF:
  case M_CRS:
    return save(s, asbin); 
  case M_DENSE:
    return save_as_dense(s, asbin);
  case M_CCS:
    return save_as_ccs(s, asbin);
  case M_HB:
    return save_as_hb(s, asbin);
  case M_COORD_CRS:
    return save_as_coord(s, asbin);
  case M_MATLAB:
    return save_as_matlab(s, asbin);
  case M_ROWCOORD:
    return save_as_rowcoord(s, asbin);
  case M_COLCOORD:
    return save_as_colcoord(s, asbin);
  default:
    matrix_error("CRS: INVALID MATRIX TYPE REQUESTED in SAVE(...)");
    return -2;
  }
}
/**
 * @param file, the name of the file to which the CCS matrix should be saved
 * in binary format. This matrix can be later read by using load with the
 * asbin argument set to true.
 * @return
 *  -   0 if file saved successfully
 *  -   1 if file saving error
 */
int crs::save (const char* s, bool asbin) 
{
  if (asbin)
    return save_binary(s);
  return save_text(s);
}

int crs::save_text(const char* s)
{
  // Set up the file names that we are gonna open
  std::string dim(s);
  std::string rows_file(dim + "_row_ccs");
  std::string cols_file(dim + "_col_ccs");
  std::string nz_file (dim + "_"+ m_type + "_nz");
  
  dim +=  "_dim";
  
  std::ofstream ofile;
  ofile.open(dim.data());
  if (ofile.fail()) {
    std::string s = "Error: could not open for writing " + dim;
    error(s);
    return -1;
  }
  
  ofile << nrows()
        << " " << ncols()
        << " " << nonzeros();
  
  ofile.close();
  ofile.clear();
  
  // Now write out all the row pointers
  ofile.open(rows_file.data());
  if (ofile.fail()) {
    std::string s = "Error: could not open for writing: " + rows_file;
    error(s);
    return -2;
  }
  // Make sure the UB on i is correct.
  for (size_t i = 0; i < nrows() + 1; i++) 
    ofile << m_rowptrs[i] << std::endl;
  ofile.close();
  ofile.clear();
  
  ofile.open(cols_file.data());
  if (ofile.fail()) {
    std::string s = "Error: could not open for writing: " + cols_file;
    error(s);
    return -3;
  }

  for (size_t i = 0; i < nonzeros(); i++)
    ofile << m_colindx[i] << std::endl;
  ofile.close();
  ofile.clear();
  
  // Now write out the actual m_values
  ofile.open(nz_file.data());
  if (ofile.fail()) {
    std::string s = "Error: could not open " + nz_file;
    error(s);
    return -4;
  }
  
  for (size_t i = 0; i < nonzeros(); i++)
    ofile << m_val[i] << std::endl;
  ofile.close();

  return 0;
}

int crs::save_binary(const char* file)
{
  FILE* fp = fopen (file, "wb");
  if (fp == 0) {

    return -1;
  }

  size_t m = nrows();
  size_t n = ncols();
  size_t nz = nonzeros();
  int mt = M_CRS;
  
  if (SSLib::write_file_header_bin(fp, &m, &n, &nz, &mt) < 0)
    return -2;

  
  // Write the rowptrs
  if (fwrite(m_rowptrs, sizeof(size_t), m+1, fp) != m+1) {
    error ("Error saving colptrs");
    return -3;
  }

  // Write the col indices
  if (fwrite(m_colindx, sizeof(size_t), nz, fp) != nz) {
    error ("Error saving rowindices");
    return -3;
  }
  // Write the nonzeros themselves
  if (fwrite(m_val, sizeof(double), nz, fp) != nz) {
    error ("Error saving nonzeros");
    return -3;
  }
  fclose(fp);
  return 0;
}
// -----------------------------------------------------------------

int crs::save_as_ccs(const char* fil, bool asbin)
{
  ccs* tmp = to_ccs();
  if (!tmp) 
    return -1;
  tmp->save(fil, asbin);
  tmp->free_data();
  delete tmp;
  return 0;
}

/**
 *
 */
int crs::save_as_colcoord(const char* fil, bool asbin)
{
  if (asbin)
    return save_colcoord_bin(fil);
  return save_colcoord_txt(fil);
}
int crs::save_colcoord_bin(const char* fil)
{
  return -1;
}
int crs::save_colcoord_txt(const char* fil)
{
  return -1;
}
// -----------------------------------------------------------------

int crs::save_as_rowcoord(const char* fil, bool asbin)
{
  ccs* tmp = to_ccs();
  if (!tmp) 
    return -1;
  tmp->save_as_rowcoord(fil, asbin);
  tmp->free_data();
  delete tmp;
  return 0;
}
/**
 *
 */
int crs::save_as_matlab(const char* fil, bool asbin)
{
  if (asbin)
    return save_matlab_bin(fil);
  return save_matlab_txt(fil);
}
int crs::save_matlab_bin(const char* fil)
{
  return -1;
}
int crs::save_matlab_txt(const char* fil)
{
  FILE* fp = fopen(fil, "w");
  if (!fp) {
    std::string s = "Error: could not open " + std::string(fil) + " for writing";
    error(s);
    return -1;
  }

  for (size_t r = 0; r < nrows(); r++) {
    for (size_t c = m_rowptrs[r]; c < m_rowptrs[r+1]; c++) {
      fprintf(fp, "%zu %zu %.14F\n", r+1, m_colindx[c]+1, m_val[c]);
    }
  }
  fclose(fp);
  return 0;
}
// -----------------------------------------------------------------

/**
 *
 */
int crs::save_as_hb(const char* fil, bool asbin)
{
  ccs* tmp = to_ccs();
  if (!tmp) 
    return -1;
  tmp->save_as_hb(fil, asbin);
  tmp->free_data();
  delete tmp;
  return 0;
}

/**
 *
 */
int crs::save_as_dense(const char* fil, bool asbin)
{
  if (asbin)
    return save_dense_bin(fil);
  else
    return save_dense_txt(fil);
}
int crs::save_dense_bin(const char* s)
{
  FILE* fp = fopen(s, "wb");
  if (!fp) {
    std::cerr << "CRS: could not open " << s << " for writing (bin)\n";
    return -1;
  }
  size_t m = nrows();
  size_t n = ncols();
  size_t st;
  st=fwrite(&m, sizeof(size_t), 1, fp);
  st=fwrite(&n, sizeof(size_t), 1, fp);

  // FIXXXXXXXXXXXXXXXXXXXXX!!!
  for (size_t i = 0; i < nrows(); i++)
    for (size_t j = 0; j < ncols(); j++) {
      double t = get(i, j);
      st=fwrite(&t, sizeof(double), 1, fp);
    }

  fclose(fp);
  return 0;
}

int crs::save_dense_txt(const char* s)
{
  FILE* fp = fopen(s, "w");
  if (!fp) {
    std::cerr << "CRS: could not open " << s << " for writing (txt)\n";
    return -1;
  }
  fprintf(fp, "%zu %zu\n", nrows(), ncols());
  for (size_t i = 0; i < nrows(); i++)
    for (size_t j = 0; j < ncols(); j++) {
      double t = get(i, j);
      fprintf(fp, "%lf\n", t);
    }
  fclose(fp);
  return 0;
}


/**
 *
 */
int crs::save_as_coord (const char* fil, bool asbin)
{
  if (asbin)
    return save_coord_bin(fil);
  else
    return save_coord_txt(fil);
}
int crs::save_coord_bin(const char* fil)
{
  FILE* fp = fopen(fil, "wb");
  if (!fp) {
    std::string s = "Error: could not open " + std::string(fil) + " for writing";
    error(s);
    return -1;
  }

  size_t m = nrows();
  size_t n = ncols();
  size_t nz = nonzeros();
  int mt = M_COORD_CRS;
  size_t st;

  if (SSLib::write_file_header_bin(fp, &m, &n, &nz, &mt) < 0) 
    return -2;

  

  for (size_t r = 0; r < nrows(); r++) {
    for (size_t c = m_rowptrs[r]; c < m_rowptrs[r+1]; c++) {
      size_t ri = m_colindx[r];
      size_t cp = r;
      double v = m_val[r];

      st=fwrite(&cp, sizeof(size_t), 1, fp);
      st=fwrite(&ri, sizeof(size_t), 1, fp);
      st=fwrite(&v, sizeof(double), 1, fp);
    }
  }
  fclose(fp);
  return 0;
}
int crs::save_coord_txt(const char* fil)
{
  FILE* fp = fopen(fil, "w");
  if (!fp) {
    std::string s = "Error: could not open " + std::string(fil) + " for writing";
    error(s);
    return -1;
  }

  size_t m = nrows();
  size_t n = ncols();
  size_t nz = nonzeros();
  int mt = M_COORD_CRS;

  if (SSLib::write_file_header_txt(fp, m, n, nz, mt) < 0) 
    return -2;

  for (size_t r = 0; r < nrows(); r++) {
    for (size_t c = m_rowptrs[r]; c < m_rowptrs[r+1]; c++) {
      fprintf(fp, "%zu %zu %lf\n", r, m_colindx[c], m_val[c]);
    }
  }
  fclose(fp);
  return 0;
}
// 



void crs::externally_alloc (long* col, long* row, double* val)
{


}

// Return a deep copy
crs* crs::clone() 
{
  return 0;
}

// Augment matrix by adding a row
int crs::matrix_add_row(vector* r)
{
  return 0;
}

// Augment by adding a column
int crs::matrix_add_col(vector* c)
{
  return 0;
}

// Augment by adding a sparsematrix
int crs::matrix_add_mat(sparsematrix* m)
{
  return 0;
}

/* Accessing the matrix elements */

// Return row 'r' of matrix
int crs::get_row(size_t r, vector*& row)
{
  if (!row || row->size < ncols())
    return -1;
  // We go thru each column, and see if row 'r' exists, if so we pick up the
  // element and put it in the vector, else we carry on...
  for (size_t c = m_rowptrs[r]; c < m_rowptrs[r+1]; c++) {
    row->data[ m_colindx[c] ] = m_val[c];
  }
  return 0;
}
 
// Return col 'c' of matrix
int crs::get_col(size_t c, vector*& v) 
{
  if (!v || v->size < nrows())
    return -1;
  int idx;
  for (size_t r = 0; r < nrows(); r++) {
    size_t sz = m_rowptrs[r+1] - m_rowptrs[r];
    idx = SSUtil::binary_search(m_colindx + m_rowptrs[r], c, sz);
    if (idx != -1)
      gsl_vector_set(v, r, (m_val + m_rowptrs[r])[idx]);
  }
  return 0;
}

// return the (i, j) entry of the matrix
double  crs::get (size_t i, size_t j)
{
  size_t sz = m_rowptrs[i+1]-m_rowptrs[i];
  int idx = SSUtil::binary_search(m_colindx + m_rowptrs[i], j, sz);
  if (idx != -1) 
    return (m_val + m_rowptrs[i])[idx];
  return 0.0;
}

/// Sets the specified row to the given vector
int crs::set_row(size_t r, vector*& v)
{
  return -1;
}

/// Sets the specified col to the given vector
int crs::set_col(size_t c, vector*& v)
{
  return -1;
}

/// Sets the specified diagonal to the given vector
int crs::set_diag(bool p, vector*& v)
{
  return -1;
}


/// Returns main or second diagonal (if p == true)
int crs::get_diag(bool p, vector*& v)
{
  return -1;
}

/// Returns a submatrix that goes from row i1--i2 and cols j1--j2
int crs::submatrix(size_t i1, size_t j1, size_t i2, size_t j2, matrix*& m)
{
  return -1;
}


/* Elementwise functions on this matrix */


// Vector l_p norms for this matrix
double crs::norm (double p)
{
  return 0;
}

// Apply an arbitrary function elementwise to this matrix
double crs::apply (double (* fn)(double))
{
  return 0;
}

/* Operations on columns of this matrix */
double crs::col_norm (size_t c)
{
  return 0.0;
}

double crs::col_dot  (size_t c1, size_t c2)
{
  
  return 0.0;
}

/// dot ( row i, col j)
double crs::row_col_dot (size_t r, size_t c)
{
  double dp = 0.0;
  vector* v = vector_alloc(nrows());
  if (get_col(c, v) < 0)
    return 0;
  dp = row_dot(r, v);
  vector_free(v);
  return dp;
}

double crs::col_dot  (size_t c, vector* v)
{
  
  return 0.0;

}

int    crs::col_add  (size_t c1, size_t c2)
{
  return -1;
}

int    crs::col_sub  (size_t c1, size_t c2)
{
  return -1;
}

int    crs::col_scale(size_t c1, double s)
{
  return -1;
}

int crs::col_sub (size_t c1, vector* c2)
{
  return -1;
}

int crs::col_add (size_t c1, vector* c2)
{
  return -1;
}

int crs::col_scale(size_t c1, double s, vector* r)
{
  return -1;
}

int crs::col_sum(vector* r)
{
  return -1;
}

double crs::col_norm(size_t c, double p)
{
  return -1;
}

/* Operations on rows of this matrix */
double crs::row_norm (size_t r, double p)
{
  double nrm = 0.0;

  if (p == 1) {
    for (size_t c = m_rowptrs[r]; c < m_rowptrs[r+1]; c++)
      nrm += fabs(m_val[c]);
    return nrm;
  } else if (p == 2) {
    for (size_t c = m_rowptrs[r]; c < m_rowptrs[r+1]; c++)
      nrm += m_val[c] * m_val[c];
    return sqrt(nrm);
  } else
    return 0.0;
}

double crs::row_norm(size_t r)
{
  return row_norm(r, 2);
}

int crs::dot(bool tranA, vector*x, vector*r)
{
  if (tranA) {
    std::cerr << "****** CRS: TRANA'*x NOT IMPLEMENTED YET *******\n";
  } else {
    for (size_t i = 0; i < nrows(); i++)
      r->data[i] = row_dot(i, x);
  }
  return 0;
}

double crs::row_dot  (size_t r1, size_t r2)
{
  double dp = 0.0;
  vector* row = vector_alloc(ncols());
  if (get_row(r2, row) < 0)
    return 0;
  dp = row_dot(r1, row);
  vector_free(row);
  return dp;
}

double crs::row_dot  (size_t r, vector* v)
{
  double dp = 0.0;
  
  for (size_t c = m_rowptrs[r]; c < m_rowptrs[r+1]; c++)
    dp += m_val[c] * v->data [ m_colindx [c] ];

  return dp;
}

double crs::row_dot  (size_t r, vector* x, size_t* ix, size_t ixn)
{
  size_t i, j;
  double dp = 0;
  size_t *idx1;
  double* dx, *dy;
  size_t nnz;
  nnz  = m_rowptrs[r+1] - m_rowptrs[r];
  idx1 = (m_colindx + m_rowptrs[r]); 
  dx   = (m_val + m_rowptrs[r]); dy = x->data;
  
  i = j = 0;

  while (i < nnz && j < ixn) {
    if (idx1[i] == ix[j]) {
      dp += dx[i] * dy[ix[j]];
      ++i; ++j;
    } else if (idx1[i] < ix[j]) {
      ++i;
    } else {
      ++j;
    }
  }
  return dp;
}

int    crs::row_add  (size_t r1, size_t r2)
{
  std::cerr << "CRS:row_add ****** NOT IMPLEMENTED YET *******\n";
  return 0;
}

int    crs::row_sub  (size_t r1, size_t r2)
{
  std::cerr << "CRS:row_sub ****** NOT IMPLEMENTED YET *******\n";
  return 0;
}

int    crs::row_scale(size_t r, double s)
{
  for (size_t c = m_rowptrs[r]; c < m_rowptrs[r+1]; c++)
    m_val[c] *= s;
  return 0;
}
 
int crs::row_add (size_t r1, vector* v)
{
  std::cerr << "CRS:row_add (vector)****** NOT IMPLEMENTED YET *******\n";
  return 0;
}

int crs::row_sub (size_t r1, vector* v)
{
  std::cerr << "CRS:row_sub (vector) ****** NOT IMPLEMENTED YET *******\n";
  return 0;
}

int crs::row_scale(size_t c1, double s, vector* r)
{
  std::cerr << "CRS:row_scale (vector)****** NOT IMPLEMENTED YET *******\n";
  return 0;
}

int crs::row_sum(vector* res)
{
  double s = 0.0;
  for (size_t r = 0; r < nrows(); r++) {
    s = 0.0;
    for (size_t j = m_rowptrs[r]; j < m_rowptrs[r+1]; j++) {
      s += m_val[j];
    }
    res->data[r] = s;
  }
  return 0;
}

int crs::row_daxpy(size_t i, double a, vector* r)
{
  for (size_t c = m_rowptrs[i]; c < m_rowptrs[i+1]; c++) {
    r->data[ m_colindx[c] ] += a*m_val[c];
  }
  return 0;
}

/* Functions or operations involving entire matrix */
// Matrix vec mpy, result = Ax
int crs::matrix_vec (vector* x, vector* result)
{
  double yi;
  for (size_t i = 0; i < nrows(); i++) {
    yi = 0;
    for (size_t j = m_rowptrs[i]; j < m_rowptrs[i+1]; j++) {
      yi += m_val[j] * x->data[ m_colindx[j] ];
    }
    result->data[i] = yi;
  }
  return 0;

}

// Transpose[Matrix].vec = A'x
int crs::matrix_tran_vec (vector* x, vector* result)
{
  std::cerr << "CRS:matrix_tran_vec****** NOT IMPLEMENTED YET *******\n";
  return 0;
}

// Add two sparse matrices
int crs::add_sparse (sparsematrix* m)
{
  std::cerr << "****** NOT IMPLEMENTED YET *******\n";
  return -1;
}

// Subtract two sparse matrices
int crs::sub_sparse (sparsematrix* m)
{
  std::cerr << "****** NOT IMPLEMENTED YET *******\n";
  return -1;
}

// Add dense to sparse
int crs::add (matrix* m)
{
  std::cerr << "****** NOT IMPLEMENTED YET *******\n";
  return -1;
}

// Sub dense 
int crs::sub (matrix* m)
{
  std::cerr << "****** NOT IMPLEMENTED YET *******\n";
  return -1;
}

// Hadamard product 
int crs::dot (matrix* m)
{
  std::cerr << "****** NOT IMPLEMENTED YET *******\n";
  return -1;
}

int crs::mul (matrix* m)
{
  return -1;
}


int crs::matrix_dot (sparsematrix* m)
{
  std::cerr << "****** NOT IMPLEMENTED YET *******\n";
  return -1;
}

int crs::compute_AtA(matrix* result)
{
  return -1;
}

void crs::print()
{

  std::cerr << nrows() << " x " << ncols() << std::endl;
  for (size_t i = 0; i < nrows(); i++) {
    for (size_t j = 0; j < ncols(); j++)
      std::cout << get(i,j) << " ";
    std::cout << std::endl;
  }

  // std::cout << nrows() << " x " << ncols() << std::endl;
//   size_t p = 0;
//   for (size_t r = 0; r < nrows(); ++r){
//     size_t colnr = 0; 
//     if (m_rowptrs[r+1] == p) {
//       for ( ; colnr < ncols(); ++colnr){
//         std::cout << 0 << " ";
//       }
//     } else {
//       for (; p < m_rowptrs[r+1]; ++p){
//         while (colnr < m_colindx[p]){
//           std::cout << 0 << " ";
//           ++colnr;
//         }
//         std::cout << m_val[p] << " ";
//         ++colnr;
//       }
//       while (colnr < ncols()){
//         std::cout << 0 << " ";
//         ++colnr;
//       }
//     }
//     std::cout << std::endl;
//   }
}


/**
 * @author  stefje
 * Converts a matrix in CRS format to a matrix in CCS format.
 */
ccs* crs::to_ccs()
{
  ccs* newm = new SSLib::ccs();

  size_t* ncp, *nrind;
  double* nval;

  newm->matrix_alloc(nrows(), ncols(), nonzeros());

  size_t* colind = new size_t[ncols()+1];
  memset(colind, 0, (ncols()+1)*sizeof(size_t)/sizeof(char));

  ncp = newm->get_colptrs();
  nrind = newm->get_rowindx();
  nval = newm->get_data();

  memset(ncp, 0, (ncols()+1)*sizeof(size_t)/sizeof(char));

  //count the nonzeros in each column
  for (size_t i = 0; i < nonzeros(); ++i) {
    ncp[m_colindx[i]+1]++;
  }

  //add for column pointers
  for (size_t i=1; i< ncols()+1; ++i)  {
    ncp[i+1] += ncp[i];
  }

  //go through the rows and copy the values into the proper locations in newm
  //at the same time, insert the row indices
  for (size_t r=0; r<nrows(); ++r) { //row r
    for (size_t i = m_rowptrs[r]; i < m_rowptrs[r+1]; ++i){
      size_t ind = ncp[m_colindx[i]] + colind[m_colindx[i]];
      ++colind[m_colindx[i]];

      nval[ind] = m_val[i];
      nrind[ind] = r;
    }
  }
  delete colind;
  return newm;
}

void crs::free_data()
{
  if (m_val) free(m_val); m_val = 0;
  if (m_rowptrs) free(m_rowptrs); m_rowptrs = 0;
  if (m_colindx) free(m_colindx); m_colindx = 0;
}

void crs::error (std::string s)
{
  std::cerr << "CRS:"  << s << std::endl;
}
