// File: coord_crs_float.cc
// Author: Suvrit Sra
// Time-stamp: <23 February 2010 10:55:59 AM CET --  suvrit>
// Copyright (C) 2005 Suvrit Sra (suvrit@cs.utexas.edu)
// Copyright The University of Texas at Austin

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

// Implements the coordinate col ccs based class.

#include <iostream>
#include <cstdio>
#include <algorithm>
#include <cmath>
#include <cstring>
#include "coord_crs_float.h"

int SSLib::coord_crs_float::load(const char* fname)
{
  return load_txt(fname);
}


int SSLib::coord_crs_float::load(const char* fn, bool asbin)
{
  if (asbin)
    return load_bin(fn);
  else
    return load_txt(fn);
}

int SSLib::coord_crs_float::load_txt(const char* fn)
{
  // Try both fast and slow versions
  int r = load_matlab_slow(fn);
  return r;
}

int SSLib::coord_crs_float::load_bin(const char* fn)
{
  return -1;
}
/**
 * This function assumes that the data is stored in the file in the
 * following manner.
 * (i1,j1,v1)
 * (i1,j2,v2)
 * (i1,j3,v3)
 * ...
 * ...
 * (ik,jx,vnz)
 * Default: indices are expected to start at '0' and not '1' like
 * matlab. The parameter 'base' can be used to control that. Base = 1 gives
 * matlab indexing and base = 0 is the 'C' style indexing.
 * Load fast expects i1 < i2 < ... < ik and the columns also to be in
 * incr. order, within a given row
 */
int SSLib::coord_crs_float::load_matlab_fast (const char* fil)
{
  FILE* fp = fopen(fil, "r");
  if (!fp) {
    error("Could not load " + std::string(fil));
    return -1;
  }

  size_t m, n, nz;
  int mt = M_COORD_CRS;

  if (SSLib::read_file_header_txt(fp, &m, &n, &nz, &mt, mt) < 0) {
    fclose(fp);
    return -1;
  }

  matrix_alloc(m, n, nz);

  return finish_load_matlab(fp);
}

int SSLib::coord_crs_float::finish_load_matlab(FILE* fp) 
{  
  int base = get_base();
  size_t idx = 0;
  long prow = -1;
  m_rowptrs[0] = 0;		// Init it!
  long int col, row;
  float val;
  ssize_t st;
  while (!feof(fp)) {
    st=fscanf(fp, "%ld %ld %f", &row, &col, &val);
    if (col > (long)ncols() || row > (long)nrows()) {
      std::cerr << "Input matrix too big, stopping at ("
                << row << " , " << col << ") entry\n";
      break;
    }

    row -= base;
    col -= base;
    
    if (row > prow) {
      for (long r = prow + 1; r <= row; r++)
        m_rowptrs[r] = idx;
      prow = row;
    }
    m_colindx[idx] = col;
    m_val[idx]     = val;
    ++idx;
  }
  fclose(fp);
  return 0;
}

/**
 * Loads a file in (i, j, val) format, and stores it as a CRS matrix
 * internally. The (i, j, val) can be in any order.
 */
int SSLib::coord_crs_float::load_matlab_slow (const char* fil)
{

  std::string s = std::string(fil) + ".dim";
  FILE* fp = fopen(s.c_str(), "r");
  if (!fp) {
    error("Could not open dimensionality data file " + s);
    return -1;
  }

  size_t m, n, nz;
  ssize_t st;
  st=fscanf(fp, "%zu %zu %zu", &m, &n, &nz);
  matrix_alloc(m, n, nz);

  fclose(fp);
  fp = fopen(fil, "r");
  if (!fp) {
    error("Could not open data file " + std::string(fil));
    return -1;
  }
  return finish_load_matlab(fp);
}

int SSLib::coord_crs_float::load_matlab_gen(const char* fil)
{
  std::string s = std::string(fil) + ".dim";
  FILE* fp = fopen(s.c_str(), "r");
  if (!fp) {
    error("Could not open dimensionality data file " + s);
    return -1;
  }

  size_t m, n, nz;
  ssize_t st;
  st=fscanf(fp, "%zu %zu %zu", &m, &n, &nz);
  matrix_calloc(m, n, nz);

  fclose(fp);
  fp = fopen(fil, "r");
  if (!fp) {
    error("Could not open data file " + std::string(fil));
    return -1;
  }
  return load_matlab_as_crs(fp);
}

int SSLib::coord_crs_float::load_matlab_as_crs(FILE* fp) 
{
  // Load all the (i, j, v) data
  size_t idx = 0;
  size_t i, j;
  float val;
  ssize_t st;
  coord_struct_row* coord = new coord_struct_row[nonzeros()];

  while (!feof(fp)) {
    if (idx >= nonzeros()) {
      error ("Matrix longer than expected. Not reading further");
      break;
    }
    st=fscanf(fp, "%zu %zu %f", &i, &j, &val);
    coord[idx].m_rowindx = i;
    coord[idx].m_colindx = j;
    coord[idx].m_val     = val;
    ++idx;
  }
  fclose(fp);

  // Now sort the values, lexicographic so that
  // (i1, j1) < (i2, j2) if (j2 < j1) or (j2 == j1) but (i1 < i2)
  std::sort(coord, coord + nonzeros());

  // Now just update the CCS data ... don't worry about efficiency rite
  // now..only if it becomes important.
  // First make a pass thru the column indices to compute colptrs
  m_rowptrs[0] = 0;
  idx = 0;
  size_t r = 0;
  while (idx < nonzeros()) {
    while (coord[idx].m_rowindx == r) {
      ++m_rowptrs[r+1];         // matrix_calloc was called
      ++idx;
    }
    r++;                        // incr. column counter
  }
  for (size_t i = 0; i < nonzeros(); i++) {
    m_colindx[i] = coord[i].m_colindx;
    m_val[i] = coord[i].m_val;
  }
  delete[] coord;
  return 0;
}

void SSLib::coord_crs_float::error (std::string s)
{
  std::cerr << "COORD_CRS_FLOAT:"  << s << std::endl;
}

