// File: coord_ccs_compact_float.cc
// Author: Suvrit Sra
// Time-stamp: <20 February 2008 10:33:32 AM CET --  suvrit>
// Copyright (C) 2005 Suvrit Sra (suvrit@cs.utexas.edu)
// Copyright The University of Texas at Austin

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

// Implements the coordinate col ccs based class.

#include <iostream>
#include <cstdio>

#include "coord_ccs_compact_float.h"

int SSLib::coord_ccs_compact_float::load(const char* fname)
{
  return load_txt(fname);
}

int SSLib::coord_ccs_compact_float::load(const char* fn, bool asbin)
{
  if (asbin)
    return load_bin(fn);
  else
    return load_txt(fn);
}

int SSLib::coord_ccs_compact_float::load_txt(const char* fn)
{
  int r = load_matlab_fast(fn);
  if (r < 0)
    r = load_matlab_jiv(fn);
  return r;
}


int SSLib::coord_ccs_compact_float::load_bin(const char* fn)
{
  return -1;
}

/**
 * This expects the (i, j, val) pairs to be ordered by columns (incr. column
 * index, and within each column, increasing row index)
 */
int SSLib::coord_ccs_compact_float::load_matlab_fast(const char* fil)
{
  FILE* fp = fopen(fil, "r");
  if (!fp) {
    error("Could not load " + std::string(fil));
    return -1;
  }

  size_t m, n;
  size_t nz;
  int emt = M_COORD_CCS_COMPACT_FLOAT;

  if (SSLib::read_file_header_txt(fp, &m, &n, &nz, emt) < 0) {
    fclose(fp);
    return -1;
  }

  if (m > USHRT_MAX || n > USHRT_MAX) {
    error("coord_ccs_compact_float::load(): Matrix dimensions to large!");
    fclose(fp);
    return -1;
  }

  matrix_alloc(m, n, nz);

  return finish_load_matlab(fp);
}

int SSLib::coord_ccs_compact_float::finish_load_matlab(FILE* fp)
{
  int base = get_base();
  size_t idx = 0;
  int pcol = -1;
  int col, row;
  float val;

  // Init the colptrs array -- impt!
  bzero(m_colptrs, sizeof(size_t)*(ncols()+1));

  while (!feof(fp)) {
    fscanf(fp, "%d %d %f", &row, &col, &val);
    //std::cout << "(" << row << ", " << col << " , " << val << ") ";
    if (col > ncols() || row > nrows()) {
      std::cerr << "Input matrix too big, stopping at ("
                << row << " , " << col << ") entry\n";
      break;
    }
    
    row -= base;
    col -= base;
    
    if (col > pcol) {
      for (int c = pcol + 1; c <= col; c++)
        m_colptrs[c] = idx;
      pcol = col;
    }
    m_rowindx[idx] = row;
    m_val[idx]     = val;
    ++idx;
  }

  std::cout << "col, row, nz " << col << " " << row << " " << idx << "\n";
  // Fix the m_colptrs array for trailing totally empty columns...
  for (int c = col+1; c <= ncols(); c++)
    m_colptrs[c] = idx-1;

  fclose(fp);
  return 0;
}

/**
 * Loads a file in (i, j, val) format, and stores it as a CCS matrix
 * internally. Needs an extra .dim file for M, N, NZ info
 */
int SSLib::coord_ccs_compact_float::load_matlab_slow(const char* fil)
{
  std::string s = std::string(fil) + ".dim";
  FILE* fp = fopen(s.c_str(), "r");
  if (!fp) {
    error("Could not open dimensionality data file " + s);
    return -1;
  }

  size_t m, n;
  size_t nz;
  fscanf(fp, "%zu %zu %zu", &m, &n, &nz);
  if (m > USHRT_MAX || n > USHRT_MAX) {
    error("coord_ccs_compact_float::load(): Matrix dimensions to large!");
    fclose(fp);
    return -1;
  }

  matrix_alloc(m, n, nz);

  fclose(fp);
  fp = fopen(fil, "r");
  if (!fp) {
    error("Could not open data file " + std::string(fil));
    return -1;
  }
  return finish_load_matlab(fp);

}

/**
 * Loads a file in (j, i, val) format -- compare to finish_load_matlab()
 * above. File is loaded, and stored as a CCS matrix internally. Needs an extra
 * .dim file for M, N, NZ info.
 * The code expects the dimension indexing to begin with 1, not zero, because
 * of the name 'matlab'! 
 * NOTE: (j1, i1, v1), (j1, i2, v2), (j1, ik, vk), (j2, i1, v1).... is the
 * expected format.
 */
int SSLib::coord_ccs_compact_float::load_matlab_jiv(const char* fil)
{
  std::string s = std::string(fil) + ".dim";
  FILE* fp = fopen(s.c_str(), "r");
  if (!fp) {
    error("coord_ccs_compact_float::load_matlab_jiv(): Could not open dimensionality data file " + s);
    return -1;
  }

  size_t m, n;
  size_t nz;
  fscanf(fp, "%zu %zu %zu", &m, &n, &nz);
  if (m > USHRT_MAX +1 || n > USHRT_MAX+1) {
    error("coord_ccs_compact_float::load(): Matrix dimensions to large!");
    fclose(fp);
    return -1;
  }

  matrix_alloc(m, n, nz);

  fclose(fp);

  fp = fopen(fil, "r");
  if (!fp) {
    error("Could not open data file " + std::string(fil));
    return -1;
  }
  return finish_load_matlab_jiv(fp);
}

/**
 * indexing s'posed to begin at 1
 */
int SSLib::coord_ccs_compact_float::finish_load_matlab_jiv(FILE* fp)
{
  int base = get_base();
  size_t idx = 0;
  int pcol = -1;
  int col, row;
  float val;

  // Init the colptrs array -- impt!
  bzero(m_colptrs, sizeof(size_t)*(ncols()+1));

  // Ok, am going thru this code once again, it has been somewhat tricky to
  // get this simple code up and running correctly.
  while (idx < nonzeros()) {
    fscanf(fp, "%d %d %f", &col, &row, &val);
    if (col > ncols() || row > nrows()) {
      std::cerr << "DIM = " << nrows() << " x " << ncols() 
                << "\nInput matrix too big, stopping at ("
                << row << " , " << col << ") entry\n";
      return -1;
    }

    row -= base;
    col -= base;

    if (col > pcol) {
      for (int c = pcol + 1; c <= col; c++)
        m_colptrs[c] = idx;
      pcol = col;
    }
    m_rowindx[idx] = row;
    m_val[idx]     = val;
    ++idx;
  }

  std::cout << "\n LOADED: " << col << " " << row << " " << idx << "\n";

  // Fix the m_colptrs array for trailing totally empty columns...
  for (int c = col+1; c <= ncols(); c++)
    m_colptrs[c] = idx;

  fclose(fp);
  return 0;
}


/**
 * Loads a file in (i, j, val) format, and stores it as a CCS matrix
 * internally. The (i, j, val) can be in any order. The dimensionality
 * information is expected to be provided in an accompanying .dim file (I
 * have made this choice so that the same data file can be loaded in matlab
 * too, without skipping the first line of the file)
 */
int SSLib::coord_ccs_compact_float::load_matlab_gen(const char* fil)
{
  std::string s = std::string(fil) + ".dim";
  FILE* fp = fopen(s.c_str(), "r");
  if (!fp) {
    error("Could not open dimensionality data file " + s);
    return -1;
  }

  size_t m, n;
  size_t nz;
  fscanf(fp, "%zu %zu %zu", &m, &n, &nz);
  if (m > USHRT_MAX || n > USHRT_MAX) {
    error("coord_ccs_compact_float::load(): Matrix dimensions to large!");
    fclose(fp);
    return -1;
  }

  fclose(fp);
  fp = 0;

  matrix_calloc(m, n, nz);
  fp = fopen(fil, "r");
  if (!fp) {
    error("Could not open data file " + std::string(fil));
    return -1;
  }
  return load_matlab_as_ccs(fp);
}

/*
 * For speed, this function does not do any error checking as such....
 */
int SSLib::coord_ccs_compact_float::load_matlab_as_ccs(FILE* fp)
{
  int base = get_base();
  // Load all the (i, j, v) data
  size_t idx = 0;
  index_t i, j;
  float val;

  coord_struct_compact_col* coord = (coord_struct_compact_col*) malloc (nonzeros() * sizeof(coord_struct_compact_col));

  while (!feof(fp) && idx < nonzeros()) {
    fscanf(fp, "%hu %hu %f", &i, &j, &val);
    coord[idx].m_rowindx = i - base;
    coord[idx].m_colindx = j - base;
    coord[idx].m_val     = val;
    ++idx;
  }
  fclose(fp);

  // Now sort the values, lexicographic so that
  // (i1, j1) < (i2, j2) if (j2 < j1) or (j2 == j1) but (i1 < i2)
  std::sort(coord, coord + nonzeros());

  // Now just update the CCS data ... don't worry about efficiency rite
  // now..only if it becomes important.
  // First make a pass thru the column indices to compute colptrs
  m_colptrs[0] = 0;
  idx = 0;
  size_t c = 0;
  while (idx < nonzeros()) {
    m_colptrs[c+1] = m_colptrs[c];
    while (idx < nonzeros() && coord[idx].m_colindx == c) {
      ++m_colptrs[c+1];         // matrix_calloc was called
      ++idx;
    }
    c++;                        // incr. column counter
  }
  for (size_t i = 0; i < nonzeros(); i++) {
    m_rowindx[i] = coord[i].m_rowindx;
    m_val[i] = coord[i].m_val;
  }
  free(coord);
  return 0;
}

void SSLib::coord_ccs_compact_float::error (std::string s)
{
  std::cerr << "COORD_CCS_COMPACT_FLOAT:"  << s << std::endl;
}

