// File: ccs_float.h -*- c++ -*-
// Time-stamp: <07 October 2006 11:39:28 AM CDT --  suvrit>
// Impl. of ccs format sparse matrices

#ifndef CCS_FLOAT_H
#define CCS_FLOAT_H

#include "sparsematrix_float.h"

namespace SSLib {
  class crs_float;
  /**
  * @class ccs, contains an implementation of a CCS sparsematrix. It supports
  * many of the operations suggested by the abstract class sparsematrix.
  *
  */
  class ccs_float : public sparsematrix_float {
  protected:			// for efficiency reasons.
    /**
    * The actual non-zeroes themselves.
    */
    float* m_val;

    /**
    * Colptrs for CCS structure. These hold indices into the
    * <code>m_rowindx</code> and <code>val</code>. The i-th column starts at
    * m_colptrs[i] and ends at m_colptrs[i+1]-1
    */
    size_t* m_colptrs;
  

    /**
    * Row indices for CCs structure. This array contain the row indices of
    * the matrix elements. For example, m_rowindx[ m_colptrs[i] ] gives the
    * row index of the 1st non-zero element in the i-th column.
    */
    size_t* m_rowindx;

  protected:    
    /**
    * Is the data in externally allocated arrays. That is, if the m_val,
    * m_colptrs and m_rowindx arrays are externally allocated (useful for
    * creating a ccs sparsematrix from another matrix of similar structure etc.
    */
    bool m_isExternal;

    /**
    * Filename prefix. A CCS matrix is loaded from a collection of 4
    * files. For example, if m_filename="mat", then the code that loads this
    * matrix will search for the files 'mat_dim', 'mat_row_ccs',
    * 'mat_col_ccs' and 'mat_<m_type>_nz' depending on the value of <code>m_type</code>
    */
    std::string m_filename;

    /* matrix type, 'txx', 'tfn', etc. */
    std::string m_type;

  private:
    /* load files */
    virtual int load_bin(std::string prefix);
    virtual int load_txt  (std::string prefix);

    virtual int save_bin(const char*); 
    virtual int save_txt  (const char*);

  public:
    ccs_float () {
      m_isExternal = false;
      m_filename = "";
      m_type = "txx";
      m_val = 0;
      m_colptrs = 0;
      m_rowindx = 0;
    }

    virtual ~ccs_float () {
      //free_data();
    }

    
    void free_data();
    

    std::string get_type() const { return m_type; }
    void set_type (std::string s) { m_type = s;}

    /** @name MatrixCreation
    * Creating and saving ccs matrix functions. These functions allow the
    * user to load a CCS matrix from file, or from already available
    * arrays. They also allow the user to save the current matrix to file.
    *
    */

    /*@{*/


    // Allocate the matrix
    int matrix_alloc (size_t m, size_t n, size_t nz);
    int matrix_calloc (size_t m, size_t n, size_t nz);
    int matrix_resize (size_t m, size_t n, size_t nz);
    virtual int load(const char*);
    virtual int load(const char*, bool asbin);

    virtual int save_coord_bin(const char*);
    virtual int save (const char*, bool asbin);

    /// Write out matrix in a specified way
    virtual int    save(const char*, bool asbin, int typ);
    
    /// Write out matrix to disk in CCS, CRS, etc. formats
    virtual int save_as_crs (const char*, bool);
    virtual int save_as_coord (const char*, bool);
    virtual int save_as_colcoord (const char*, bool);

    virtual int save_as_rowcoord (const char*, bool);
    virtual int save_rowcoord_bin(const char*);
    virtual int save_rowcoord_txt(const char*);

    virtual int save_as_matlab (const char*, bool);
    virtual int save_matlab_bin(const char* fil);
    virtual int save_matlab_txt(const char*);

    virtual int save_as_hb (const char*, bool);
    virtual int save_hb_bin(const char*);
    virtual int save_hb_txt(const char*);

    virtual int save_as_dense (const char*, bool);
    virtual int save_dense_bin(const char*);
    virtual int save_dense_txt(const char*);

    void externally_alloc (size_t* col, size_t* row, float* val);
    // Return a deep copy
    ccs_float* clone();

    // Augment matrix by adding a row
    int matrix_add_row(vector_float* r);

    // Augment by adding a column
    int matrix_add_col(vector_float* c);

    // Augment by adding a sparsematrix
    int matrix_add_mat(sparsematrix_float* m);
    

    /*@}*/


    int     scale(float);
    int     add_const(float);

    /** @name Accessors
    * Accessing the matrix elements 
    */
    /*@{*/

    size_t* get_colptrs() { return m_colptrs;}
    size_t* get_rowindx() { return m_rowindx;}
    float* get_data() { return m_val;}

    /// Return row 'r' of matrix
    vector_float* get_row(size_t r);
    /// Return col 'c' of matrix
    vector_float* get_col(size_t c);
    /// return the (i, j) entry of the matrix
    float  get (size_t i, size_t j);
    float  get (size_t i, size_t j, size_t* posn);
    float  operator()(size_t i, size_t j);
    /// set the (i, j) entry of the matrix.
    int     set (size_t i, size_t j, float val);
    int      set (size_t posn, float val);
    /// Sets the specified row to the given vector
    virtual int set_row(size_t r, vector_float*);

    /// Sets the existing non-zeros, ignores other entries
    virtual int fast_set_row(size_t r, vector_float*);


    /// Sets the specified col to the given vector
    virtual int set_col(size_t c, vector_float*);

    /// Sets only the existing non-zeros, ignores others
    virtual int fast_set_col(size_t c, vector_float*);

    /// Returns main or second diagonal (if p == true)
    virtual vector_float* get_diag(bool p  );
    /// Sets the specified diagonal to the given vector
    virtual int set_diag(bool p, vector_float*);

    /// Returns a submatrix that goes from row i1--i2 and cols j1--j2
    virtual matrix_float* submatrix(size_t i1, size_t j1, 
                                    size_t i2, size_t j2);
    
    /*@}*/

    /** @name Elementwise
    * Elementwise functions/operations on this matrix.
    */
    /*@{*/

    // Vector l_p norms for this matrix
    double norm (double p);
    double norm (char*  p);
    // Apply an arbitrary function elementwise to this matrix
    float apply (float (* fn)(float));

    /*@}*/

    /*@{*/

    /// Compute the vector l_p norm of column c
    virtual double col_norm (size_t c, double p);
    /// Compute dot product of column c1 with c2
    virtual double col_dot  (size_t c1, size_t c2);
    /// Compute dot product of column c with vector v
    virtual double col_dot  (size_t c, vector_float* v);
    /// Adds columns c2 to column c1. c1 := c1 + c2
    virtual int    col_add  (size_t c1, size_t c2);
    /// Subtracts column c2 from c1. c1 := c1 - c2
    virtual int    col_sub  (size_t c1, size_t c2);
    /// Scales column c1. c1 := s * c1
    virtual int    col_scale(size_t c1, float s);
    /// Subtracts vector v from column c1
    virtual int    col_sub  (size_t c1, vector_float* v);
    /// Adds vector v to column c1
    virtual int    col_add  (size_t c1, vector_float* v);
    /// Scales column c1 by 's' and stores result in 'r'
    virtual int    col_scale(size_t c1, float s, vector_float* r);

    ///
    virtual int    col_sum(vector_float* r);
    ///
    virtual int    col_avg(vector_float* r);
    /// 
    virtual int    col_num_nonzeros(vector_float* r);
    /*@}*/

    
    /** @name
    * Operations on rows of this matrix
    */
    /*@{*/
    /// Compute the vector l_p norm of row r
    virtual double row_norm (size_t r, double p);
    /// Compute dot product of row r1 with r2
    virtual double row_dot  (size_t r1, size_t r2);
    /// Compute dot product of row r with vector v
    virtual double row_dot  (size_t r, vector_float* v);
    /// Adds row r2 to row r1. r1 := r1 + r2
    virtual int    row_add  (size_t r1, size_t r2);
    /// Subtracts row r2 from r1.     r1 := r1 - r2
    virtual int    row_sub  (size_t r1, size_t r2);
    /// Scales column r1.       r1 := s * r1
    virtual int    row_scale(size_t r1, float s);
    /// Adds vector v to row r1
    virtual int    row_add  (size_t r1, vector_float* r);
    /// Subtracts vector v from row r1
    virtual int    row_sub  (size_t r1, vector_float* r);
    /// Scales row r = s*row(r1)
    virtual int    row_scale(size_t r1, float s, vector_float* r);

    /// r = a*row(i) + r
    virtual int    row_saxpy(size_t i, float a, vector_float* r);

    virtual int    leading_svd(vector_float* u, vector_float* v, int m);

    int dot (bool tranA, vector_float* x, vector_float* r); 
    matrix_float* transpose_copy();
    /*@}*/

    /*@{*/


    // Multiplication of two sparse matrices this = this * m or m * this
    int mult_sparse (sparsematrix_float* m, int leftOrRight);

    /**
    *
    */
    int mult_dense (matrix_float* m, int leftOrRight);

    /**
    * Subtract a sparse matrix from this matrix. The operation implemented is 
    * this := this + m.
    * @param m is a sparsematrix* arg
    * @return 
    *   -     0 if subtraction succeeded
    *   -     1 if size mismatch
    *   -     2 if some other error
    * @warning This function can cause fill-in by destroying non-zeros, and
    * can be slow.
    */
    int add_sparse (sparsematrix_float* m);


    /**
    * Subtract a sparse matrix from this matrix. The operation implemented is 
    * this := this - m.
    * @param m is a sparsematrix* arg
    * @return 
    *   -     0 if subtraction succeeded
    *   -     1 if size mismatch
    *   -     2 if some other error
    * @warning This function can cause fill-in by destroying non-zeros, and
    * can be slow.
    */  
    int sub_sparse (sparsematrix_float* m);

    /**
    * Add a dense matrix from this matrix. The operation implemented is 
    * this := this + m
    * @param m is a gsl_matrix* arg
    * @return
    *   -     0 if subtraction succeeded
    *   -     1 if size mismatch
    *   -     2 if some other error
    * @warning This function can cause fill-in by destroying non-zeros, and
    * can be slow.
    */
    int add (matrix_float* m);

    /**
    * Subtract a dense matrix from this matrix. The operation implemented is 
    * this := this - m
    * @param m is a gsl_matrix* arg
    * @return
    *   -     0 if subtraction succeeded
    *   -     1 if size mismatch
    *   -     2 if some other error
    * @warning This function can cause fill-in by destroying non-zeros, and
    * can be slow.
    */ 
    int sub (matrix_float* m);

    /**
    * Hadamard product. After the execution of this function our current
    * matrix is: this := this .* m (in matlab notation)
    * @param m is a gsl_matrix* argument
    * @return 
    *  -  0 if the dot product was carried out
    *  -  1 if matrix size mismatch
    *  -  2 if any other error
    */
    int dot (matrix_float* m);

    /**
    * Hadamard product. After the execution of this function our current
    * matrix is: this := this .* m (in matlab notation). This function does
    * a runtime type check to determine the type of 'm' to optimize the dot
    * product operation (since two ccs matrices or ccs * harwell boeing can
    * be dotted much faster without using any get(i, j) operations).
    * @param m is a sparsematrix* argument
    * @return 
    *  -  0 if the dot product was carried out
    *  -  1 if matrix size mismatch
    *  -  2 if any other error
    */
    int div (matrix_float* m);

    /*@}*/

    /** @name Misc 
    * Miscellanous functions
    */
    /*@{*/

    /**
    * @return true if this matrix is external, i.e., the rows, cols
    * etc. arrays are directly initialized from outside, and not allocated
    * by us.
    */
    bool isExternal() const { return m_isExternal;}
    void print ();
    void transpose();

    virtual void error(std::string);
    /*@}*/


    /** 
     * conversions
     */
    crs_float* to_crs();
    
    

  private:
    // Matrix vec mpy, result = Ax
    int matrix_vec (vector_float* x, vector_float* result);

    // Transpose[Matrix].vec = A'x
    int matrix_tran_vec (vector_float* x, vector_float* result);
  };

}
#endif
