// File: ccs_compact_float.cc
// Time-stamp: <20 February 2008 10:20:54 AM CET --  suvrit>
// Implements the compressed column storage class
// The compact version thereof --- see the header file for more comments


#include "exceptions.h"
#include <fstream>
#include <cstdio>
#include <cstdlib>
#include <cmath>
#include <netinet/in.h>

//#include "crs_compact.h"
//#include "coord_struct_compact.h"

#include "util.h"

#include <algorithm>

#include "ccs_compact_float.h"

using namespace SSLib;

/**
* @param m num of rows in matrix
* @param n num of cols in matrix
* @param nz num of nonzeros
* @throws sparsematrix::exception_malloc if an invalid matrix size is
* requested or the new operator fails to allocate storage.
* @return  0 if successful in allocating matrix.
*
*/
int ccs_compact_float::matrix_alloc (size_t m, size_t n, size_t nz)
{
  if (m == 0 || n == 0) {
    sparsematrix_exception* e = new exception_malloc("CCS_COMPACT_FLOAT: Invalid matrix size requested...", m, n);
    throw e;
  }
  
  setsize(m, n, nz);
  // Allocate the arrays
  m_colptrs = (size_t*) malloc (sizeof (size_t) * (n+1));
  m_rowindx = (index_t*) malloc (sizeof (index_t) * nz);
  m_val     = (float*) malloc (sizeof (float) * nz);
  
  if (m_colptrs == 0 || m_rowindx == 0 || m_val == 0) {
    sparsematrix_exception* e = new exception_malloc("CCS_COMPACT_FLOAT: Memory allocation error");
    throw e;
  }

  m_isExternal = false;
  return 0;
}
/**
* @param m num of rows in matrix
* @param n num of cols in matrix
* @param nz num of nonzeros
* @throws sparsematrix::exception_malloc if an invalid matrix size is
* requested or the new operator fails to allocate storage.
* @return  0 if successful in allocating matrix.
*
*/
int ccs_compact_float::matrix_calloc (size_t m, size_t n, size_t nz)
{
  if (m == 0 || n == 0) {
    sparsematrix_exception* e = new exception_malloc("Invalid matrix size requested...", m, n);
    throw e;
  }
  
  setsize(m, n, nz);
  size_t np = n+1;
  // Allocate the arrays
  m_colptrs = (size_t*) calloc (np, sizeof (size_t));
  m_rowindx = (index_t*) calloc (nz,  sizeof (index_t));
  m_val     = (float*) malloc (nz *sizeof (float));
  
  if (m_colptrs == 0 || m_rowindx == 0 || m_val == 0) {
    sparsematrix_exception* e = new exception_malloc("CCS_COMPACT: Memory allocation error");
    throw e;
  }

  m_isExternal = false;
  return 0;
}

int ccs_compact_float::matrix_resize (size_t m, size_t n, size_t nz)
{
  if (m == 0 || n == 0) {
    sparsematrix_exception* e = new exception_malloc("Invalid matrix size requested...", m, n);
    throw e;
  }

  size_t np = n+1;
  // Try to resize the arrays
  m_colptrs = (size_t*) realloc (m_colptrs, sizeof(size_t) * np);
  if (m_colptrs == NULL) {
    std::cerr << "COLPTRS: Could not resize matrix\n";
    return -1;
  }

  m_rowindx = (index_t*) realloc (m_rowindx, sizeof(index_t) * nz);
  if (m_rowindx == NULL) {
    std::cerr << "ROWINDX: Could not resize matrix\n";
    return -1;
  }

  m_val = (float*) realloc (m_val, sizeof(float) * nz);
  if (m_val == NULL) {
    std::cerr << "VAL: Could not resize matrix\n";
    return -1;
  }
  
  setsize(m, n, nz);
  return 0;
}

/**
* @param prefix -- the file name prefix that is attached to the 4 ccs format
* files.
* @param asbin -- a boolean, which if true indicates that we should load the
* ccs matrix data from the file <em>prefix</em> in binary mode. In this case
* one does not need the 4 diff CCS files, and all are assumed to be stored
* in the CCS binary file format @see save
* @see m_filename
* @return
*   -     0 if file loaded successfully
*   -     1 if file loading error
*/
int ccs_compact_float::load (const char* pr, bool asbin)
{
  std::string s = std::string(pr);
  m_filename = s;
  if (asbin) {
    return load_bin(s);
  } else {
    return load_txt(s);
  }
}

/**
 * @param file, the name of the file from which to load the CCS matrix in
 * binary format. NOTE that in the binary format, a CCS matrix is stored in
 * just one file.
 * @return
 *   -   0 if load was successful
 *   -   1 if file loading error
 */
int ccs_compact_float::load_bin(std::string file)
{
  m_filename = file;
  FILE* fp = fopen (file.data(), "rb");
  if (fp == 0) {
    std::string s = "Could not open file " + file;
    error(s);
    return -1;
  }

  size_t m;
  size_t n;
  size_t np;
  size_t nz;
  int    emt = M_CCS_COMPACT_FLOAT;

  if (SSLib::read_file_header_bin(fp, &m, &n, &nz, emt) < 0) {
    fclose(fp);
    return -2;
  }
  std::cerr << "Trying to load: matrix (" << m << " , " << n << " ,  " << nz << ")\n";
  /*std::string s = file + ".dim";
  FILE* fp2 = fopen(s.data(), "r");
  fscanf(fp2, "%ld %ld %ld", &m, &n, &nz);
  fclose(fp2);

  
  
  */
  // Allocate the arrays
  np = n + 1; // with n = USHRT_MAX, n+1 overflows, so....
  setsize(m, n, nz);
  m_colptrs = (size_t*) malloc (sizeof (size_t) * np);
  m_rowindx = (index_t*) malloc (sizeof (index_t) * nz);
  m_val     = (float*) malloc (sizeof (float) * nz);
  
  // REad the colptrs
  if (fread(m_colptrs, sizeof(size_t), np, fp) != np) {
    error ("CCS_COMPACT_FLOAT: Error reading colptrs");
    return -3;
  }

  // REad the row indices
  // if nz happens to be very big, we break up the reads
  if (fread(m_rowindx, sizeof(index_t), nz, fp) != nz) {
    error ("CCS_COMPACT_FLOAT: Error reading rowindices");
    return -3;
  }
  // REad the nonzeros themselves
  if (fread(m_val, sizeof(float), nz, fp) != nz) {
    error ("CCS_COMPACT_FLOAT: Error reading nonzeros");
    return -3;
  }
  fclose(fp);
  return 0;
}

/**
 * @param prefix ccs filename prefix
 * This function loads the file as a text file. This function is called by
 * @see load
 */
int ccs_compact_float::load_txt(std::string prefix)
{
  // This function assumes that m_type is already set
  m_filename = prefix;
  // Set up the file names that we are gonna open
  std::string dim(prefix);
  std::string rows_file(dim + "_row_ccs");
  std::string cols_file(dim + "_col_ccs");
  std::string nz_file (dim + "_"+ m_type + "_nz");
  
  dim +=  "_dim";
  
  std::ifstream infile;
  infile.clear();
  infile.open(dim.data());
  if (infile.fail()) {
    std::string s = "CCS_COMPACT_FLOAT: Error: could not open ***" + dim + "***";
    error(s);
    return 1;
  }
  
  size_t m_rows, m_cols;
  size_t m_nz;

  infile >> m_rows >> m_cols >> m_nz;
  infile.close();
  infile.clear();

  // Allocate the arrays
  size_t np = m_cols + 1;
  m_colptrs = (size_t*) malloc (sizeof (size_t) * np);
  m_rowindx = (index_t*) malloc (sizeof (index_t) * m_nz);
  m_val     = (float*) malloc (sizeof (float) * m_nz);
  m_colptrs[m_cols] = m_nz; // Add the convenience value
    
  setsize (m_rows, m_cols, m_nz);

  // Now read in all the columnpointers
  infile.open(cols_file.data());
  if (infile.fail()) {
    std::string s = "CCS_COMPACT_FLOAT: Error: could not open " + cols_file;
    error(s);
    return 2;
  }
  // Make sure the UB on i is correct.
  for (size_t i = 0; i < m_cols; i++) 
    infile >> m_colptrs[i];
  infile.close();
  infile.clear();
  
  infile.open(rows_file.data());
  if (infile.fail()) {
    std::string s = "CCS_COMPACT_FLOAT: Error: could not open " + rows_file;
    error(s);
    return 3;
  }

  for (size_t i = 0; i < m_nz; i++)
    infile >> m_rowindx[i];
  infile.close();
  infile.clear();
  
  // Now read in the actual m_values
  infile.open(nz_file.data());
  if (infile.fail()) {
    std::string s = "CCS_COMPACT_FLOAT: Error: could not open " + nz_file;
    error(s);
    return 4;
  }
  
  for (size_t i = 0; i < m_nz; i++)
    infile >> m_val[i];
  infile.close();

  return 0;
}


/// Write out matrix to disk in CCS, CRS, etc. formats
int ccs_compact_float::save (const char* s, bool asbin, int typ)
{
  switch (typ) {
  case M_SELF:
  case M_CCS_COMPACT_FLOAT:
    return save(s, asbin); 
  case M_DENSE_COMPACT_FLOAT:
    return save_as_dense(s, asbin);
  case M_CRS_COMPACT_FLOAT:
    return save_as_crs(s, asbin);
  case M_HB_COMPACT_FLOAT:
    return save_as_hb(s, asbin);
  case M_COORD_CCS_COMPACT_FLOAT:
    return save_as_coord(s, asbin);
  case M_MATLAB_COMPACT_FLOAT:
    return save_as_matlab(s, asbin);
  case M_ROWCOORD_COMPACT_FLOAT:
    return save_as_rowcoord(s, asbin);
  case M_COLCOORD_COMPACT_FLOAT:
    return save_as_colcoord(s, asbin);
  default:
    matrix_error("CCS_COMPACT_FLOAT: Invalid Matrix Type Requested In save(...)");
    return -2;
  }
}


/**
 * @param prefix -- gives the string names that is used as a prefix while
 * saving out the matrix to disk. the 4 ccs files are created..
 * @param asbin saves the CCs matrix in the CCS matrix binary file format
 * that is defined as follows: 
 * M,N,NZ (size_t, size_t, size_t)
 * M_COLPTRS (array of size N+1 of size_ts)
 * M_ROWINDX (array of size NZ of size_ts)
 * M_VALS    (array of size NZ of floats)
 * @return
 *   -  0 if saved successfully
 *   -  1,2,3,4 error in saving one of the CCS files
 */
int ccs_compact_float::save (const char* prefix, bool asbin) 
{
  std::string s = std::string(prefix);
  if (asbin) {
    s = s + "_" + m_type;
    return save_bin(s.data());
  } else 
    return save_txt(s.data());
}
/**
 * @param file, the name of the file to which the CCS matrix should be saved
 * in binary format. This matrix can be later read by using load with the
 * asbin argument set to true.
 * @return
 *  -   0 if file saved successfully
 *  -   1 if file saving error
 */
int ccs_compact_float::save_bin(const char* file)
{
  FILE* fp = fopen (file, "wb");
  if (fp == 0) {

    return 1;
  }
  size_t m = nrows();
  size_t n = ncols();
  size_t nz = nonzeros();
  int mt = M_CCS_COMPACT_FLOAT;

  //std::cerr << "Trying to save: " << m << " " << n << " " << nz << "\n";
  if (SSLib::write_file_header_bin(fp, &m, &n, &nz, &mt) < 0) {
    fclose(fp);
    return -2;
  }
  size_t n1 = n + 1;
  // Write the colptrs
  if (fwrite(m_colptrs, sizeof(size_t), n1, fp) != n1) {
    error ("CCS_COMPACT_FLOAT: Error saving colptrs");
    return 3;
  }

  // Write the row indices
  if (fwrite(m_rowindx, sizeof(index_t), nz, fp) != nz) {
    error ("CCS_COMPACT_FLOAT: Error saving rowindices");
    return 3;
  }
  // Write the nonzeros themselves
  if (fwrite(m_val, sizeof(float), nz, fp) != nz) {
    error ("CCS_COMPACT_FLOAT: Error saving nonzeros");
    return 3;
  }
  fclose(fp);
  return 0;
}
/**
 * @param prefix, the CCS file name prefix to which the CCS matrix is
 * saved. Whether it is 'txx', or 'tfn' or something else is determined by
 * the value of the @see m_type field.
 * @return
 *  -  0  if file saved successfully
 *  -  1  if file saving error
 */
int ccs_compact_float::save_txt(const char* prefix)
{
  // Set up the file names that we are gonna open
  std::string dim(prefix);
  std::string rows_file(dim + "_row_ccs");
  std::string cols_file(dim + "_col_ccs");
  std::string nz_file (dim + "_"+ m_type + "_nz");
  
  dim +=  "_dim";
  
  std::ofstream ofile;
  ofile.open(dim.data());
  if (ofile.fail()) {
    std::string s = "CCS_COMPACT_FLOAT: Error: could not open for writing " + dim;
    error(s);
    return 1;
  }
  
  ofile << nrows()
        << " " << ncols()
        << " " << nonzeros();

  ofile.close();
  ofile.clear();
  
  // Now write out all the columnpointers
  ofile.open(cols_file.data());
  if (ofile.fail()) {
    std::string s = "CCS_COMPACT_FLOAT: Error: could not open for writing: " + cols_file;
    error(s);
    return 2;
  }
  // Make sure the UB on i is correct.
  for (size_t i = 0; i < ncols() + 1; i++) 
    ofile << m_colptrs[i] << std::endl;
  ofile.close();
  ofile.clear();
  
  ofile.open(rows_file.data());
  if (ofile.fail()) {
    std::string s = "CCS_COMPACT_FLOAT: Error: could not open for writing: " + rows_file;
    error(s);
    return 3;
  }

  for (size_t i = 0; i < nonzeros(); i++)
    ofile << m_rowindx[i] << std::endl;
  ofile.close();
  ofile.clear();
  
  // Now write out the actual m_values
  ofile.open(nz_file.data());
  if (ofile.fail()) {
    std::string s = "CCS_COMPACT_FLOAT: Error: could not open " + nz_file;
    error(s);
    return 4;
  }
  
  for (size_t i = 0; i < nonzeros(); i++)
    ofile << m_val[i] << std::endl;
  ofile.close();

  return 0;
}
// -----------------------------------------------------------------

/**
 * save as crs format 
 * -----------------------------------------------------------------
 */


int ccs_compact_float::save_as_crs(const char* fil, bool asbin)
{

#if 0
  crs_compact_float* tmp = to_crs();
  if (!tmp) 
    return -1;
  tmp->save(fil, asbin);
  tmp->free_data();
  delete tmp;
#endif 
  return -1;
}

int ccs_compact_float::save_as_coord(const char* fil, bool asbin)
{
  if (asbin)
    return save_coord_bin(fil);
  return save_matlab_txt(fil);

}

int ccs_compact_float::save_coord_bin(const char* fil)
{
  return -1;
}
// -----------------------------------------------------------------

/**
 * save as column-coordinate matrix format
 * -----------------------------------------------------------------
 */
int ccs_compact_float::save_as_colcoord(const char* fil, bool asbin)
{
#if 0
  // Save a CCS file as a coord_bin file...convert to CRS first.
  crs_compact_float* tmp = to_crs();
  if (!tmp) 
    return -1;
  int rv = tmp->save_as_colcoord(fil, asbin);
  tmp->free_data();
  delete tmp;
#endif 
  return -1;
}
// -----------------------------------------------------------------


/**
 * save as row-coordinate matrix format
 * -----------------------------------------------------------------
 */
int ccs_compact_float::save_as_rowcoord(const char* fil, bool asbin)
{
  if (asbin)
    return save_rowcoord_bin(fil);
  else
    return save_rowcoord_txt(fil);
}

int ccs_compact_float::save_rowcoord_bin(const char* fil)
{
  return -1;
}
int ccs_compact_float::save_rowcoord_txt(const char* fil)
{
  return -1;
}

// -----------------------------------------------------------------


/**
 * save ccs matrix as uncompressed matlab bin, and ordinary txt files
 * -----------------------------------------------------------------
 */
int ccs_compact_float::save_as_matlab(const char* fil, bool asbin)
{
  if (asbin)
    return save_matlab_bin(fil);
  else
    return save_matlab_txt(fil);

}
int ccs_compact_float::save_matlab_bin(const char* fil)
{
  error("CCS_COMPACT_FLOAT: NOT YET IMPLEMENTED");
  return -2;

  FILE* fp = fopen (fil, "wb");
  if (!fp) {
    return -1;
  }
  
  fclose(fp);
}
 
int ccs_compact_float::save_matlab_txt(const char* fil)
{
  FILE* fp = fopen(fil, "w");
  if (!fp) {
    std::string s = "CCS_COMPACT_FLOAT: Error: could not open " + std::string(fil) + " for writing";
    error(s);
    return -1;
  }

  for (size_t c = 0; c < ncols(); c++) {
    for (size_t r = m_colptrs[c]; r < m_colptrs[c+1]; r++) {
      fprintf(fp, "%zu %zu %f\n", m_rowindx[r]+1, c+1, m_val[r]);
    }
  }
  fclose(fp);
  return 0;
}
// -----------------------------------------------------------------


/**
 * Save ccs matrix as a HB format matrix
 * -----------------------------------------------------------------
 */
int ccs_compact_float::save_as_hb(const char* s, bool asbin)
{
  if (asbin)
    return save_hb_bin(s);
  else
    return save_hb_txt(s);
}
int ccs_compact_float::save_hb_bin(const char* s)
{
  return -1;
}
int ccs_compact_float::save_hb_txt(const char* s)
{
  return -1;
}
// -----------------------------------------------------------------

/**
 * Save ccs matrix as a dense matrix
 * -----------------------------------------------------------------
 */
int ccs_compact_float::save_as_dense(const char* fil, bool asbin)
{
  if (asbin)
    return save_dense_bin(fil);
  else
    return save_dense_txt(fil);
}
int ccs_compact_float::save_dense_bin(const char* s)
{
  FILE* fp = fopen(s, "wb");
  if (!fp) {
    std::cerr << "CCS_COMPACT_FLOAT: could not open " << s << " for writing (bin)\n";
    return -1;
  }
 
  size_t m = nrows();
  size_t n = ncols();
  size_t nz = nonzeros();
  int mt = M_DENSE_COMPACT_FLOAT;

  if (SSLib::write_file_header_bin(fp, &m, &n, &nz, &mt) < 0)
    return -2;

  for (size_t i = 0; i < nrows(); i++)
    for (size_t j = 0; j < ncols(); j++) {
      float t = get(i, j);
      fwrite(&t, sizeof(float), 1, fp);
    }

  fclose(fp);
  return 0;
}

int ccs_compact_float::save_dense_txt(const char* s)
{
  FILE* fp = fopen(s, "w");
  if (!fp) {
    std::cerr << "CCS_COMPACT_FLOAT: could not open " << s << " for writing (txt)\n";
    return -1;
  }

  size_t m = nrows();
  size_t n = ncols();
  size_t nz = nonzeros();
  int mt = M_DENSE;

  if (SSLib::write_file_header_txt(fp, m, n, nz, mt) < 0)
    return -2;

  for (size_t i = 0; i < nrows(); i++)
    for (size_t j = 0; j < ncols(); j++) {
      float t = get(i, j);
      fprintf(fp, "%f\n", t);
    }
  fclose(fp);
  return 0;
}

// -----------------------------------------------------------------


/**
 * @param col   A valid colptrs array
 * @param row   A valid rowindx array
 * @param val   Array of NZ values
 * Use this function if the CCS data is loaded elsewhere, and you don't want
 * to use a 'load' command to allocate the data. Note, however, that you
 * will still need to call the @see setsize function before/after calling
 * this function.
 * @return  None
 */
void ccs_compact_float::externally_alloc (size_t* col, index_t* row, float* val)
{
  m_colptrs = col;
  m_rowindx = row;
  m_val     = val;
  m_isExternal = true;
}

/**
 * This function creates a deep copy of the CCS matrix, appropriately
 * allocating the required memory. Then it copies over the nonzeros too.
 */
ccs_compact_float* ccs_compact_float::clone() 
{
  ccs_compact_float* copy = new ccs_compact_float();
  copy->m_isExternal = m_isExternal;
  copy->m_filename   = m_filename;
  copy->m_type       = m_type;
  copy->matrix_alloc (nrows(), ncols(), nonzeros());

  // Do the memcpy of the arrays now ...
  size_t np = ncols() + 1;
  memcpy(copy->m_colptrs, m_colptrs, sizeof (size_t) * np);
  memcpy(copy->m_rowindx, m_rowindx, sizeof (size_t) * (nonzeros()));
  memcpy(copy->m_val    , m_val    , sizeof (float)* (nonzeros()));

  return copy;
}

/**
 * @todo Add a new row to the matrix.
 */
int ccs_compact_float::matrix_add_row(vector_float* r)
{
  return -1;
}
int ccs_compact_float::matrix_remove_row(size_t r)
{
  return -1;
}

/**
 * Augments the current matrix by taggin on the new column c. After
 * completion, we now have one more column. This is an expensive function to
 * call, because the ccs structure is like that. matlab mode sparse matrices
 * are more amenable to such operations.
 * @return
 *  -       0 if augmentation was successful
 *  -       1 if c was of wrong dimension and augmentation failed
 *  -       2 realloc error
 */
int ccs_compact_float::matrix_add_col(vector_float* c)

{
  if (c->size != nrows()) 
    return 1;

  // We try to do a realloc here
  size_t np = ncols() + 2; // Actually....a bad idea??
  m_colptrs = (size_t*) realloc (m_colptrs, sizeof(size_t) * np);

  if (m_colptrs == NULL) {
    return 2;
  }
  
  // We need to also increase the size of the rowindx and nz arrays
  // depending on the structure of the vector c
  size_t nz = SSUtil::num_nonzeros (c->data, c->size);

  m_rowindx = (index_t*) realloc (m_rowindx, sizeof(index_t) * (nonzeros() + nz));
  
  if (m_rowindx == NULL)
    return 2;

  m_val     = (float*) realloc (m_val, sizeof(float) * (nonzeros() + nz));
  
  if (m_val == NULL)
    return 2;

  // If we reached here, all operations succeeded, we just add on the last
  // column easily now
  m_colptrs[nrows()+1] = nonzeros() + nz;
  size_t k = 0;
  for (size_t i = 0; i < c->size; i++)
    if (c->data[i] != 0) {
      m_rowindx[nz + k] = i;
      m_val    [nz + k] = c->data[i];
      ++k;
    }
  return 0;
}
int ccs_compact_float::matrix_remove_col(size_t c)
{
  return -1;
}


/*
 * @param r  The row requested
 * @return row 'r' of ccs matrix. Usu. avoid this operation as it is
 * expensive. When using CCS, try to write algorithms that go over columns
 * of the matrix rather than rows. Provided here for completion.
 */
int ccs_compact_float::get_row(size_t r, vector_float*& v)
{
  if (!v || v->size < ncols())
    return -1;
  int result; index_t idx;
  if (r >= USHRT_MAX)
    return -2;
  index_t rr = r;
  for (size_t c = 0; c < ncols(); c++) {
    size_t sz = m_colptrs[c+1] - m_colptrs[c];
    result = SSUtil::binary_search_compact(m_rowindx + m_colptrs[c], rr, sz, idx);
    if (result != -1)
      gsl_vector_float_set(v, c, (m_val + m_colptrs[c])[idx]);
  }
  return 0;
}
 
/**
 * @param c the requested column of the matrix
 * @param v a pre-calloced vector
 * @return column of matrix as a vector. No error/bounds checking is done. 
 */
int ccs_compact_float::get_col(size_t c, vector_float*& v) 
{
  if (!v || v->size < nrows())
    return -1;
  for (size_t i = m_colptrs[c]; i < m_colptrs[c+1]; i++)
    v->data[ m_rowindx[i] ] = m_val[i];
  return 0;
}

/**
 * @todo Sets the specified row to the given vector
 */
int ccs_compact_float::set_row (size_t r, vector_float*& v)
{
  return -1;
}

/**
 * Sets the existing non-zeros of matrix, by picking new non-zeros from the
 * vector. Does not add any new non-zero entries, i.e., the sparsity
 * structure of the matrix is preserved.
 * No bounds/error checking is done.
 * @return
 *  - 0 Success
 *  - -1 Failure (if length of input vector is .ne. ncols() of matrix
 */
int ccs_compact_float::fast_set_row (size_t r, vector_float*& v)
{
  if (v->size != ncols())
    return -1;
  if (r >= USHRT_MAX)
    return -2;
  index_t rr = r;
  int result; index_t idx;
  for (size_t c = 0; c < ncols(); c++) {
    size_t sz = m_colptrs[c+1] - m_colptrs[c];
    result = SSUtil::binary_search_compact(m_rowindx + m_colptrs[c], rr, sz, idx);
    if (result != -1)
      m_val[idx + sz] = v->data[c];
  }
  return 0;
}

/**
 * @todo Sets the specified col to the given vector
 */
int ccs_compact_float::set_col(size_t c, vector_float*&)
{
  return -1;
}

/**
 * Sets the existing non-zeros of matrix, by picking new non-zeros from the
 * vector. Does not add any new non-zero entries, i.e., the sparsity
 * structure of the matrix is preserved.
 * No bounds/error checking is done.
 * @return
 *  - 0 Success
 *  - -1 Failure (if length of input vector is .ne. nrows() of matrix
 */
int ccs_compact_float::fast_set_col (size_t c, vector_float*& v)
{
  if (v->size != nrows())
    return -1;
  for (size_t i = m_colptrs[c]; i < m_colptrs[c+1]; i++)
    m_val[i] = v->data[ m_rowindx[i] ];
  return 0;
}


/**
 * @todo Returns main or second diagonal (if p == true)
 */
int ccs_compact_float::get_diag(bool p, vector_float*& v )
{
  if (p) {
    for (size_t i = 0; i < std::min(nrows(), ncols()); i++)
      v->data[i] = get(i, ncols()-i-1);
  } else {
    for (size_t i = 0; i < std::min(nrows(), ncols()); i++) {
      v->data[i] = get(i, i);
    }
  }
  return 0;
}

/**
 * @todo Returns a submatrix that goes from row i1--i2 and cols j1--j2
 */
int ccs_compact_float::submatrix(size_t i1, size_t j1, size_t i2, size_t j2, matrix_compact_float*& m)
{
  for (size_t i = i1; i < i2; i++)
    for (size_t j = j1; j < j2; j++)
      m->set(i, j, get(i, j));
  return 0;
}

/**
 * return the (i, j) entry of the matrix
 * Note: no bounds checking is done
 */
float  ccs_compact_float::get (size_t i, size_t j)
{
  size_t sz = m_colptrs[j+1]-m_colptrs[j];
  index_t idx;
  if (i >= USHRT_MAX || j >= USHRT_MAX)
    return -2;
  index_t ii = i;
  index_t jj = j;
  int result = SSUtil::binary_search_compact(m_rowindx + m_colptrs[jj], ii, sz, idx);
  if (result != -1) 
    return (m_val + m_colptrs[j])[idx];
  return 0.0;
}

/**
 * @param i   The row
 * @param j   The col
 * @param posn  Pointer to save the position in the rowindx array where the
 * requested element was found.
 */
float ccs_compact_float::get (size_t i, size_t j, size_t* posn)
{
  for (size_t t = m_colptrs[j]; t < m_colptrs[j+1]; t++)
    if (m_rowindx[t] == i) {
      *posn = t;
      return m_val[t];
    }
  *posn = 0;
  return 0.0;
}

/**
 * Overloaded operator() for get operation.
 */
float ccs_compact_float::operator() (size_t i, size_t j) 
{
  size_t sz = m_colptrs[j+1]-m_colptrs[j];
  int result; 
  index_t idx;
  if (i >= USHRT_MAX || j >= USHRT_MAX)
    return -2;
  index_t ii = i;
  index_t jj = j;
  result = SSUtil::binary_search_compact(m_rowindx + m_colptrs[jj], ii, sz, idx);
  if (result != -1) 
    return (m_val + m_colptrs[jj])[idx];
  return 0.0;
}

/**
 * Set the (i, j) entry of the matrix
 * @return
 *   -     0 if successfully set
 *   -     -1 if entry (i, j) is a zero, then currently can't set it.
 */
int ccs_compact_float::set (size_t i, size_t j, float val)
{
  size_t sz = m_colptrs[j+1]-m_colptrs[j];
  index_t idx;
  if (i >= USHRT_MAX || j >= USHRT_MAX)
    return -2;

  index_t ii = i; index_t jj = j;

  int result = SSUtil::binary_search_compact(m_rowindx + m_colptrs[jj], ii, sz, idx);
  if (result != -1) {
    (m_val + m_colptrs[jj])[idx] = val;
    return 0;
  }
  return -1;
}

/**
 * @param posn   Index into the values or the rowindx array
 * @param val    the value that is written. I.e. m_val[posn] = val
 * @return
 *   -    0 If value was written.
 *   -    -1 If invalid position was specified. Matrix remains unchanged
 */
int ccs_compact_float::set (size_t posn, float val)
{
  if (posn < 0 || posn >= nonzeros())
    return -1;
  m_val[posn] = val;
  return 0;
}


/* Elementwise functions on this matrix */


/// Vector l_p norms for this matrix
/// @todo Faster exponentiation.
/// @param 'p', for lp norms.
float ccs_compact_float::norm (float p)
{
  float nrm = 0.0;
  // we take advantage of non-negativity while computing the matrix norms. 
  // Also we compute exponents faster TODO
  if (is_non_negative()) {
    if (p == 1) {
      nrm = SSUtil::sum(m_val, nonzeros());
    } else if (p == 2) {
      nrm = SSUtil::euclidean_norm(m_val, nonzeros());
    } else {
      nrm = SSUtil::lp_norm(m_val, nonzeros(), p);
    }
  } else {
    if (p == 1) {
      nrm = SSUtil::abs_sum(m_val, nonzeros());
    } else if (p == 2) {
      nrm = SSUtil::euclidean_norm(m_val, nonzeros());
    } else {
      nrm = SSUtil::lp_norm(m_val, nonzeros(), p, true);
    }
  }
  return nrm;
}

/**
 * @param p specifies 'fro', 'l1', 'matrix-l1', 'inf', 'matrix-inf'
 * @return 
 * -    the specified norm
 * -    -1 if could not compute specified norm...
 */
float ccs_compact_float::norm(char* p) 
{
  std::string s(p);
  if (s == "fro") {
    return norm(2);
  } else if (s == "l1") {
    return norm(1);
  } else if (s == "inf") {
    return SSUtil::linf_norm(m_val, nonzeros());
  } else if (s == "matrix-l1") {
    error("Not yet implemented!");
  } else if (s == "matrix-inf") {
    error ("Not yet implemented");
  } else {
    error ("Invalid or unsupported norm requested");
  }
  return -1;
}

/**
 * @param fn  -- Pointer to a function that returns a float, given a float
 * as an argument.
 * Apply an arbitrary function elementwise to this matrix. The function is
 * applied only to the non-zeros of the matrix.
 * @return 0.0
 */
float ccs_compact_float::apply (float (* fn)(float))
{
  for (size_t i = 0; i < nonzeros(); i++) {
    m_val[i] = fn(m_val[i]);
  }
  return 0.0;
}

void ccs_compact_float::normalize_columns()
{
  for (size_t c = 0; c < ncols(); c++) {
    float dp = col_norm(c, 2);
    col_scale(c, 1.0/dp);
  }
}

/* Operations on columns of this matrix */


float ccs_compact_float::col_norm(size_t c)
{
  return col_norm(c, 2);
}

/**
 * @param c      the column whose norm to computer
 * @param p      compute l_p norm
 * @return
 *    -   -1 if invalid 'p' norm requested
 *    -   -2 if invalid column specified
 *    -   float value as the requested norm
 */
float ccs_compact_float::col_norm (size_t c, float p)
{
  float n = 0.0;
  if (p <= 0)
    return -1;
  if (c >= ncols())
    return -2;

  if (p == 2) {
    for (size_t i = m_colptrs[c]; i < m_colptrs[c+1]; i++)
      n += m_val[i]*m_val[i];
    return sqrt(n);
  } else if (p == 1) {
    if (is_non_negative()) {
      for (size_t i = m_colptrs[c]; i < m_colptrs[c+1]; i++)
        n += m_val[i];
      return n;
    } else {
      for (size_t i = m_colptrs[c]; i < m_colptrs[c+1]; i++)
        n += fabs(m_val[i]);
      return n;
    }
  } else {
    if (is_non_negative()) {
      for (size_t i = m_colptrs[c]; i < m_colptrs[c+1]; i++)
        n += pow (m_val[i], p);
      return pow (n, 1/p);
    } else {
      for (size_t i = m_colptrs[c]; i < m_colptrs[c+1]; i++)
        n += pow ( fabs(m_val[i]), p);
      return pow (n, 1/p);
    }
  }
  
  return n;
}

/**
 * @todo
 */
float ccs_compact_float::col_dot  (size_t c1, size_t c2)
{
  vector_float* col2 = gsl_vector_float_calloc(nrows());
  get_col(c2, col2);
  float dp = col_dot(c1, col2);
  gsl_vector_float_free(col2);
  return dp;
}

/**
 * Does dot product of column 'c' with vector 'v'
 */
float ccs_compact_float::col_dot(size_t c, vector_float* v)
{
  float dp = 0.0;
  for (size_t i = m_colptrs[c]; i < m_colptrs[c+1]; i++) {
    dp += m_val[i] * v->data[ m_rowindx[i] ];
  }
  return dp;
}

/**
 * @todo Since this function can introduce new non-zeros ....
 */
int    ccs_compact_float::col_add  (size_t c1, size_t c2)
{
  return -1;
}

// v <-- v + col_c
int ccs_compact_float::col_add(size_t c, vector_float* v)
{
  for (size_t i = m_colptrs[c]; i < m_colptrs[c+1]; i++)
    v->data[ m_rowindx[i] ] += m_val[i];
  return 0;
}

/**
 * @todo Since this function can introduce new non-zeros ....
 */
int    ccs_compact_float::col_sub  (size_t c1, size_t c2)
{
  return -1;
}

int ccs_compact_float::col_sum (vector_float* r)
{
  float s = 0.0;
  for (size_t c = 0; c < ncols(); c++) {
    s = 0.0;
    for (size_t i = m_colptrs[c]; i < m_colptrs[c+1]; i++) {
      s += m_val[i];
    }
    r->data[c] = s;
  }
  return 0;
}
/**
 * @todo
 */
int    ccs_compact_float::col_scale(size_t c, float s)
{
  for (size_t i = m_colptrs[c]; i < m_colptrs[c+1]; i++)
    m_val[i] *= s;
  return 0;
}

/**
 * @todo
 */
int ccs_compact_float::col_sub (size_t c1, vector_float* c2)
{
  return -1;
}


/**
 * Scaled col c1, and stores result in a preallocated vector r
 * r <-- r +  s*column
 * @return
 *    -    0 if computation was successful
 *    -    -1 if dimension mismatch
 *    -    -2 if invalid column requested
 *    
 */
int ccs_compact_float::col_scale(size_t c1, float s, vector_float* r)
{
  if (c1 >= ncols())
    return -2;

  if (r->size != nrows())
    return -1;

  // 'r' had better be inited properly before sending in here.

  for (size_t i = m_colptrs[c1]; i < m_colptrs[c1+1]; i++)
    r->data [ m_rowindx [ i ] ] += s * m_val [ m_rowindx [i] ];
  
  return 0;
}

/* Operations on rows of this matrix */

float ccs_compact_float::row_norm(size_t r)
{
  return row_norm(r, 2);
}

/**
 * @todo
 */
float ccs_compact_float::row_norm (size_t r, float p)
{
  float nrm = 0.0;
  int result; index_t idx1;

  if (r >= USHRT_MAX)
    return -2;

  index_t rr = r;

  if (p == 1) {
    for (size_t c = 0; c < ncols(); c++) {
      size_t sz = m_colptrs[c+1] - m_colptrs[c];
      result = SSUtil::binary_search_compact(m_rowindx + m_colptrs[c], rr, sz, idx1);
      if (result != -1) {
        nrm += fabs( (m_val + m_colptrs[c])[idx1] );
      }
    }
    return nrm;
  } else if (p == 2) {
    for (size_t c = 0; c < ncols(); c++) {
      size_t sz = m_colptrs[c+1] - m_colptrs[c];
      result = SSUtil::binary_search_compact(m_rowindx + m_colptrs[c], rr, sz, idx1);
      if (result != -1) {
        float t = (m_val + m_colptrs[c])[idx1];
        nrm += t*t;
      }
    }
    return sqrt(nrm);
  } else 
    return nrm;
}

/**
 * @param r1 --- First row
 * @param r2 --- Second row
 * Does <r1, r2> inner product. This function uses SSUtil::binary search to
 * determine if a particular column has row elements or not.
 */
float ccs_compact_float::row_dot  (size_t r1, size_t r2)
{
  // For each column, we search the row indx array for the row_entry.
  index_t idx1, idx2;
  int result;
  float dot = 0.0;

  // How to complain here ???
  if (r1 >= USHRT_MAX || r2 >= USHRT_MAX)
    return 0.0;

  index_t rr1 = r1;
  index_t rr2 = r2;
  
  
  for (size_t c = 0; c < ncols(); c++) {
    size_t sz = m_colptrs[c+1] - m_colptrs[c];
    result = SSUtil::binary_search_compact(m_rowindx + m_colptrs[c], rr1, sz, idx1);
    if (result != -1) {
      result = SSUtil::binary_search_compact(m_rowindx + m_colptrs[c], rr2, sz, idx2);
      if (result != -1)
        dot += (m_val + m_colptrs[c])[idx1] * (m_val + m_colptrs[c])[idx2];
    }
  }
  return dot;
}

/**
 * Does dot product of row r of matrix with vector v
 */
float ccs_compact_float::row_dot  (size_t r, vector_float* v)
{
  float dot = 0.0;
  index_t idx;
  int result;

  // TODO: Error checking...
  if (r >= USHRT_MAX)
    return dot;

  index_t rr = r;
  for (size_t c = 0; c < ncols(); c++) {
    size_t sz = m_colptrs[c+1] - m_colptrs[c];
    result = SSUtil::binary_search_compact(m_rowindx + m_colptrs[c], rr, sz, idx);
    if (result != -1)
      dot += (m_val + m_colptrs[c])[idx] * gsl_vector_float_get(v, c);
  }
  return dot;
}

float ccs_compact_float::row_dot (size_t r, vector_float* x, size_t* ix, size_t ixn)
{
  error("CCS_COMPACT_FLOAT::row_dot(): Not implemented");
  return 0.0;
}

/// dot ( row i, col j)
float ccs_compact_float::row_col_dot (size_t r, size_t c)
{
  float dp = 0.0;
  vector_float* v = gsl_vector_float_alloc(ncols());
  get_row(r, v);
  dp = col_dot(c, v);
  gsl_vector_float_free(v);
  return dp;
}


/**
 * @todo TODO
 */
int    ccs_compact_float::row_add  (size_t r1, size_t r2)
{
  return -1;
}

/**
 * @todo
 */
int    ccs_compact_float::row_sub  (size_t r1, size_t r2)
{
  return -1;
}


/*
 * Computes the sum of each row of the input matrix ...
 */
int ccs_compact_float::row_sum(vector_float* r)
{
  for (size_t i = 0; i < ncols(); i++) {
    for (size_t j = m_colptrs[i]; j < m_colptrs[i+1]; j++)
      r->data[m_rowindx[j]] += m_val[j];
  }
  return 0;
}

int    ccs_compact_float::row_scale(size_t r, float s)
{
  index_t idx; int result;
  if (r >= USHRT_MAX)
    return -1;
  
  index_t rr = r;

  for (size_t c = 0; c < ncols(); c++) {
    size_t sz = m_colptrs[c+1] - m_colptrs[c];
    result = SSUtil::binary_search_compact(m_rowindx + m_colptrs[c], rr, sz, idx);
    if (result != -1)
      (m_val + m_colptrs[c])[idx] *= s;
  }
  return 0;
}
 
int ccs_compact_float::row_daxpy(size_t i, float a, vector_float* r)
{
  index_t idx; int result;
  if (i >= USHRT_MAX)
    return -1;

  index_t ii = i;

  for (size_t c = 0; c < ncols(); c++) {
    size_t sz = m_colptrs[c+1] - m_colptrs[c];
    result = SSUtil::binary_search_compact(m_rowindx + m_colptrs[c], ii, sz, idx);
    if (result != -1) {
      float t = gsl_vector_float_get(r, c);
      gsl_vector_float_set(r, c, a * ((m_val + m_colptrs[c])[idx]) + t);
    }
  }
  return 0;
}

/**
 * @todo
 */
int ccs_compact_float::row_add (size_t r1, vector_float* v)
{
  return -1;
}

/**
 * @todo
 */
int ccs_compact_float::row_sub (size_t r1, vector_float* v)
{
  return -1;
}


/**
 * Scaled row r, and stores result in a preallocated vector r
 * @return
 *    -    0 if computation was successful
 *    -    -2 if invalid row requested
 * Dimension based error checking skipped because it's useful to sometimes
 * have an 'r' larger than ncols()
 */
int ccs_compact_float::row_scale(size_t r, float s, vector_float* row)
{
  vector_float* v = gsl_vector_float_alloc(ncols());
  if (get_row(r, v) < 0)
    return -1;

  for (size_t c = 0; c < std::min(v->size, row->size); c++)
    gsl_vector_float_set(row, c, s * v->data[c]);
  gsl_vector_float_free(v);
  return 0;
}


/**
 * This function computes y := A*x or y := A'*x
 * @param tranA, whether to use A or A' (true means use A')
 * @param x      A dense vector
 * @param y      Output (must be a preallocated vector)
 * @return
 *   -   0 If computation was ok
 *   -   1 Computation failed due to dimension mismatch
 */
int ccs_compact_float::dot(bool tranA, vector_float*x, vector_float* y)
{
  if (tranA)
    return matrix_tran_vec(x, y);
  else
    return matrix_vec(x, y);
}

/**
 * Function does result <- Ax
 * @param x The vector to be multiplied by this,
 * @param result the *UNALLOCATED* pointer result, which  will be allocated
 * to hold the result if the computation can be carried out.
 * @return
 *    -    0 if computation was ok
 *    -    1 if computation could not be carried otu (dimension mismatch)
 */
int ccs_compact_float::matrix_vec (vector_float* x, vector_float* result)
{
  // Dimension check
  if (ncols() != x->size || result->size != nrows()) {
    return 1;
  }
    
  for (size_t i = 0; i < ncols(); i++) {
    for (size_t j = m_colptrs[i]; j < m_colptrs[i+1]; j++)
      result->data[m_rowindx[j]] += m_val[j]*x->data[i];
  }
  
  return 0;
}

/**
 * result <- Transpose[Matrix].x = A'x
 * @see dot
 */
int ccs_compact_float::matrix_tran_vec (vector_float* x, vector_float* result)
{
   // Dimension check.
  if (nrows() != x->size || result->size != ncols()) {
    return 1;
  }
  
  float yi;
  for (size_t i = 0; i < ncols(); i++) {
    yi = 0;
    for (size_t j = m_colptrs[i]; j < m_colptrs[i+1]; j++) {
      yi += m_val[j] * x->data[ m_rowindx[j] ];
    }
    result->data[i] = yi;
  }
  return 0;
}

/**
 * @todo Multiplication of two sparse matrices this = this * m or m * this
 * need to do RTTI here
 */
int ccs_compact_float::mult(matrix_compact_float* m, int leftOrRight)
{
  // Check what type of matrix 'm' is and invoke appropriate methods ...
  return -1;
}

int ccs_compact_float::mul (matrix_compact_float* m)
{
  return -1;
}

/**
 * @todo Add two sparse matrices
 */
int ccs_compact_float::add (matrix_compact_float* m)
{
  return -1;
}

/**
 * @todo
 */
int ccs_compact_float::sub (matrix_compact_float* m)
{
  return -1;
}


/**
 * Computes the elementwise matrix product of this sparsematrix with a
 * general matrix (dense or otherwise)
 * this <- this .* m
 * @param m a generic matrix
 * @return
 *  -  0 If multiplication carried out successfully
 *  -  1 If dimension mismatch
 */
int ccs_compact_float::dot (matrix_compact_float* m)
{
  if (nrows() != m->nrows() || ncols() != m->ncols())
    return 1;

  for (size_t c = 0; c < ncols(); c++)
    for (size_t r = m_colptrs[c]; r < m_colptrs[c+1]; r++)
      m_val[r] *=  m->get( m_rowindx[r], c);
  return 0;
}


/**
 * Computes the elementwise matrix division of this sparsematrix with a
 * general matrix (dense or otherwise)
 * this <- this ./ m
 * @param m a generic matrix
 * @return
 *  -  0 If multiplication carried out successfully
 *  -  1 If dimension mismatch
 */
int ccs_compact_float::div (matrix_compact_float* m)
{
  if (nrows() != m->nrows() || ncols() != m->ncols())
    return 1;

  for (size_t c = 0; c < ncols(); c++)
    for (size_t r = m_colptrs[c]; r < m_colptrs[c+1]; r++)
      m_val[r] /=  m->get( m_rowindx[r], c);

  return 0;
}


/**
 * Prints out matrix to stdout
 * General debugging routine as of now.
 */
void ccs_compact_float::print()
{
  /*
    std::cerr << nrows() << " x " << ncols() << std::endl;
  for (index_t i = 0; i < nrows(); i++) {
    for (index_t j = 0; j < ncols(); j++)
      std::cout << get(i,j) << " ";
    std::cout << std::endl;
  }

  */
  for (size_t c = 0; c < ncols(); c++) {
    for (size_t r = m_colptrs[c]; r < m_colptrs[c+1]; r++)
      std::cout << m_rowindx[r] << ", " << c << ", " << m_val[r] << "\n";
  }
}

/**
 * This function creates a sparse CCS matrix as the transpose of 'this'
 * matrix. 
 * @author stefje
 * @todo Creating a transpose function that returns the CRS version.
 */
matrix_compact_float* ccs_compact_float::transpose_copy()
{
  ccs_compact_float* trans = new ccs_compact_float();
  trans->matrix_alloc(ncols(), nrows(), nonzeros());
  // pointers to indicate where to fill in the next value in trans's columns
  size_t* colind = (size_t*) malloc ( sizeof(size_t) * (nrows()+1) );
  memset(colind, 0, (nrows()+1)*sizeof(size_t)/sizeof(char));
  memset(trans->m_colptrs, 0, (nrows()+1)*sizeof(size_t)/sizeof(char));


  //count the nonzeros in each row
  for (size_t i = 0; i < nonzeros(); ++i) {
    trans->m_colptrs[m_rowindx[i]+1]++;
  }

  //add for column pointers
  for (size_t i=1; i<nrows()+1; ++i)  {
    trans->m_colptrs[i+1] += trans->m_colptrs[i];
  }

  for (size_t col=0; col<ncols(); ++col)  {
    for (size_t i = m_colptrs[col]; i<m_colptrs[col+1]; ++i) {
      size_t ind = trans->m_colptrs[m_rowindx[i]] + colind[m_rowindx[i]];
      ++colind[m_rowindx[i]];

      trans->m_val[ind] = m_val[i];
      trans->m_rowindx[ind] = col;
    }
  }

  free(colind);
  return trans;
}

void ccs_compact_float::free_data()
{
  if (m_val) free(m_val); m_val = 0;
  if (m_colptrs) free(m_colptrs); m_colptrs = 0;
  if (m_rowindx) free(m_rowindx); m_rowindx = 0;
}

#if 0
/**
 * This function converts the current matrix to a CRS version. A copy of the
 * original data is created and returned.
 * @author stefje
 */
crs_compact_float* ccs_compact_float::to_crs()
{
  size_t* nrp, *ncind;
  float* nval;

  crs* newm = new crs();
  newm->matrix_alloc(nrows(), ncols(), nonzeros());
  // pointers to indicate where to fill in the next value in newm's columns
  size_t* rowind = new size_t[nrows()+1];
  memset(rowind, 0, (nrows()+1)*sizeof(size_t)/sizeof(char));

  nrp = newm->get_rowptrs();
  ncind = newm->get_colindx();
  nval  = newm->get_data();

  memset(nrp, 0, (nrows()+1)*sizeof(size_t)/sizeof(char));

  //count the nonzeros in each row
  for (size_t i = 0; i < nonzeros(); ++i) {
    nrp[m_rowindx[i]+1]++;
  }

  //add for column pointers
  for (size_t i=1; i< nrows(); ++i)  {
    nrp[i+1] += nrp[i];
  }  

  // go through the columns and copy the values into the proper location in
  // newm. We know the index in newm->m_val by the column pointers created
  // above and the counters in rowind
  // at the same time we insert the column indices  
  for (size_t col=0; col<ncols(); ++col)  {
    for (size_t i = m_colptrs[col]; i< m_colptrs[col+1]; ++i) {
      size_t ind = nrp[m_rowindx[i]] + rowind[m_rowindx[i]];
      ++rowind[m_rowindx[i]];

      nval[ind] = m_val[i];
      ncind[ind] = col;
    }
  }

  delete rowind;
  return newm;  

}
#endif

/**
 * @todo Maybe will not do an in-place tranpose, not worth the hassle?
 */
void ccs_compact_float::transpose()
{
  return;
}

int ccs_compact_float::compute_AtA(matrix_compact_float* result)
{
  return -1;
}

/**
 * @param s  a scalar by which all the lements of the matrix are scaled.
 * @return 0
 */
int ccs_compact_float::scale(float s)
{
  for (size_t i = 0; i < NZ; i++)
    m_val[i] *= s;
  return 0;
}

/**
 * @param s  A scalar that is added to all the nonzeros of the matrix
 */
int ccs_compact_float::add_const(float s)
{
  for (size_t i = 0; i < NZ; i++)
    m_val[i] += s;
  return 0;
}


void SSLib::ccs_compact_float::error (std::string s)
{
  std::cerr << "CCS_COMPACT:"  << s << std::endl;
}



