// File: ccs_compact.h -*- c++ -*-
// Time-stamp: <13 Februar 2008 02:09:28  CET --  suvrit>
// Impl. of ccs format sparse matrices

#ifndef CCS_COMPACT_H
#define CCS_COMPACT_H

#include "sparsematrix_compact.h"

namespace SSLib {
  class crs_compact;
  /**
  * @class ccs, contains an implementation of a CCS sparsematrix. It supports
  * many of the operations suggested by the abstract class sparsematrix.
  *
  */
  class ccs_compact : public sparsematrix_compact {
  protected:			// for efficiency reasons.
    /**
    * The actual non-zeroes themselves.
    */
    double* m_val;

    /**
    * Colptrs for CCS structure. These hold indices into the
    * <code>m_rowindx</code> and <code>val</code>. The i-th column starts at
    * m_colptrs[i] and ends at m_colptrs[i+1]-1
    */
    index_t* m_colptrs;
  

    /**
    * Row indices for CCs structure. This array contain the row indices of
    * the matrix elements. For example, m_rowindx[ m_colptrs[i] ] gives the
    * row index of the 1st non-zero element in the i-th column.
    */
    index_t* m_rowindx;

  protected:    
    /**
    * Is the data in externally allocated arrays. That is, if the m_val,
    * m_colptrs and m_rowindx arrays are externally allocated (useful for
    * creating a ccs sparsematrix from another matrix of similar structure etc.
    */
    bool m_isExternal;

    /**
    * Filename prefix. A CCS matrix is loaded from a collection of 4
    * files. For example, if m_filename="mat", then the code that loads this
    * matrix will search for the files 'mat_dim', 'mat_row_ccs',
    * 'mat_col_ccs' and 'mat_<m_type>_nz' depending on the value of <code>m_type</code>
    */
    std::string m_filename;

    /* matrix type, 'txx', 'tfn', etc. */
    std::string m_type;

  private:
    /* load files */
    virtual int load_bin(std::string prefix);
    virtual int load_txt  (std::string prefix);
    virtual int save_bin(const char*); 
    virtual int save_txt  (const char*);

  public:
    ccs () {
      m_isExternal = false;
      m_filename = "";
      m_type = "txx";
      m_val = 0;
      m_colptrs = 0;
      m_rowindx = 0;
    }

    virtual ~ccs () {
      //free_data();
    }

    
    void free_data();
    

    std::string get_type() const { return m_type; }
    void set_type (std::string s) { m_type = s;}

    /** @name MatrixCreation
    * Creating and saving ccs matrix functions. These functions allow the
    * user to load a CCS matrix from file, or from already available
    * arrays. They also allow the user to save the current matrix to file.
    *
    */

    /*@{*/


    // Allocate the matrix
    int matrix_alloc (index_t m, index_t n, index_t nz);
    int matrix_calloc (index_t m, index_t n, index_t nz);
    int matrix_resize (index_t m, index_t n, index_t nz);
    virtual int load(const char*, bool asbin);
    virtual int save_coord_bin(const char*);
    virtual int save (const char*, bool asbin);

    /// Write out matrix in a specified way
    virtual int    save(const char*, bool asbin, int typ);
    
    /// Write out matrix to disk in CCS, CRS, etc. formats
    virtual int save_as_crs (const char*, bool);
    virtual int save_as_coord (const char*, bool);
    virtual int save_as_colcoord (const char*, bool);

    virtual int save_as_rowcoord (const char*, bool);
    virtual int save_rowcoord_bin(const char*);
    virtual int save_rowcoord_txt(const char*);

    virtual int save_as_matlab (const char*, bool);
    virtual int save_matlab_bin(const char* fil);
    virtual int save_matlab_txt(const char*);

    virtual int save_as_hb (const char*, bool);
    virtual int save_hb_bin(const char*);
    virtual int save_hb_txt(const char*);

    virtual int save_as_dense (const char*, bool);
    virtual int save_dense_bin(const char*);
    virtual int save_dense_txt(const char*);

    void externally_alloc (index_t* col, index_t* row, double* val);
    // Return a deep copy
    ccs* clone();

    // Augment matrix by adding a row
    virtual int matrix_add_row(vector* r);
    virtual int matrix_remove_row(index_t r);

    // Augment by adding a column
    virtual int matrix_add_col(vector* c);
    virtual int matrix_remove_col(index_t c);

    /*@}*/


    int     scale(double);
    int     add_const(double);

    /** @name Accessors
    * Accessing the matrix elements 
    */
    /*@{*/

    index_t* get_colptrs() { return m_colptrs;}
    index_t* get_rowindx() { return m_rowindx;}
    double* get_data() { return m_val;}

    /// Return row 'r' of matrix
    int get_row(index_t r, vector*&);
    /// Return col 'c' of matrix
    int get_col(index_t c, vector*&);
    /// return the (i, j) entry of the matrix
    double  get (index_t i, index_t j);
    double  get (index_t i, index_t j, index_t* posn);
    double  operator()(index_t i, index_t j);
    /// set the (i, j) entry of the matrix.
    int     set (index_t i, index_t j, double val);
    int      set (index_t posn, double val);
    /// Sets the specified row to the given vector
    virtual int set_row(index_t r, vector*&);

    /// Sets the existing non-zeros, ignores other entries
    virtual int fast_set_row(index_t r, vector*&);


    /// Sets the specified col to the given vector
    virtual int set_col(index_t c, vector*&);

    /// Sets only the existing non-zeros, ignores others
    virtual int fast_set_col(index_t c, vector*&);

    /// Returns main or second diagonal (if p == true)
    virtual int get_diag(bool, vector*&);

    /// NOP
    virtual int set_diag(bool b, vector*& v) {
      return -1;
    }

    /// Returns a submatrix that goes from row i1--i2 and cols j1--j2
    virtual int submatrix(index_t i1, index_t j1, index_t i2, index_t j2, matrix*&);
    
    /// Returns a submatrix given the row-col index sets
    virtual int submatrix(gsl_vector_uint* I, gsl_vector_uint* J, matrix*& m) {
      return -1;
    }

    /*@}*/

    /** @name Elementwise
    * Elementwise functions/operations on this matrix.
    */
    /*@{*/

    // Vector l_p norms for this matrix
    virtual double norm (double p);
    virtual double norm (char*  p);
    // Apply an arbitrary function elementwise to this matrix
    virtual double apply (double (* fn)(double));

    /*@}*/

    /*@{*/
    virtual double col_norm (index_t c);
    virtual double col_norm (index_t c, double p);
    virtual double col_dot  (index_t c1, index_t c2) ;
    virtual double col_dot  (index_t c, vector* v);
    virtual int    col_add  (index_t c1, index_t c2) ;
    virtual int    col_sub  (index_t c1, index_t c2) ;
    virtual int    col_scale(index_t c1, double s);
    virtual int    col_sub  (index_t c1, vector* v);
    virtual int    col_add  (index_t c1, vector* v) ;
    virtual int    col_scale(index_t c1, double s, vector* r);
    virtual int    col_sum  (vector*);
    /*@}*/

    
    /** @name
    * Operations on rows of this matrix
    */
    /*@{*/
    virtual double row_norm (index_t r);
    virtual double row_norm (index_t r, double p);
    virtual double row_dot  (index_t r1, index_t r2) ;
    virtual double row_dot  (index_t r, vector* v);
    virtual double row_dot  (index_t r, vector* x, index_t* ix,index_t);
    virtual int    row_add  (index_t r1, index_t r2) ;
    virtual int    row_sub  (index_t r1, index_t r2) ;
    virtual int    row_scale(index_t c1, double s);
    virtual int    row_add  (index_t r1, vector* r);
    virtual int    row_sub  (index_t r1, vector* r);
    virtual int row_scale(index_t c1, double s, vector* r);
    virtual int row_daxpy(index_t i, double a, vector* r);
    virtual int    row_sum  (vector*);

    /// dot ( row i, col j)
    virtual double row_col_dot (index_t r, index_t c);

    virtual int dot (bool tranA, vector* x, vector* r); 
    matrix* transpose_copy();
    /*@}*/

    /*@{*/


    // Multiplication of two matrices this = this * m or m * this
    virtual int mult (matrix* m, int leftOrRight);

    /**
    * Subtract a sparse matrix from this matrix. The operation implemented is 
    * this := this + m.
    * @param m is a sparsematrix* arg
    * @return 
    *   -     0 if subtraction succeeded
    *   -     1 if size mismatch
    *   -     2 if some other error
    * @warning This function can cause fill-in by destroying non-zeros, and
    * can be slow.
    */
    virtual int add (matrix* m);


    /**
    * Subtract a sparse matrix from this matrix. The operation implemented is 
    * this := this - m.
    * @param m is a sparsematrix* arg
    * @return 
    *   -     0 if subtraction succeeded
    *   -     1 if size mismatch
    *   -     2 if some other error
    * @warning This function can cause fill-in by destroying non-zeros, and
    * can be slow.
    */  
    virtual int sub (matrix* m);


    /**
    * Hadamard product. After the execution of this function our current
    * matrix is: this := this .* m (in matlab notation)
    * @param m is a gsl_matrix* argument
    * @return 
    *  -  0 if the dot product was carried out
    *  -  1 if matrix size mismatch
    *  -  2 if any other error
    */
    virtual int dot (matrix* m);

    /**
    * Hadamard product. After the execution of this function our current
    * matrix is: this := this .* m (in matlab notation). This function does
    * a runtime type check to determine the type of 'm' to optimize the dot
    * product operation (since two ccs matrices or ccs * harwell boeing can
    * be dotted much faster without using any get(i, j) operations).
    * @param m is a sparsematrix* argument
    * @return 
    *  -  0 if the dot product was carried out
    *  -  1 if matrix size mismatch
    *  -  2 if any other error
    */
    virtual int div (matrix* m);


    virtual int mul (matrix* m);
    /*@}*/

    /** @name Misc 
    * Miscellanous functions
    */
    /*@{*/

    /**
    * @return true if this matrix is external, i.e., the rows, cols
    * etc. arrays are directly initialized from outside, and not allocated
    * by us.
    */
    bool isExternal() const { return m_isExternal;}
    virtual void print ();
    virtual void transpose();

    /// Normalize the columns of the matrix to have unit L2 norm
    virtual void normalize_columns();

    virtual int compute_AtA(matrix* result);
    virtual void error(std::string);
    /*@}*/


    /** 
     * conversions
     */
    crs* to_crs();

  private:
    // Matrix vec mpy, result = Ax
    int matrix_vec (vector* x, vector* result);

    // Transpose[Matrix].vec = A'x
    int matrix_tran_vec (vector* x, vector* result);
  };

}

#endif // CCS_COMPACT
