// File: nmfdriver.cc
// Author: Suvrit Sra
// Time-stamp: <18 January 2006 04:41:07 PM CST --  suvrit>
// Date of Creation: 24th February, 2003

// Implements the NmfDriver class

// Copied directly over from original main.cc
//
// main.cc -- driver routine for the factorization algorithms
// Copyright (C) 2003 Suvrit Sra (suvrit@cs.utexas.edu)

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

/* ****************************************************************************
 * File: nmfdriver.cc
 * Contains the main driver program for running the non-negative factorizations 
 * Author: Suvrit Sra
 * 
 * Revision History:
 *
 * 01/18/2006   Cleaned out to prepare for initial release.
 * 11/01/2005   Revised to clean it a bit.
 * 09/14/2004   added Breg2Test1
 * 03/09/2004   added BregTest1
 * 03/09/2004   added Breg 
 * 02/26/2004   added FrobWt2
 * 02/26/2004   added FrobWt1
 * 01/22/2004   added FactorizationHybrid2 also
 * 01/22/2004   added FactorizationFrobSlow also...
 * 12/11/2003   added option to normalize V and scale H corr.
 * 12/03/2003   Wrote the long help doc. comment.
 * 12/01/2003   Moved over to nmfdriver.cc
 * 11/24/2003   Added SVD for sparse matrices via libmyssvd
 * 11/15/2003   Added <cmath> header
 * 04/18/2003   Added option for AKL algo
 * 04/18/2003   Added options for Hybrid algorithm
 * 04/15/2003   Added options for avoiding calc. of obj function.
 * 04/12/2003   Added options for running with just compute obj
 * 03/27/2003   Some organization and cleaning.
 * 03/17/2003   Modified to use gsl and ATLAS
 * 02/24/2003   Created
 *
 * See: Algorithms for non-negative matrix factorization: Lee, Seung and
 * also the Lawson-Hanson NNLS algorithm.
 *
 * ****************************************************************************/

#include <iostream>
#include <cstdio>
#include <cmath>

#include <ssvd/driver.h>
#include <ssvd/s_opts.h>

#include "SparseMatrix.h"
#include "Factorization.h"
#include "FactorizationFrob.h"
#include "FactorizationKL.h"
#include "FactorizationNNLS.h"
#include "FactorizationSNNLS.h"
#include "FactorizationSKL.h"
#include "FactorizationSFrob.h"
#include "FactorizationHybrid.h"
#include "FactorizationSHybrid.h"
#include "FactorizationAKL.h"
#include "FactorizationSAKL.h"
#include "FactorizationFrobSlow.h"
#include "FactorizationHybrid2.h"
#include "FactorizationFrobWt1.h"
#include "FactorizationFrobWt2.h"
#include "FactorizationBreg.h"
#include "FactorizationBregTest1.h"
#include "FactorizationBreg2Test1.h"

#include "util.h"
#include "nmfdriver.h"

extern "C" {
#include <f2c.h>
#include <clapack.h>
}


/**
 * Process command line arguments in traditional (k&r) style
 * Here the DriverOpts struct has to be allocated as it does not exist by
 * default... 
 */
int NmfDriver::execute(char **argv)
{
  opts = new DriverOpts();
  opts->progname = *argv;
  // Traditional style
  while (*++argv) {
    if (*argv[0] == '-') {
      ++*argv;
      switch (*argv[0]) {
      case 'h': 
	showHelp(1);
	return -1;
      case 'o':
	++argv;
	opts->objonly = true;
	if (*argv) {
	  if (*argv[0] != '-') {
	    opts->objvhprefix = strdup(*argv);
	  } else {
	    --argv;
	  }
	} else {
	  std::cerr << "Option -o requires an argument. Use -h for usage\n";
	  return -1;
	}
	break;
      case 'c':
	opts->perturb = true;
	break;
      case 'f':
	++argv;
	if (*argv) {
	  if (*argv[0] != '-') {
	    opts->fname = strdup(*argv);
	    opts->cmdok = true;
	  } else {
	    --argv;
	  }
	} else {
	  std::cerr << "Option -f requires an argument. Use -h for usage\n";
	  return -1;
	}
	break;
      case 's':
	opts->sparse = true;
	break;
      case 'b':
	opts->binfile = true;
	break;
      case 'x':
	opts->normalize = true;
	break;
      case 'd':
	opts->dosvd = true;
	break;
      case 't':
	++argv;
	if (*argv && *argv[0] != '-') {
	  strcpy(opts->txx, *argv);
	} else {
	  std::cerr << "Option -t requires an argument. Use -h for usage\n";
	  return -1;
	}
	break;
      case 'a':
	++argv;
	if (*argv && *argv[0] != '-') {
	  opts->algo = atoi(*argv);
	} else {
	  std::cerr << "Option -a requires an argument. Use -h for usage\n";
	  return -1;
	}
	break;
      case 'j':
	++argv;
	if (*argv && *argv[0] != '-') {
	  opts->svdfile = strdup(*argv);
	} else {
	  std::cerr << "Option -j requires an argument. Use -h for usage\n";
	  return -1;
	}
	break;
      case 'e':
	++argv;
	if (*argv && *argv[0] != '-') {
	  opts->epsilon = atof(*argv);
	} else {
	  --argv;
	  std::cerr << "Option -e  argument not specified. Using default\n";
	}
	break;
      case 'v':
	opts->verbose = true;
	break;
      case 'm':
	++argv;
	if (*argv && *argv[0] != '-') {
	  opts->maxiter = atoi(*argv);
	} else {
	  --argv;
	  std::cerr << "Option -m  argument not specified. Using default\n";
	}
	break;
      case 'r':
	++argv;
	if (*argv && *argv[0] != '-') {
	  opts->rank = atoi(*argv);
	} else {
	  --argv;
	  std::cerr << "Option -r  argument not specified. Using default\n";
	  opts->rseed = time(0);
	}
	break;
      case 'p':
	++argv;
	if (*argv) {
	  if (*argv[0] != '-') {
	  opts->rseed = atol(*argv);
	  } else {
	    --argv;
	    std::cerr << "Option -p  argument not specified. Using default\n";
	    opts->rseed = time(0);
	  }
	} else {
	  --argv;
	  std::cerr << "Option -p  argument not specified. Using default\n";
	  opts->rseed = time(0);
	}
	break;
      case 'l':
	opts->slowobj = true;
	break;
      case 'z':
	++argv;
	if (*argv) {
	  if (*argv[0] != '-') {
	    opts->objmodulo = atoi(*argv);
	  } else {
	    --argv;
	    std::cerr << "Option -z  argument not specified. Using default\n";
	    opts->objmodulo = 1;
	  }
	} else {
	  --argv;
	  std::cerr << "Option -z  argument not specified. Using default\n";
	  opts->objmodulo = 1;
	}
	break;
      case 'w':
	++argv;
	if (*argv) {
	  if (*argv[0] != '-') {
	    opts->resfile = strdup(*argv);
	  } else {
	    --argv;
	    std::cerr << "Option -w argument not specified. Using default\n";
	    opts->resfile = "NMFRESULTS";
	  }
	} else {
	  --argv;
	  std::cerr << "Option -w argument not specified. Using default\n";
	  opts->resfile = "NMFRESULTS";
	}
	break;
      case 'W':
	++argv;
	if (*argv) {
	  if (*argv[0] != '-') {
	    opts->wt1file = strdup(*argv);
	  } else {
	    --argv;
	    std::cerr << "Option -W argument not specified. Using default\n";
	  }
	} else {
	  --argv;
	  std::cerr << "Option -W argument not specified. Using default\n";
	}
	break;
      case 'L':
	++argv;
	if (*argv) {
	  if (*argv[0] != '-') {
	    opts->wtLfile = strdup(*argv);
	  } else {
	    --argv;
	    std::cerr << "Option -W argument not specified. Using Identity\n";
	  }
	} else {
	  --argv;
	  std::cerr << "Option -L argument not specified. Using Identity\n";
	}
	break;
      case 'R':
	++argv;
	if (*argv) {
	  if (*argv[0] != '-') {
	    opts->wtRfile = strdup(*argv);
	  } else {
	    --argv;
	    std::cerr << "Option -R argument not specified. Using Identity\n";
	  }
	} else {
	  --argv;
	  std::cerr << "Option -R argument not specified. Using Identity\n";
	}
	break;
      case 'u':
	++argv;
	if (*argv) {
	  if (*argv[0] != '-') {
	    opts->vhprefix = strdup(*argv);
	  } else {
	    --argv;
	    std::cerr << "Option -u argument not specified. Using default\n";
	    opts->vhprefix = 0;
	  }
	} else {
	  --argv;
	  std::cerr << "Option -u argument not specified. Using default\n";
	  opts->vhprefix = 0;
	}
	break;
      case 'n':
	++argv;
	if (*argv) {
	  if (*argv[0] != '-') {
	    opts->nnlsiter = atoi(*argv);
	  } else {
	    --argv;
	    std::cerr << "Option -n  argument not specified. Using default\n";
	    opts->nnlsiter = -1;	// This value is used as a flag to see if we want to call
				// f->setnnlsiter() later on or not. well a little twisted but...
	  }
	} else {
	  --argv;
	  std::cerr << "Option -n  argument not specified. Using default\n";
	  opts->nnlsiter = -1;
	}
	break;
      default:
	std::cerr << *argv << " Unknown option. Use -h for usage\n";
	break;
      }
    }
  }

  //DBG("Options->dosvd == " << opts->dosvd << "\n");
  /* Check to see if cmd line is ok. i.e. -f was specified*/
  if (!opts->cmdok) {
    showHelp();
    return -1;
  }
  //DBG("Options->dosvd == " << opts->dosvd << "\n");
  switch (opts->algo) {
  case FROBENIUS:
    opts->algoname =  "Lee/Seung FROBENIUS";
    break;
  case KL:
    opts->algoname =  "Lee/Seung KL";
    break;
  case NNLS:
    opts->algoname = "ALS using NNLS";
    break;
  case HYBRID1:
    opts->algoname = "Hybrid of NNLS, FROB";
    break;
  case AKL:
    opts->algoname = "Alternating KL";
    break;
  case FROBSLOW:
    opts->algoname = "Lee/Seung Frobenius BLAS2";
    break;
  case HYBRID2:
    opts->algoname = "Hybrid of NNLS, FROBSLOW";
    break;
  case FROBWT1:
    opts->algoname = "Weighted LS Frob, Elementwise";
    break;
  case FROBWT2:
    opts->algoname = "Weighted LS Frob, L A R";
    break;
  case BREGA_B:
    opts->algoname = "Bregman div D(A; BC)";
    break;
  case BREGB_A:
	opts->algoname = "Bregman div D(BC; A)";
	break;
  default:
    opts->algoname = "Invalid specification";
    break;
  }

  if (opts->verbose) {
    std::cerr << *opts << std::endl;
  }

  return performFactorization();
}

/**************************************************************************
 * Display help about the usage. Have kept option of level as of now to   *
 * enable generalization later in the future. Detailed information is as  *
 * below (to be available as documentation one day)                       *
 *                                                                        *
 *                                                                        *
 * -f: (Also see -s, -t) Specifies the file name prefix of the matrix to  *
 * use in case -s has also been specified, since -s indicates we are      *
 * working with sparse                                                    *
 * matrices, and the sparse matrices that are supported are in the so     *
 * called CCS format that we use here. For e.g., on the command line we   *
 * might have                                                             *
 *                                                                        *
 * nmf -s -f classic3 -t txx -r 5 -p 100 -d                               *
 *                                                                        *
 * The -s indicates that the argument given to -f is a file name          *
 * prefix. Consequently the program will search for the files,            *
 * classic3_dim, classic3_txx_nz, classic3_col_ccs, classic3_row_ccs. Note*
 * that it searches for classic3_txx_nz because of the argument to -t flag*
 *                                                                        *
 * -s: Already described above, it is necessary when dealing with sparse  *
 * matrices. If the -s flag is not given, then the argument to -f is      *
 * treated as a filename, that is a matrix written out as a text file of  *
 * the format:                                                            *
 *                                                                        *
 * rows cols                                                              *
 * row11 row12 row13.....                                                 *
 * row21 row22 ....                                                       *
 *                                                                        *
 * -b: If the -b flag is given, it implies that the matrix given to -f is a
 * matrix that was written to disk using fwrite_gsl_matrix (i.e., matrix is
 * in binary rather than text form....(this function is in util.cc). The  *
 * resulting matrices V, H are also stored in binary format on disk       *
 * then.... Note that my function fwrite_gsl_matrix, in util.cc differs   *
 * from the gsl_matrix_fwrite function in that it also stores the matrix  *
 * size in the file allowing for easy reading of the matrix later on. This*
 * option is incompletely implemented right now in the sense that if -b is*
 * given and also -o vhprefix or -u vhprefix etc. are given, those files  *
 * are assumed to be in regular text format ... that can be read using the*
 * read_gsl_matrix function in util.cc                                    *
 *                                                                        *
 * -o: This option takes an argument specifying the prefix of the V, H    *
 * matrices from which to calculate the objective value. For example, the *
 * nmf invocation above would generate files of the names:                *
 *   classic3.fact.5.frob.H and classic3.fact.5.frob.V so the vhprefix    *
 * argument to option -o would be classic3.fact.5.frob                    *
 * NOTE: -o has interaction with -a ... that is, when -o is specified,    *
 *                                                                        *
 * -u: filename prefix from which to initialize the V, H matrices (if one *
 * wants to initialize the iteration from given files rather than randomly*
 * generate V, H each time. This can be useful when conducting experiments*
 * to compare different algorithms and starting them off from the same    *
 * initialization. One way to obtain random V, H init. matrices is to just*
 * use for example, nmf -s -f classic3 -t txx -r 4 -m 0 ... which will stop
 * after 0 iterations but will write V, H to file which can be later used *
 * for initialization. Note there *must* exist matrices of the name       *
 * vhprefix.V and vhprefix.H for this to work.                            *
 *                                                                        *
 * -r: <rank> is used to specify the desired rank of the                  *
 * approximation. Truly speaking the rank might be less than equal to the *
 * desired rank. All that nmf guarantees is that for an input matrix A that
 * is m x n, we will have V = m x <rank>  and H = <rank> x n. Usually V, H*
 * are of full rank but they could lose rank during the iteration (rare   *
 * occurrence).                                                           *
 *                                                                        *
 * -m: <maxiter> Is used to specify the maximum number of iterations that *
 * any chosen algorithm is allowed to make. The default value is 300. When*
 * usign ALS it is advisable to set this to some small value.             *
 *                                                                        *
 * -z: <num> Since the computation of the obj. function itself is quite   *
 * expensive for large matrices, this flag allows one to compute the      *
 * obj. function value once every <num> iterations. (This could make sense*
 * only for sparse matrices, coz. for dense matrices anyway forming the   *
 * product matrix P = V*H is O(n^3) and computation of obj. is only O(n^2)*
 *                                                                        *
 * -a: <0-6> Allows user to select which particular objective function to *
 * minimize and using which algorithm. For example the default value is 0,*
 * which corresponds to a simple implementation of the Lee/Seung Frobenius*
 * error minimization updates. The value 1 corresponds to the Lee/Seung   *
 * I-divergence case and uses their updates. Value 2 is for invoking an   *
 * Alternating Least squares algorithm based on Lawson-Hanson NNLS routine*
 * to minimize the Frobenius error. -a 3 selects a hybrid algorithm that  *
 * ping-pong's between ALS and algo 0 (yet to implement fully).  -a 4     *
 * selects an Alternating minimization algorithm that minimizes the       *
 * I-divergence (of course this has not been implemented yet...)          *
 * -5 invokes a BLAS2 level implementation of the lee/seung algo that     *
 * proceeds by updating a row of H then a col of V and uses the latest    *
 * values of V, H after every such update....better final values......    *
 *                                                                        *
 * -c: perturb entries while doing Lee/Seung iterations (helps avoid      *
 *     divide by zero problems. Default is NO                             *
 *                                                                        *
 * -p: <seed> is used to set the seed of the random number generator (which
 * is otherwise set using the time(0) function.                           *
 *                                                                        *
 * -n: <num> allows the user to set the number of ALS iterations (using   *
 * nnls) do be done in a Hybrid algorithm. This is by default set to 2. It*
 * should usually be set to a small number because the ALS iterations are *
 * expensive.                                                             *
 *                                                                        *
 * -d: Compute the error for a rank <rank> SVD approximation to the input *
 * matrix. For sparse input matrices This invokes the las2 SVDPACKC routine
 * from libmyssvd.a. For dense matrices it invokes LAPACK dgesvd          *
 * routine. Right now the user is not given a choice on which SVD routine *
 * he wants to invoke.                                                    *
 *                                                                        *
 * -e: <epsilon> The tolerance factor. When the change in obj. value      *
 * between two iterations falls below epsilon the iterations stop. This   *
 * param can be useless if used together with -z <num> with <num> > 1.    *
 *                                                                        *
 * -h: Displays a help message                                            *
 * -v: Turns on verbosity leading to printing of information about the    *
 * execution of the program. Also, if -h -v is used verbose and more      *
 * detailed help about command line options is given.                     *
 *                                                                        *
 * -l: Use slow objective value computation (note -l is not the default but
 *   using -l causes objective value to be computed correctly (this is an *
 * issue with sparse matrices, with sparse matrices, we might not want to *
 * compute the objective value using all m x n entries in the product V*H *
 * and instead use only those corresponding to the non-zeros of A. But if *
 * -l flag is given, the obj. value is computed using all the m x n entries
 * and takes commensurate time.                                           *
 *                                                                        *
 * -x: Normalize the V matrix at each iteration (multiply H               *
 * correspondingly)                                                       *
 *                                                                        *
 * -w: <reslt file> Writes out final relative error onto result file      *
 *
 *************************************************************************/
void NmfDriver::showHelp(int level)
{
  std::cerr << opts->progname << ":  [-s] [-b] [-o vhprefix] -f file [-t txx|txn] [-r<rank>] [-a 0-4] [-e epsilon] [-m maxiter] [-p seed] [-u] [-d] [-z <num>][-v] [-w resltfile] [-j svals_file]\n";
  if (opts->level > 0) {
    std::cerr << "This program calculates two non-negative approximate factors for an input matrix A. The factors are called V, H"
              << " they correspond to A \\approx V*H\n";
    std::cerr 
      << "\t-a [0--13]\n"
      << "\t\t 0: lee seung frob\t 1: lee seung KL\n"
      << "\t\t 2: ANNLS,  3: Hybrid of ANNLS and LS Frob\n"
      << "\t\t 4: Alternating KL, 5: LS frob (without gemm)\n"
      << "\t\t 6: Hybrid of ANNLS and LS Frob (Slower Version)\n"
      << "\t\t 7: Elementwise weighted Frob.\n"
      << "\t\t 8: min ||L ( A - Ah ) R||^2 for general matrices L, R\n"
      << "\t\t 9, 10, 11: Reserved for future\n"
      << "\t\t 12: Bregman Divergence D (A ; VH )\n"
      << "\t\t 13: Bregman Divergece D (VH ; A)\n"
      << "\t NOTE: All options except 0, 1 require dense matrices as input\n"
      << "\t-b read <filename> as a binary file\n"
      << "\t-c do perturbation of values for V, H\n"
      << "\t-d Compute rank r svd approx of input matrix\n"
      << "\t-e <epsilon>  Convergence tolerance\n"
      << "\t-f <matrix prefix|matrix filename>\n"
      << "\t-h This help message\n"
      << "\t-j <svalsfile> Use sing. values in SVALSFILE instead of calc. again\n"
      << "\t-l Lazy obj. computation for first time (avoids Sparsity)\n"
      << "\t-L <left scaling matrix>, default = Identity\n"
      << "\t-R <right scaling matrix>, default = Identity\n"
      << "\t-m <maxiter> Maximum number of iterations\n"
      << "\t-n NUM Set number of NNLS iters to NUM, makes sense only if -a 3\n"
      << "\t-o <vhprefix> filename prefix for V, H matrices. Compute OBJ only\n"
      << "\t When using -o the argument of -a specifies type of objval to find\n"
      << "\t-p <seed> pseudo-random number seed\n"
      << "\t-r <rank> column rank of approx\n"
      << "\t-s use sparse matrix implementation\n"
      << "\t-t <txx|tfn|lfn etc.>\n"
      << "\t-u <file> prefix of files (<file>.H, <file>.V) from which to init V, H\n"
      << "\t-v Be verbose, prints out useful information about progress\n"
      << "\t-w <result file> Write out final relative error onto result file\n"
      << "\t-W <weight matrix> works only algos doing elementwise weighting\n"
      << "\t-x Normalize the first factor to have unit L_2 norm columns\n"
      << "\t-z <n> Compute obj. val once every <n> steps. Default n = 1\n";
  }
}


/**
 * Based on the various initialized parameters factorize the input
 * matrix. If input matrix is sparse sA contains it else A contains it.
 */
int NmfDriver::performFactorization()
{
  DBG("NmfDriver::performFactorization() Entered\n");
  SparseMatrix* sA = 0;
  Factorization* f = 0;
  gsl_matrix* A = 0;

  // If a sparse matrix is to be factorized...
  if (opts->sparse) {
    sA = new SparseMatrix();
    
    if (opts->slowobj)
      sA->setObjFlag(true);
    else 
      sA->setObjFlag(false);

    if (sA->read_ccs_file(opts->fname, opts->txx) < 0) {
      std::cerr << "Error opening " << opts->fname << " CCS files\n";
      return -1;
    }
    opts->sA = sA;
    if (opts->rank > min(sA->numRows(), sA->numCols())) {
      std::cerr << opts->progname << ": Invalid rank specification " << opts->rank 
                << " for A_{" << sA->numRows() << " x " << sA->numCols()
                << '}' << std::endl;

      return -1;
    }
    
    switch (opts->algo) {
    case FROBENIUS: 
      f = new FactorizationSFrob(opts); 
      //new FactorizationSFrob(opts->rank, opts->maxiter, opts->epsilon, sA, opts->verbose, opts->rseed, opts->objmodulo, opts->vhprefix, opts->normalize);
      break;
    case KL:
      f = new FactorizationSKL(opts);
      //new FactorizationSKL(opts->rank, opts->maxiter, opts->epsilon, sA, opts->verbose, opts->rseed, opts->objmodulo, opts->vhprefix, opts->normalize);
      break;
    case NNLS:
      f = new FactorizationSNNLS(opts);
      //new FactorizationSNNLS(opts->rank, opts->maxiter, opts->epsilon, sA, opts->verbose, opts->rseed, opts->objmodulo, opts->vhprefix, opts->normalize);
      break;
    case HYBRID1:
      f = new FactorizationSHybrid(opts);
      // new FactorizationSHybrid(opts->rank, opts->maxiter, opts->epsilon, sA, opts->verbose, opts->rseed, opts->objmodulo, opts->vhprefix, opts->normalize);
      if (opts->nnlsiter != -1)
	((FactorizationSHybrid*)f)->setnnlsiter(opts->nnlsiter);
      break;
    case AKL:
      f = new FactorizationSAKL(opts); 
      // new FactorizationSAKL(opts->rank, opts->maxiter, opts->epsilon, sA, opts->verbose, opts->rseed, opts->objmodulo, opts->vhprefix, opts->normalize);
      break;
    case FROBSLOW:
      // f = new FactorizationSFrobSlow(opts);
      break;
    default:
      std::cerr << "Invalid algorithm specification. Not supported\n";
      return -1;
    }
  }
  else {
    if (opts->binfile) {
      A = SSUtil::fread_gsl_matrix(opts->fname);
    } else {
      A = SSUtil::read_gsl_matrix(opts->fname);
    }
    if (A == 0) return -1;
    opts->A = A;		// Important!!!!!!!!!!!!!!
    if (opts->rank > (int)min(A->size1, A->size2)) {
      std::cerr << opts->progname << ": Invalid rank specification " << opts->rank
                << " for A_{" << A->size1 << " x " << A->size2 << '}'
                << std::endl;
      return -1;
    }
    
    switch (opts->algo) {
    case FROBENIUS: 
      f = new FactorizationFrob(opts);
      // new FactorizationFrob(opts->rank, opts->maxiter, opts->epsilon, A, opts->verbose, opts->rseed, opts->objmodulo, opts->vhprefix, opts->normalize);
      break;
    case KL:
      f = new FactorizationKL(opts);
      // new FactorizationKL(opts->rank, opts->maxiter, opts->epsilon, A, opts->verbose, opts->rseed, opts->objmodulo, opts->vhprefix, opts->normalize);
      break;
    case NNLS:
      f = new FactorizationNNLS(opts);
      // new FactorizationNNLS(opts->rank, opts->maxiter, opts->epsilon, A, opts->verbose, opts->rseed, opts->objmodulo, opts->vhprefix, opts->normalize);
      break;
    case HYBRID1:
      f = new FactorizationHybrid(opts);
      // new FactorizationHybrid(opts->rank, opts->maxiter, opts->epsilon, A, opts->verbose, opts->rseed, opts->objmodulo, opts->vhprefix, opts->normalize);
      if (opts->nnlsiter != -1)
        ((FactorizationHybrid*)f)->setnnlsiter(opts->nnlsiter);
      // else nnlsiter will have a good value due to constructor above.
      break;
    case AKL:
      f = new FactorizationAKL(opts);
      // new FactorizationAKL(opts->rank, opts->maxiter, opts->epsilon, A, opts->verbose, opts->rseed, opts->objmodulo, opts->vhprefix, opts->normalize);
      break;
    case FROBSLOW:
      f = new FactorizationFrobSlow(opts);
      break;
    case HYBRID2:
      f = new FactorizationHybrid2(opts);
      break;
    case FROBWT1:
      f = new FactorizationFrobWt1(opts);
      break;
    case FROBWT2:
      f = new FactorizationFrobWt2(opts);
      break;
    case BREGA_B:
      f = new FactorizationBregTest1(opts);
      break;
	case BREGB_A:
	  f = new FactorizationBreg2Test1(opts);
	  break;
    default:
      std::cerr << "Invalid algorithm specification" << std::endl;
      return -1;
    }
  }
  
  DBG("Now about to do SVD if needed\n");
  // See what would best rank k approx yield...
  // Use LAPACK svd computing routine
  if (opts->dosvd and !opts->sparse and !opts->svdfile) {
    char jobu;
    char jobvt;
    integer M = A->size1;
    integer N = A->size2;
    std::cerr << M << " x " << N << std::endl;
    //gsl_matrix_view mt = gsl_view_array(A->data, M, N);
    integer LDA = M;
    integer LDU = M;
    integer LDVT = N;
    integer LWORK;
    integer INFO;
    int maxrank = min(A->size1, A->size2);
    double* s = new double[maxrank];
    double* wk = new double[maxrank*10]; 
    double* uu = new double[2];	// dummy
    double* vt = new double[2];	// dummy
    
    // We want only the singular values so set job for both U and V' to be 'N'
    jobu = 'N';
    jobvt = 'N';
    LWORK =  maxrank*10;
    //double fnormA = f->fnorm(A); // A gets changed by dgesvd i think
    gsl_matrix* copyA = SSUtil::prepare_for_clapack(A);
    dgesvd_( &jobu, &jobvt, &M, &N, copyA->data, &LDA, s, uu, 
	     &LDU, vt, &LDVT, wk, &LWORK, &INFO);

    double best = 0.0;
    
    for (int i = 0; i < opts->rank; i++) //i < (int)min(A->size1, A->size2); i++)
      best += (s[i] * s[i]);

    double fnormA = SSUtil::fnorm(A);
    double svderror = sqrt(fabs(fnormA*fnormA - best))/fnormA ;

    std::cerr << "NmfDriver::performFactorization():   Norm^2 of A_" 
	      << opts->rank
	      << " := " << best 
	      << "\nNmfDriver::performFactorization(): Norm^2 of A := " 
	      << fnormA*fnormA
	      << "\nSVD Approx would have relative error: ==> "
	      << svderror
	      << std::endl;
    f->set_svderror(svderror);
  } 
  
  // Use libmyssvd to compute svd approximation for sparse matrix.
  if (opts->dosvd && opts->sparse and !opts->svdfile) {
    std::cerr << "Going to do SVD Now\n";
    //double fnormA = sA->fnorm();
    ssvd::OptionsStruct svdopts;
    strcpy(svdopts.prefix, opts->fname);
    strcpy(svdopts.txx, opts->txx);
    svdopts.lanmax = sA->numCols();
    svdopts.endl = -1e-30;
    svdopts.endr = 1e-30;
    svdopts.kappa = 1e-6;
    svdopts.in1 = 0;
    svdopts.dofiles = 1;
    svdopts.vectors = false;
    sprintf(svdopts.out1, "%s.svd", opts->fname);
    svdopts.nums = opts->rank+1;
    svdopts.sA   = sA;
    ssvd::Driver* d = new ssvd::Driver(3, &svdopts);
    d->executeWithOptions();
    double *svals = d->getSing();
    for (int i = 0; i < opts->rank; i++)
      std::cout << svals[i] << " ";
    std::cout << "\n";
    double best = 0.0;
    for (int i = 0; i < opts->rank; i++) {
      best += (svals[i]*svals[i]);
      //cerr << svals[i] << std::endl;
    }
    double fnormA = sA->fnorm();
    double svderror = sqrt(fabs(fnormA*fnormA - best))/fnormA ;
    std::cerr << "NmfDriver::performFactorization(): Norm^2 of A_" 
	      << opts->rank
	      << " := " << best 
	      << "\nNmfDriver::performFactorization(): Norm^2 of A := " 
	      << fnormA*fnormA
	      << "\nNmfDriver::performFactorization(): Relative error of rank " 
	      << opts->rank << " SVD Approx: = " 
	      << svderror
	      << std::endl;
    f->set_svderror(svderror);	// Needed for writeResults
  }
  
  if (opts->svdfile) { // Means svals are already avail, no need
    // to perform SVD in here....
    
    FILE* fp = fopen(opts->svdfile, "r");
    if (!fp) {
      std::cerr << "Could not open file " << opts->svdfile << " for reading svals\n";
      return -1;
    }
    int nk;
    fscanf(fp, "%d", &nk);
    double* svals = new double[nk];
    for (int i = 0; i < opts->rank; i++)
      fscanf(fp, "%lf", &svals[i]);
    double best = 0.0;

    for (int i = 0; i < opts->rank; i++) {
      best += (svals[i]*svals[i]);
      //cerr << svals[i] << std::endl;
    }

    double fnormA;
    if (opts->sparse) 
      fnormA = sA->fnorm();
    else
      fnormA = SSUtil::fnorm(A);

    double svderror = sqrt(fabs(fnormA*fnormA - best))/fnormA ;
    std::cerr << "NmfDriver::performFactorization(): Norm^2 of A_" 
	      << opts->rank
	      << " := " << best 
	      << "\nNmfDriver::performFactorization(): Norm^2 of A := " 
	      << fnormA*fnormA
	      << "\nNmfDriver::performFactorization(): Relative error of rank " 
	      << opts->rank << " SVD Approx: = " 
	      << svderror
	      << std::endl;
    f->set_svderror(svderror);	// Needed for writeResults
  }

  DBG("Now to either perform Approx. or just compute OBJ\n");
  if (opts->objonly) {
    double o = f->compute_objonly(opts->objvhprefix);
    double relerror = o / (opts->sparse ? sA->fnorm() : SSUtil::fnorm(A));
    f->set_relerror(relerror);
    switch (opts->algo) {
    case FROBENIUS: case NNLS: case HYBRID1:
      std::cerr << "Relative error for given V, H is ==> "  << relerror << std::endl;
      break;
    default:
      ;
    }
  }
  else {  // otherwise we perform an approximation.
    int result = f->perform();
    if (result == 0) {
      char file[1024];
      switch (opts->algo) {
      case FROBENIUS:
		sprintf(file, "%s%s%d%s", opts->fname, ".fact.", opts->rank, ".frob");
		break;
      case KL:
		sprintf(file, "%s%s%d%s", opts->fname, ".fact.", opts->rank, ".KL");
		break;
      case NNLS:
		sprintf(file, "%s%s%d%s", opts->fname, ".fact.", opts->rank, ".nnls");
		break;
      case HYBRID1:
		sprintf(file, "%s%s%d%s", opts->fname, ".fact.", opts->rank, ".hybrid1");
		break;
      case AKL:
		sprintf(file, "%s%s%d%s", opts->fname, ".fact.", opts->rank, ".akl");
		break;
      case FROBSLOW:
		sprintf(file, "%s%s%d%s", opts->fname, ".fact.", opts->rank, ".frob2");
		break;
      case HYBRID2:
		sprintf(file, "%s%s%d%s", opts->fname, ".fact.", opts->rank, ".hybrid2");
		break;
      case FROBWT1:
		sprintf(file, "%s%s%d%s", opts->fname, ".fact.", opts->rank, ".frobwt1");
		break;
      case FROBWT2:
		sprintf(file, "%s%s%d%s", opts->fname, ".fact.", opts->rank, ".frobwt2");
		break;
      case BREGA_B:
		sprintf(file, "%s%s%d%s", opts->fname, ".fact.", opts->rank, ".breg");
		break;
	  case BREGB_A:
		sprintf(file, "%s%s%d%s", opts->fname, ".fact.", opts->rank, ".breg2");
		break;
      default:
        file[0] = '\0';
		break;
      }
      f->writeResults(file, opts->binfile);
    }
    else {
      std::cerr << opts->progname << ": Could not perform factorization" << std::endl;
    }
  }

  return 0;
}
