// File: DriverOpts.h -*- c++ -*-
// Author: Suvrit Sra
// Date: 01 Dec 03

// Options structure for the driver, in case it needs to be called
// standalone on an already available matrix ....

#ifndef _DRIVEROPTS_H
#define _DRIVEROPTS_H

#include <ctime>
#include <cstring>

#include <ostream>

#include <gsl/gsl_matrix.h>
#include "SparseMatrix.h"

/*
 * This is the driver structure. I have tried to factor out stuff common to
 * the Various NMF implementations into this structure. This structure
 * contains fields that are based upon a knowledge of the internals of the
 * various NMF routines. That is necessary because this is for a driver of
 * those routines, and a driver needs to know the internals ... unless each
 * class had some more detailed interface for its execution ....
 *
 * @Author: Suvrit Sra
 * @Date:  01 Dec 2003
 */
struct DriverOpts {

  /* Various default constant values */
  static const  double MINEPS   = 0.001f;
  static const  int MAXITER     = 300;

  char*  progname;
  char*  fname;
  char   txx[6];
  char*  resfile;

  gsl_matrix* A;
  SparseMatrix* sA;

  int    maxiter;
  unsigned long int rseed;
  int    algo;
  int    rank;
  int    objmodulo;
  int    nnlsiter;
  int    level;
  char*  vhprefix;
  char*  objvhprefix;
  double epsilon;

  char* algoname;
  char* svdfile;
  char* wt1file;
  char* wtLfile;
  char* wtRfile;
  bool dosvd;   
  bool sparse;  
  bool cmdok;   
  bool verbose; 
  bool objonly;
  bool slowobj;
  bool binfile;
  bool normalize;
  bool perturb;

  DriverOpts() {
    progname = 0;
    vhprefix = 0;
    A = 0;
    sA = 0;
    fname = 0;
    svdfile = 0;
    strcpy(txx, "txx");
    algo = 0;
    rank = objmodulo = 1;
    rseed = time(0);    maxiter = MAXITER;
    epsilon = MINEPS; level = 1; nnlsiter = 2;
    dosvd = sparse = cmdok = verbose = false;
    perturb = false;
    wt1file = 0;
    wtLfile = wtRfile = 0;
    normalize = objonly = slowobj = binfile = false;
  }

};

std::ostream& operator << (std::ostream& os, DriverOpts& o);

/*
 * The algorithms supported (invocable by the driver routine)
 * FROBENIUS: Lee/Seung Frobenius updates, blas3 style
 * KL       : Lee/Seung Generalized KL updates
 * NNLS     : ANNLS algorithm using lawson-hanson NNLS code
 * AKL      : Alternating KL algorithm similar to ANNLS 
 * FROBSLOW : Blas2 style frob algo (updates row by row etc.)
 * HYBRID1  : Hybrid of Lee/Seung Frob and ANNLS
 * HYBRID2  : Hybrid of ANNLS and FROBSLOW
 * FROBWT1  : Weighted Frob. norm (Lee/S style upates) - Elementwise
 * FROBWT2  : minimizes ||LAR - LVHR|| for inputs L, A, R
 * FROBWT3  : minimizes Tr( (A-VH)'K(A-VH) ) where K is p.d. matrix
 * KLWT1    : minimizes I-div( W . A | W. VH ) (elementwise)
 * KLWT2    : minimizes I-div( LAR | LVHR ), 
 * BREGA_B  : minimizes D(A; VH)
 * BREGB_A  : minimizes D(VH; A)
 */
enum algoType { FROBENIUS,
		KL, NNLS, HYBRID1, 
		AKL, FROBSLOW, HYBRID2, 
		FROBWT1, FROBWT2, FROBWT3,
		KLWT1, KLWT2, BREGA_B, BREGB_A
};
#endif
