/* ************************************************************
 * File: SparseMatrix.cc
 * Author: Suvrit Sra
 * Revision History:
 * 02/25/03 Created (Imported from smatrix.cc)
 * ***********************************************************/

// SparseMatrix.cc -- Class for sparse matrices....this is a little dirty
// class with some hacks thrown in for efficiency....of course that makes
// it sorta bad to read but....
// Copyright (C) 2003 Suvrit Sra (suvrit@cs.utexas.edu)

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#include "SparseMatrix.h"
#include <gsl/gsl_blas.h>
#include <gsl/gsl_vector.h>
#include <math.h>

/**
 * A hack necessitated by my unwillingness to immed. implement my 'apply'
 * idea to the SparseMatrix class.
 */

double SparseMatrix::compute_obj(gsl_matrix* P, tObjtype obj)
{
  double objval = 0.0;

  switch(obj) {
  case OBJ_FROB:
    { 
      if (first_time) {
	/*
	 * Compute || A - P || in a smart fashion, exploiting the sparsity
	 * of A. Since A is sparse, \sum (a_ij - p_ij)^2 can be divided
	 * into \sum_NZ(A) a_ij(a_ij - 2p_ij) + fnorm(P)^2. Both these can
	 * be done sufficiently fast...The time should ideally be dominated
	 * by fnorm(P)^2 since P is dense and HUGE. Still the
	 * implementation below is a significant improvement over the
	 * trivial approach.
	 */
	///DBG("SparseMatrix::compute_obj() Doing first time special case\n");
	//first_time = false;
	for (int j = 0; j < numCols(); j++) {
	  for (int i = m_colptrs[j]; i < m_colptrs[j+1]; i++) {
	    double t = m_val[i];
	    double pt = gsl_matrix_get(P, m_rowindx[i], j);
	    objval += (t * (t - 2*pt));
	  }
	}
	//cerr << "OBJ Temp = " << objval << endl;
	gsl_vector_view v = gsl_vector_view_array(P->data, P->size1*P->size2);
	double bn =  gsl_blas_dnrm2(&v.vector);
	//cerr << "Norm of P = " << bn << endl;
	objval += bn*bn;
	objval = sqrt(objval);
      } else {
	for (int j = 0; j < numCols(); j++)
	  for (int i = m_colptrs[j]; i < m_colptrs[j+1]; i++) {
	    double t = m_val[i];
	    double pt = gsl_matrix_get(P, m_rowindx[i], j);
	    objval += ((t - pt)*(t-pt));
	  }
	objval = sqrt(objval);
      }
    }
    break;
  case OBJ_KL:
    {
      for (int j = 0; j < numCols(); j++) {
	for (int i = m_colptrs[j]; i < m_colptrs[j+1]; i++) {
	  double t = m_val[i];
	  double pt = gsl_matrix_get(P, m_rowindx[i], j);
	  // ideally we should not be hitting any such t's in a
	  // sparsematrix since itz supposed to have only nonzeros.
	  if (t == 0) continue;
	  if (pt == 0) { std::cerr << "Badness\n"; pt = .001;}
	  objval += t * log(t/pt) - t + pt;
	  //objval += t * log(pt) - pt;
	}
      }
    }
    break;
  default:
    std::cerr << "Invalid objective requested for computation" << std::endl;
    break;
  }

  return objval;
}

/**
 * this = l ./ r
 */
void SparseMatrix::dotdiv(SparseMatrix* l, SparseMatrix* r)
{
  if (l->m_nz != r->m_nz or l->m_nz != m_nz) {
    std::cerr << "Incompatible matrices for dot div\n";
    return;
  }

  // A potential divide by zero -- not checked here.
  for (int i = 0; i < m_nz; i++)
    m_val[i] = l->m_val[i] / r->m_val[i];
}

/**
 * Computes: this ./ X and stores into res.
 * This works easily because res will have zeroes exactly where 'this' has
 * zeroes ... hence we can't destroy sparsity.
 */
void SparseMatrix::dotdiv(gsl_matrix* X, SparseMatrix* res)
{
  for (int c = 0; c < numCols(); c++) {
    for (int j = m_colptrs[c]; j < m_colptrs[c+1]; j++) {
      double dr = gsl_matrix_get(X, m_rowindx[j], c);
      if (dr == 0) std::cerr << "0 in dotdiv";
      res->set(m_rowindx[j], c, m_val[j]/dr);
    }
  }
}

/**
 * Perform A * B where A is the sparse matrix and B is a regular dense
 * matrix. We use the traditional outproduct organization of the MM loop.
 * 
 * Input: Matrix B, Allocated storage space for result C
 *        tA and tB that allow computation of A'B, AB' etc prods.
 * Output: Modified C = A*B (not doing A*B + C)
 * TBD: Implement the transposition flags, remove calloc dep.
 *
 */
void SparseMatrix::lmult(gsl_matrix* B, gsl_matrix* C, bool tA, bool tB)
{
  gsl_vector_view c;
  // Make sure proper init of C happens?
  if (!tA and !tB) { // Compute A*B
    gsl_matrix_set_zero(C);	// Needed as of now
    for (int k = 0; k < numCols(); k++) {
      for (uint j = 0; j < B->size2; j++) {
	c = gsl_matrix_column(C, j);
	saxpy(gsl_matrix_get(B, k, j), k, &c.vector);
      }
    }
  } else if (tA and !tB) { // Compute A'B
    gsl_vector_view colb;
    // A' is of dim numCols x numRows
    // C  is of dim numCols x B->size2
    for (int i = 0; i < numCols(); i++) {
      c = gsl_matrix_row(C, i);
      // Now we compute one row of C
      for (uint j = 0;  j < B->size2; j++) {
	colb = gsl_matrix_column(B, j);
	// c[1..m] = col i dot prod colb[1..m]
	c.vector.data[j] = dot(i, &colb.vector);
      }
    }
  } else if (!tA and tB) { // Compute AB'
    // THis could be a little slow I think.
    gsl_matrix_set_zero(C);
    for (int k = 0; k < numCols(); k++) {
      for (uint j = 0; j < B->size1; j++) {
	c = gsl_matrix_column(C, j);
	saxpy(gsl_matrix_get(B, j, k), k, &c.vector);
      }
    }
  } else { // Compute A'B'
    gsl_vector_view colb;
    for (int i = 0; i < numCols(); i++) {
      c = gsl_matrix_row(C,i);
      // Now compute row i of C
      for (uint j = 0; j < B->size1; j++) {
	colb = gsl_matrix_row(B, j);
	c.vector.data[j] = dot(i, &colb.vector);
      }
    }
  }
}

// Adds onto v given as input so make sure v is in good shape b4 callin
void SparseMatrix::saxpy(double alpha, int col, gsl_vector* v)
{
  double datum;
  for (int i = m_colptrs[col]; i < m_colptrs[col+1]; i++) {
    datum = gsl_vector_get(v, m_rowindx[i]);
    datum += alpha*m_val[i];
    gsl_vector_set(v, m_rowindx[i], datum); 
  }
}

/** 
 * We do B * this where lop is dense and 'this' is sparse. Look at B as    
 * [b1' b2' ... bm']' and 'this' as [a1 a2 ... an]
 * We assume C already has storage allocated to it.
 * Input: B, C as above
 *        Flags tA, tB that control B*A, B'A, B'A' or BA'
 * Output: C = tB(B) * tA(A) the product as desired
 * TBD: Implement the transposition flags
 */
void SparseMatrix::rmult(gsl_matrix* B, gsl_matrix* C, bool tA, bool tB)
{
  gsl_vector_view c;
  gsl_vector_view b;

  if (!tA and !tB) { // BA
    for (uint i = 0; i < B->size1; i++) {
      c = gsl_matrix_row(C, i);
      b = gsl_matrix_row(B, i);
      for (int j = 0; j < numCols(); j++) {
	c.vector.data[j] = dot(j, &b.vector);
      }
    }
  } else if (!tA and tB) { //B'A
    //cerr << "Doing B' A " << endl;
    for (uint i = 0; i < B->size2; i++) {
      // Form first row of C
      c = gsl_matrix_row(C, i);
      b = gsl_matrix_column(B, i);
      for (int j = 0; j < numCols(); j++) {
	c.vector.data[j] = dot(j, &b.vector);
      }
    }
  } else if (tA and !tB) { //BA'
    std::cerr << "ERRRRRRRRRRRRRRRRR\n";
  } else { // B'A'
    std::cerr << "ERRRRRRRRRRRRRRRRR\n";
  }
}


/**
 * Perform dot product of column 'col' of matrix with vector 'v'
 * return: dot product value
 */
double SparseMatrix::dot(int col, gsl_vector* v)
{
  double dp = 0.0;
  for (int i = m_colptrs[col]; i < m_colptrs[col+1]; i++) {
    //cerr << m_val[i] << "@" << m_rowindx[i] << " ";
    dp += m_val[i] * gsl_vector_get(v, m_rowindx[i]);;
    //cerr << "b" << gsl_vector_get(v, m_rowindx[i]) << "@" << m_rowindx[i] << " ";
  }
  return dp;
}

SparseMatrix* SparseMatrix::clone() 
{
  std::cerr << "SparseMatrix::clone()\n";
  //cerr << "(r, n, nz) = " << m_rows << " " << m_cols << " "<< m_nz << std::endl;
  SparseMatrix* m = new SparseMatrix(m_rows, m_cols, m_nz);
  
  // NOTE: m_cols+1 is impt. below!
  for (int i = 0; i < m_cols+1; i++)
    m->m_colptrs[i] = m_colptrs[i];

  for (int i = 0; i < m_nz; i++) {
    m->m_rowindx[i] = m_rowindx[i];
    m->m_val[i] = m_val[i];
  }
  return m;
}
// Copies sparse matrix onto A .. copies only nz's
void SparseMatrix::makefull(gsl_matrix* A)
{
  for (int i = 0; i < numCols(); i++)
    for (int j = m_colptrs[i]; j < m_colptrs[i+1]; j++)
      gsl_matrix_set(A, m_rowindx[j], i, m_val[j]);
}


void SparseMatrix::compute_fnorm()
{
  double no = 0;
  for (int i = 0; i < m_nz; i++)
    no += m_val[i]*m_val[i];
  
  fnorm_avail = true;
  fnormA = sqrt(no);
}

double SparseMatrix::col_norm(int i)
{
  //assert((i >= 0 and  i < m_cols));
  
  double n = 0;
  
  for (int j = m_colptrs[i]; j < m_colptrs[i+1]; j++) {
    n += m_val[j]*m_val[j];
  }
  
  return sqrt(n);
}


double SparseMatrix::col_diff(int i, int j)
{
  double dif = 0;

  gsl_vector* j_th_col = gsl_vector_alloc(m_rows);
  for (int c = m_colptrs[j]; c < m_colptrs[j+1]; c++)
    j_th_col->data[m_rowindx[c]] = m_val[c];
  
  for (int c = m_colptrs[i]; c < m_colptrs[i+1]; c++) {
    dif += (m_val[c]-j_th_col->data[m_rowindx[c]])*
      (m_val[c]-j_th_col->data[m_rowindx[c]]); 
  }
  
  gsl_vector_free(j_th_col);
  return dif;
}


double SparseMatrix::col_delta(int i, double* v)
{
  double dif = 0;
  for (int c = m_colptrs[i]; c < m_colptrs[i+1]; c++) {
    dif += (m_val[c]-v[m_rowindx[c]])*
      (m_val[c]-v[m_rowindx[c]]); 
  }
  
  return dif;
}


// This returns (d_i^T d_j)/norm(d_i).norm(d_j)
double SparseMatrix::col_cosine(int i, int j)
{
  //assert (i >= 0 and i < m_cols);
  //assert (j >= 0 and j < m_cols);

  if (i == j) return 1;
  
  double dp = 0;

  // Here we build the full vector for the jth col...to simplify
  // my program.
  gsl_vector* j_th_col = gsl_vector_alloc(m_rows);
  
  for (int c = m_colptrs[j]; c < m_colptrs[j+1]; c++)
    j_th_col->data[m_rowindx[c]] = m_val[c];
  
  for (int c = m_colptrs[i]; c < m_colptrs[i+1]; c++) { 
    dp += m_val[c]*j_th_col->data[m_rowindx[c]];
  }
  
  gsl_vector_free(j_th_col);
  return dp;
}

// Fill column vector
void SparseMatrix::getcol(int i, double* v)
{
  for (int j = m_colptrs[i]; j < m_colptrs[i+1]; j++)
    v[m_rowindx[j]] = m_val[j];
}

// Col cosine with input vector --
// TBD: METHOD YET TO BE IMPLEMENTED...
double SparseMatrix::col_cosine2(int i, double* v)
{
  /*if (i < 0 || i > m_cols) 
    throw MatrixExceptionRange(i);
  
  if (m_rows != v.size())
    throw MatrixExceptionRange(v.size());
  */
  double dp = 0;

  for (int c = m_colptrs[i]; c < m_colptrs[i+1]; c++) { 
    dp += m_val[c]*v[m_rowindx[c]];
  }

  double np = 0; // gsl_norm(v);
  return dp/np;

}

// Print out the CCS Matrix in a format suitalbe for use with matlab.

void SparseMatrix::output_matlab(std::ofstream& file, float scaling)
{
  for (int i = 0; i < m_cols-1; i++) {
    for (int j = m_colptrs[i]; j < m_colptrs[i+1]; j++) {
      file << m_rowindx[j]+1 << " " << i+1 << " " << m_val[j] << std::endl;
    }
  }
}

// This function does a dot product of column i with column j.
// Right now as a temporary hack it does not do dot product of a
// column with itself. As I need this for the docdot.cc program.

double SparseMatrix::col_dotprod(int i, int j)
{
  // if (i < 0 || i > m_cols) 
//     throw MatrixExceptionRange(i);
  
//   if (j < 0 || j > m_cols)
//     throw MatrixExceptionRange(j);
  
  if (i == j) return 0;
  
  double dp = 0;

  // Here we build the full vector for the jth col...to simplify
  // my program.
  gsl_vector* j_th_col = gsl_vector_alloc(m_rows);
  //for (int r = 0; r < m_rows; r++)
  ///	  j_th_col[r] = 0;
  
  for (int c = m_colptrs[j]; c < m_colptrs[j+1]; c++)
    j_th_col->data[m_rowindx[c]] = m_val[c];
  for (int c = m_colptrs[i], l=0 ; c < m_colptrs[i+1]; c++, l++) {
    dp += m_val[c]*j_th_col->data[m_rowindx[c]];
  }
  gsl_vector_free(j_th_col);
  return dp;
     
}


// ------------------------------------------------------------------
// Dot product operator.
// returns y = A.x where y is a vector<T>
// A is the sparse matrix stored in this object
// ------------------------------------------------------------------
gsl_vector* SparseMatrix::operator * (const gsl_vector* x)
{
  // Dimension check
  if (m_cols != (int)x->size)
    return 0;
     
  gsl_vector* y = gsl_vector_calloc(m_rows);
     
  for (int i = 0; i < m_cols; i++) {
    for (int j = m_colptrs[i]; j < m_colptrs[i+1]; j++)
      y->data[m_rowindx[j]] += m_val[j]*x->data[i];
  }
  
  return y;
  
}

// ------------------------------------------------------------------
// Another dot product
// returns y = A^T . x as a vector<T>
// A is the sparse matrix stored in this object
// ------------------------------------------------------------------
gsl_vector* SparseMatrix::tran_times(const gsl_vector* x)
{
  // Dimension check.
  if (m_rows != (int)x->size)
    return 0;
     
  gsl_vector*  y = gsl_vector_alloc(m_cols);
     
  double yi;
  for (int i = 0; i < m_cols; i++) {
    yi = 0;
    for (int j = m_colptrs[i]; j < m_colptrs[i+1]; j++) {
      yi += m_val[j] * x->data[ m_rowindx[j] ];
    }
    y->data[i] = yi;
  }
  return y;
}

// This function writes out the matrix in matlab sparse matrix format.
// basically (i,j) val triples for non-zeros.

void SparseMatrix::print(std::ostream& os)
{
  for (int i = 0; i < numRows(); i++) {
    for (int j = 0; j < numCols(); j++) {
      double x = dataAt(i,j);
      if (x != 0) {
	os << i << " " << j << " " << x << std::endl;
      }
    }
  }
}

void SparseMatrix::printmat(std::ostream& os)
{
  std::cerr << numRows() << " x " << numCols() << std::endl;
  for (int i = 0; i < numRows(); i++) {
    for (int j = 0; j < numCols(); j++)
      os << dataAt(i,j) << " ";
    os << std::endl;
  }
}

int SparseMatrix::read_ccs_file(char* fname, char* txx)
{
  std::cerr << "Building sparsematrix from " << fname << " (" << txx << ") scaling\n";
  std::string scaling(txx);

  // Set up the file names that we are gonna open
  std::string dim(fname);
  std::string rows_file(dim + "_row_ccs");
  std::string cols_file(dim + "_col_ccs");
  std::string nz_file (dim + "_"+ scaling + "_nz");
  
  dim +=  "_dim";
  
  std::ifstream infile;
  infile.open(dim.data());
  if (infile.fail()) {
    std::cerr << "Error: could not open " << dim << std::endl;
    return -1;
  }
  
  infile >> m_rows >> m_cols >> m_nz;
  infile.close();
  infile.clear();
  
  //std::cerr << "(m, n, nz)" << m_rows << " " << m_cols << " " << m_nz << std::endl;
  m_colptrs  = new long[m_cols+1];
  m_rowindx  = new long[m_nz];
  m_val   = new double[m_nz];
  m_colptrs[m_cols] = m_nz; // Add the convenience value
  
  // Now read in all the columnpointers
  infile.open(cols_file.data());
  if (infile.fail()) {
    std::cerr << "Error: could not open " << cols_file << std::endl;
    return -1;
  }
  // Make sure the UB on i is correct.
  for (int i = 0; i < m_cols+1; i++) 
    infile >> m_colptrs[i];
  infile.close();
  infile.clear();
  
  infile.open(rows_file.data());
  if (infile.fail()) {
    std::cerr << "Error: could not open " << rows_file << std::endl;
    return -1;
  }

  for (int i = 0; i < m_nz; i++)
    infile >> m_rowindx[i];
  infile.close();
  infile.clear();
  
    // Now read in the actual m_values
  infile.open(nz_file.data());
  if (infile.fail()) {
    std::cerr << "Error: could not open " << nz_file << std::endl;
    return -1;
  }
  for (int i = 0; i < m_nz; i++)
    infile >> m_val[i];
  infile.close();

  return 0;
}


void SparseMatrix::debuginfo()
{
  std::cerr << "(m, n, nz) == " 
       << numRows() << " " 
       << numCols() << " " 
       << numNz() << std::endl; 

  for (int j = 0; j < numCols(); j++) {
    for (int i = m_colptrs[j]; i < m_colptrs[j+1]; i++)
      std::cerr << m_val[i] << " ";
  }
  std::cerr << std::endl;
  for (int i = 0; i < numNz(); i++)
    std::cerr << m_rowindx[i] << " ";
}
