// FactorizationSAKL.cc -- implements sparse AKL algo
// Copyright (C) 2003 Suvrit Sra (suvrit@cs.utexas.edu)

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


#include <iostream>
#include <gsl/gsl_blas.h>
#include <math.h>

#include "FactorizationSAKL.h"
#include "objective.h"


int FactorizationSAKL::perform()
{
  if (!checkRank()) {
    std::cerr << "FactorizationSAKL::perform() Invalid rank: " 
	 << getRank() << std::endl;
    return -1;
  }
  
  DBG("FactorizationSAKL::perform()\n");
  DBG("UNDER CONSTRUCTION\n");

  char *vhprefix = get_vh_file();
  if (vhprefix)
    init_matrices(Factorization::FRMFILE);
  else 
    init_matrices(Factorization::RANDOM);

  //init_matrices(Factorization::RANDOM);
  gsl_matrix* P = gsl_matrix_alloc(V->size1, H->size2);
  gsl_blas_dgemm(CblasNoTrans, CblasNoTrans, 1.0, V, H, 0.0, P);
  
  objval = sA->compute_obj(P, OBJ_KL); // A call to compute obj is impt.

  double objold = objval + 100;
  double error = 1;
  int    iter = 0;

  //DBG("Init obj = " << objval << endl);
  while (error > epsilon() && iter < getMaxiter()) {
    if (iter % objmodulo == 0)
      DBG("AKL: " << iter << " Obj := " << objval << std::endl);

    if (objval > objold) {
      DBG("SAKL: OBJVAL increased at iteration ==> " << iter << std::endl);
    }
   
    // Fix V and compute H so that we min D(A || VH)
    //myskl(A, V, H, arg1);

    // Fix H and compute V so that we min D(A || VH)
    // myskl(A, V, H, arg2)
    
    if (iter % objmodulo == 0) {
      objold  = objval;
      objval  = sA->compute_obj(P, OBJ_KL);
      error   = fabs(objval - objold);
    }
    ++iter;
  }
  return 0;
}


/**
 * Just compute the KL objective function and return it. It initializes V, H matrices from filename
 * prefix given by vhprefix
 */
double FactorizationSAKL::compute_objonly(char* vhprefix)
{
  set_vh_file(vhprefix);
  init_file();
  gsl_matrix* P = gsl_matrix_alloc(V->size1, H->size2);
  gsl_blas_dgemm(CblasNoTrans, CblasNoTrans, 1.0, V, H, 0.0, P);
  double val = sA->compute_obj(P, OBJ_KL);
  std::cerr << "FactorizationSAKL::objval ==> " << val << std::endl;
  gsl_matrix_free(P);
  return val;
}
