// FactorizationNNLSnew.cc
// Copyright (C) 2003 Suvrit Sra (suvrit@cs.utexas.edu)

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#include "nnls.h"
#include "FactorizationNNLS.h"
#include <fstream>

int FactorizationNNLS::perform()
{

  char *vhprefix = get_vh_file();
  if (vhprefix)
    init_matrices(Factorization::FRMFILE);
  else 
    init_matrices(Factorization::RANDOM);

  // Initialize V, H and compute init objective value
  //init_matrices(Factorization::RANDOM);
  gsl_matrix* P = gsl_matrix_alloc(V->size1, H->size2);
  gsl_blas_dgemm(CblasNoTrans, CblasNoTrans, 1.0, V, H, 0.0, P);
  objval = compute_obj(P);

  double objold = objval + 100;

  int m = A->size1;
  int r = getRank();
  int n = A->size2;
  int iter = 1;			// Iteration number
  double error = objold - objval;
  int* flg = new int[r];

  int** P1;
  int** P2;
  int** Z1;
  int** Z2;

  /** **************************************************
   *  INIT INDEX SETS FOR VH - A PART
   *  **************************************************/
  P1 = new int*[r];	
  for (int i = 0; i < r; i++) {
    P1[i] = new int[n];
    memset(P1[i], -1, n*sizeof(int));	// P = NULL on step 1 of algo
  }
  
  // r zero index sets
  Z1 = new int*[r];	
  for (int i = 0; i < r; i++) {
    Z1[i] = new int[n];
  }

  // Init. the index set of zeros
  for (int i = 0; i < n; i++) {
    for (int j = 0; j < r; j++)
      Z1[j][i] = i;
  }

  /** **************************************************
   *  INIT INDEX SETS FOR H'V' - A' PART
   *  **************************************************/
  while (error > epsilon() && iter < getMaxiter()) {
    DBG("NNLS2: " << iter << ": " << objval << endl);
    // Fix V and compute H so that we min ||VH - A||_F
    // Solves VH approx eq A and sets the 'r' flags
    mynnls(V, H, A, P1, Z1, flg);		

    // Tranpose H, V, A for computation (i.e. for ||H'V' - A'|| )
    mynnls(H, V, A, P2, Z2, flg, true);
    gsl_blas_dgemm(CblasNoTrans, CblasNoTrans, 1.0, V, H, 0.0, P);
    objold = objval;
    objval = compute_obj(P);
    error = fabs(objval - objold);
    iter++;
  }
  relerror = objva/fnorm(A);
  cerr << "NNLS: Complete. Relative error ==> " << objval/fnorm(A) << endl;
  return 0;
}

double FactorizationNNLS::compute_obj(const gsl_matrix* prod)
{
  double obj = 0.0;
  for (uint i = 0; i < A->size1 * A->size2; i++) {
    double diff = A->data[i] - prod->data[i];
    obj += (diff*diff);
  }
  return sqrt(obj);
}
