// File: Factorization.cc -- Implements base class of all Factorizations. 
// Copyright (C) 2003 Suvrit Sra (suvrit@cs.utexas.edu)

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#include <iostream>
#include <fstream>
#include "Factorization.h"
#include <math.h>
#include "util.h"


void Factorization::normalizeVH()
{
  for (uint j = 0; j != H->size1; j++) {
    gsl_vector_view column = gsl_matrix_column (V, j);
    double d;
    d = gsl_blas_dnrm2 (&column.vector);

    for (uint i = 0; i != H->size2; i++) {
      gsl_matrix_set(H, j, i, gsl_matrix_get(H, j, i)*d);
    }
  }
 
  for (uint j = 0; j != V->size2; ++j) {
    gsl_vector_view column = gsl_matrix_column (V, j);
    double d;
    d = gsl_blas_dnrm2 (&column.vector);

    for (uint i = 0; i != V->size1; ++i) {
      gsl_matrix_set(V, i, j, gsl_matrix_get(V, i, j)/d);
    }
  }
}


void Factorization::printmat(gsl_matrix* g)
{
  std::cout << g->size1 << " x " << g->size2 << std::endl;
  for (uint i = 0; i < g->size1; i++) {
    for (uint j = 0; j < g->size2; j++)
      std::cout << gsl_matrix_get(g, i, j) << " ";
    std::cout << std::endl;
  }
}

// Writes out result of factorization to disk.
void Factorization::writeResults(char* fname, bool bin) 
{
  std::cerr << "in Factorization::writeResults()\n";
  std::string v(fname);
  v += ".V";
  std::string h(fname);
  h += ".H";

  std::ofstream res(opts->resfile, std::ios::app);
  if (res.fail()) {
    res.clear();
    res.open(opts->resfile);	// Try for writing...
    if (res.fail()) {
      std::cerr << "Factorization::writeResults(), error writing results to **" << opts->resfile << "**\n";
      return;
    }
  }
  res << opts->algo 
      << " " << rank
      << " " << relerror
      << " " << svderror 
      << std::endl;
  res.close();

  if (bin) {
    if (SSUtil::fwrite_gsl_matrix(V, v.data()) < 0) {
      std::cerr << "Factorization::writeResults(), file error\n";
      return;
    }
    if (SSUtil::fwrite_gsl_matrix(H, h.data()) < 0) {
      std::cerr << "Factorization::writeResults(), file error\n";
      return;
    }
  } else {
    std::ofstream ofile(v.data());
    
    if (ofile.fail()) {
      std::cerr << "Could not open " << v << "for writing results" << std::endl;
      return;
    }

    SSUtil::printon(V, ofile);
    ofile.close();
    ofile.clear();
    ofile.open(h.data());
    if (ofile.fail()) {
      std::cerr << "Could not open " << v << "for writing results" << std::endl;
      return;
    }
    SSUtil::printon(H, ofile);
    ofile.close();
  }
}

// Initialize matrices for iteration
void Factorization::init_matrices(tIniType init)
{
  std::cerr << "Factorization::init_matrices()\n";
  switch (init) {
  case RANDOM:
    init_random();
    break;
    // Have not settled the behavior below....
  case FRMFILE:
    std::cerr << "Initializing from files " << vhprefix << ".[VH]\n";
    init_file();
    break;
  case OTHER:
  default:
    std::cerr << "Not implemented as yet" << std::endl;
    exit(-1);
  }
}


// random initialization of matrices
void Factorization::init_random()
{
  std::cerr << "Factorization::init_random()\n";
  // Allocate random number generator
  gsl_rng* r = gsl_rng_alloc(gsl_rng_mt19937);

  if (r == 0) 
    throw new FactMemoryAllocation();
  // Seed the random number generator
  gsl_rng_set(r, m_seed);

  // First we init V
  for (uint i = 0; i < V->size1*V->size2; i++) {
      V->data[i] = gsl_rng_uniform(r);
   
  }

  gsl_matrix_add_constant(V, 10e-12);
  // Let us test rank of iniited matrix.
  
  // Next we init H
  for (uint i = 0; i < H->size1 * H->size2; i++) {
      H->data[i] = gsl_rng_uniform(r);
  }

  // To avoid 0's in the initialization..
  gsl_matrix_add_constant(H, 10e-12);
  // Now we deallocate the random number generator
  gsl_rng_free(r);
}

/**
 * Read in V, H matrices from File name prefix that is supposed to have been set in vhprefix.
 * Have to yet add in error handling.
 */
void Factorization::init_file()
{
  if (vhprefix == 0) {
    std::cerr << "Factorization::init_file: vh filename prefix not set!" << std::endl;
    return;
  }
  int l = strlen(vhprefix);
  char* fname = new char[l+3];
  sprintf(fname, "%s.V", vhprefix);
  if (SSUtil::read_gsl_matrix(V, fname) < 0) {
    std::cerr << "File reading error!!!!!!!!" << std::endl; 
    return;
  }
  sprintf(fname, "%s.H", vhprefix);
  if (SSUtil::read_gsl_matrix(H, fname) < 0)
    return;
}

/*
void Factorization::init_random(double* dv, double* dh, int m, int n, int r)
{
  //cerr << "In init_random()\n";
  // Allocate random number generator
  gsl_rng* rg = gsl_rng_alloc(gsl_rng_mt19937);

  if (rg == 0) 
    throw new FactMemoryAllocation();
  // Seed the random number generator
  gsl_rng_set(rg, time(0));

  // First we ini dv
  for (int i = 0; i < m*r; i++)
    dv[i] = gsl_rng_uniform(rg);

  // next we ini dh
  for (int i = 0; i < r*n; i++)
    dh[i] = gsl_rng_uniform(rg);

  // deallocate
  gsl_rng_free(rg);
}

void Factorization::init_matrices(tIniType t, double* dv, double* dh, int m, int n, int r)
{
  switch (t) {
  case RANDOM:
    init_random(dv, dh, m, n, r);
    break;
  case OTHER:
  default:
    std::cerr << "Not implemented as yet" << std::endl;
    exit(-1);
  }
}

*/

double Factorization::fnorm(const gsl_matrix* X)
{
  if (X == 0) {
    std::cerr << "Factorization::fnorm(X) called for unallocated X\n";
    return -1;
  }

  double no = 0.0;
  for (uint i = 0; i < X->size1 * X->size2; i++) {
    double t = X->data[i];
    no += (t*t);
  }
  return sqrt(no);
}

double Factorization::fnorm(const gsl_matrix* X, const gsl_matrix* W)
{
  if (X == 0 || W == 0) {
    std::cerr << "Factorization::fnorm(X, W) called for unallocated X or W\n";
    return -1;
  }

  double no = 0.0;
  for (uint i = 0; i < X->size1 * X->size2; i++) {
    double t = X->data[i] * W->data[i];
    no += (t*t);
  }
  return sqrt(no);
}

double Factorization::bregmanDivergence(double (*f)(double), double (*y)(double), double p, double q)
{
  double result = 0;
  result = f(p) - f(q) -y(q)*(p - q);
  return result;
}


double Factorization::bregmanDivergence(double (*f)(double), double (*y)(double), gsl_matrix* A, gsl_matrix* B)
{
  double result = 0;
  double tmp;
  for (uint i = 0; i < A->size1 * A->size2; i++) {
    double p = A->data[i];
    double q = B->data[i];
    tmp = f(p) - f(q) -y(q)*(p - q);
    result += tmp;
  }
  return result;
}


