// File: dlas2.cc -- Implements the dlas2 class
// Author: Suvrit Sra
// Date: 14 nov, 2003
// Time-stamp: <19 February 2010 04:43:21 PM CET --  suvrit>

/*************************************************************************
 THE ORIGINAL SVDPAKC COPYRIGHT
                           (c) Copyright 1993
                        University of Tennessee
                          All Rights Reserved                          
 *************************************************************************/

#include <cstdio>
#include <cmath>
#include <cerrno>
#include <cstdlib>
#include <cstring>
#include <unistd.h>
#include <fcntl.h>

#include "error.h"
#include "dlas2.h"

#include <algorithm>
#include <functional>

#define STORQ 1
#define RETRQ 2
#define STORP 3
#define RETRP 4
#define MAXLL 2

using namespace mysvd;
/***********************************************************************
 *                                                                     *
 *				landr()				       *
 *        Lanczos algorithm with selective orthogonalization           *
 *                    Using Simon's Recurrence                         *
 *                       (double precision)                            *
 *                                                                     *
 ***********************************************************************/
/***********************************************************************

   Description
   -----------

   landr() is the LAS2 driver routine that, upon entry,
     (1)  checks for the validity of input parameters of the 
	  B-eigenproblem 
     (2)  determines several machine constants
     (3)  makes a Lanczos run
     (4)  calculates B-eigenvectors (singular vectors of A) if requested 
	  by user


   arguments
   ---------

   (input)
   n        dimension of the eigenproblem for A'A
   lanmax   upper limit of desired number of Lanczos steps
   maxprs   upper limit of desired number of eigenpairs
   nnzero   number of nonzeros in matrix A
   endl     left end of interval containing unwanted eigenvalues of B
   endr     right end of interval containing unwanted eigenvalues of B
   vectors  1 indicates both eigenvalues and eigenvectors are wanted
              and they can be found in output file lav2; 
	    0 indicates only eigenvalues are wanted
   kappa    relative accuracy of ritz values acceptable as eigenvalues
	      of B (singular values of A)
   r        work array

   (output)
   j        number of Lanczos steps actually taken                     
   neig     number of ritz values stabilized                           
   ritz     array to hold the ritz values                              
   bnd      array to hold the error bounds


   External parameters
   -------------------

   Defined and documented in las2.h


   local parameters
   -------------------

   ibeta    radix for the floating-point representation
   it       number of base ibeta digits in the floating-point significand
   irnd     floating-point addition rounded or chopped
   machep   machine relative precision or round-off error
   negeps   largest negative integer
   wptr	    array of pointers each pointing to a work space


   Functions used
   --------------

   MISC         dmax, machar, check_parameters
   LAS2         ritvec, lanso

 ***********************************************************************/

long dlas2::landr(long fpo2, FILE* fp, bool ascii, long n, 
		   long lanmax, long maxprs, long nnzero, double endl,  
		   double endr, bool vectors, double kappa, double *ritz, 
		   double *bnd, double *r)

{
   long i, size, ibeta, it, irnd, machep, negep;
   double *wptr[10], *tptr, *tptr2;

   /* data validation */
   if (check_parameters(maxprs, lanmax, n, endl, endr, vectors, nnzero))
      return(-1);

   /* Compute machine precision */ 
   machar(&ibeta, &it, &irnd, &machep, &negep);

   eps1 = eps * sqrt( (double) n );
   reps = sqrt(eps);
   eps34 = reps * sqrt(reps);

   /* allocate work area and initialize pointers         *
    * ptr             symbolic name         size         *
    * wptr[0]             r                  n           *
    * wptr[1]             q		     n           *
    * wptr[2]             q_previous         n           *
    * wptr[3]             p		     n           *
    * wptr[4]             p_previous         n           *
    * wptr[5]             wrk                n           *
    * wptr[6]             alf              lanmax        *
    * wptr[7]             eta              lanmax        *
    * wptr[8]             oldeta           lanmax        *
    * wptr[9]             bet              lanmax+1      */
   //fprintf(stderr, "dlas2::landr() about to start doing computation\n");
   size = 5 * n + (lanmax * 4 + 1);
   tptr = NULL;
   if (!(tptr = (double *) malloc(size * sizeof(double)))){
      perror("FIRST MALLOC FAILED in LANDR()");
      exit(errno);
   }
   tptr2 = tptr;
   wptr[0] = r;
   for (i = 1; i <= 5; i++) {
      wptr[i] = tptr;
      tptr += n;
   }
   for (i = 6; i <= 9; i++) {
      wptr[i] = tptr;
      tptr += lanmax;
   }

   lanso(n, lanmax, maxprs, endl, endr, ritz, bnd, wptr);

   /* compute eigenvectors */
   if (vectors) {
     size_t size = (nrow+ncol)*(j+1);
     msg_or_error(verbose, stderr, "Trying to allocate %3.3lfMB of RAM\n", double(size)/(1024.0*1024.0));
     if (!(xv1 = (double*) malloc(sizeof(double)*size)) ||
         !(xv2 = (double*) malloc(sizeof(double)*ncol))) {
       perror("MALLOC FOR SINGULAR VECTORS FAILED in LANDR()");
       exit(errno);
	 }
     kappa = std::max(fabs(kappa), eps34);
     ritvec(fpo2, fp, ascii, n, kappa, ritz, bnd, wptr[6], wptr[9], wptr[4], wptr[5]);
   }
   free(tptr2);
   return(0);
}





/***********************************************************************
 *                                                                     *
 *                         stpone()                                    *
 *                                                                     *
 ***********************************************************************/
/***********************************************************************

   Description
   -----------

   Function performs the first step of the Lanczos algorithm.  It also
   does a step of extended local re-orthogonalization.

   Arguments 
   ---------

   (input)
   n      dimension of the eigenproblem for matrix B

   (output)
   ierr   error flag
   wptr   array of pointers that point to work space that contains
	    wptr[0]             r[j]
	    wptr[1]             q[j]
	    wptr[2]             q[j-1]
	    wptr[3]             p
	    wptr[4]             p[j-1]
	    wptr[6]             diagonal elements of matrix T 


   Functions used
   --------------

   BLAS		cblas_daxpy, datx, cblas_dcopy, cblas_ddot, cblas_dscal
   USER		store, opb
   LAS		startv

 ***********************************************************************/

void dlas2::stpone(long n, double *wrkptr[])

{
   double t, *alf;
   alf = wrkptr[6];

   /* get initial vector; default is random */
   rnm = startv(n, wrkptr);
   if (rnm == 0.0 || ierr != 0) return;

   /* normalize starting vector */
   t = 1.0 / rnm;
   datx(n, t, wrkptr[0], 1, wrkptr[1], 1);
   cblas_dscal(n, t, wrkptr[3], 1);

   /* take the first step */
   mopb(n, wrkptr[3], wrkptr[0]);
   alf[0] = cblas_ddot(n, wrkptr[0], 1, wrkptr[3], 1);
   cblas_daxpy(n, -alf[0], wrkptr[1], 1, wrkptr[0], 1);
   t = cblas_ddot(n, wrkptr[0], 1, wrkptr[3], 1);
   cblas_daxpy(n, -t, wrkptr[1], 1, wrkptr[0], 1);
   alf[0] += t;
   cblas_dcopy(n, wrkptr[0], 1, wrkptr[4], 1);
   rnm = sqrt(cblas_ddot(n, wrkptr[0], 1, wrkptr[4], 1));
   anorm = rnm + fabs(alf[0]);
   tol = reps * anorm;
   return;
}


/***********************************************************************
 *                                                                     *
 *                         startv()                                    *
 *                                                                     *
 ***********************************************************************/
/***********************************************************************

   Description
   -----------

   Function delivers a starting vector in r and returns |r|; it returns 
   zero if the range is spanned, and ierr is non-zero if no starting 
   vector within range of operator can be found.

   Parameters 
   ---------

   (input)
   n      dimension of the eigenproblem matrix B
   wptr   array of pointers that point to work space
   j      starting index for a Lanczos run
   eps    machine epsilon (relative precision)

   (output)
   wptr   array of pointers that point to work space that contains
	  r[j], q[j], q[j-1], p[j], p[j-1]
   ierr   error flag (nonzero if no starting vector can be found)

   Functions used
   --------------

   BLAS		cblas_ddot, cblas_dcopy, cblas_daxpy
   USER		opb, store
   MISC		random

 ***********************************************************************/

double dlas2::startv(long n, double *wptr[])

{
   double rnm2, *r, t;
   long irand;
   long id, i;

   /* get initial vector; default is random */
   rnm2 = cblas_ddot(n, wptr[0], 1, wptr[0], 1);
   irand = 918273 + j;
   r = wptr[0];
   for (id = 0; id < 3; id++) {
      if (id > 0 || j > 0 || rnm2 == 0) 
	 for (i = 0; i < n; i++) r[i] = mrandom(&irand);
      cblas_dcopy(n, wptr[0], 1, wptr[3], 1);

      /* apply operator to put r in range (essential if m singular) */
      mopb(n, wptr[3], wptr[0]);
      cblas_dcopy(n, wptr[0], 1, wptr[3], 1);
      rnm2 = cblas_ddot(n, wptr[0], 1, wptr[3], 1);
      if (rnm2 > 0.0) break;
   }

   /* fatal error */
   if (rnm2 <= 0.0) {
      ierr = 8192;
      return(-1);
   }
   if (j > 0) {
      for (i = 0; i < j; i++) {
         store(n, RETRQ, i, wptr[5]);
	 t = -cblas_ddot(n, wptr[3], 1, wptr[5], 1);
	 cblas_daxpy(n, t, wptr[5], 1, wptr[0], 1);
      }

      /* make sure q[j] is orthogonal to q[j-1] */
      t = cblas_ddot(n, wptr[4], 1, wptr[0], 1);
      cblas_daxpy(n, -t, wptr[2], 1, wptr[0], 1);
      cblas_dcopy(n, wptr[0], 1, wptr[3], 1);
      t = cblas_ddot(n, wptr[3], 1, wptr[0], 1);
      if (t <= eps * rnm2) t = 0.0;
      rnm2 = t;
   }
   return(sqrt(rnm2));
}



/***********************************************************************
 *                                                                     *
 *				imtqlb()			       *
 *                                                                     *
 ***********************************************************************/
/***********************************************************************

   Description
   -----------

   imtqlb() is a translation of a Fortran version of the Algol
   procedure IMTQL1, Num. Math. 12, 377-383(1968) by Martin and 
   Wilkinson, as modified in Num. Math. 15, 450(1970) by Dubrulle.  
   Handbook for Auto. Comp., vol.II-Linear Algebra, 241-248(1971).  
   See also B. T. Smith et al, Eispack Guide, Lecture Notes in 
   Computer Science, Springer-Verlag, (1976).

   The function finds the eigenvalues of a symmetric tridiagonal
   matrix by the implicit QL method.


   Arguments 
   ---------

   (input)
   n      order of the symmetric tridiagonal matrix                   
   d      contains the diagonal elements of the input matrix           
   e      contains the subdiagonal elements of the input matrix in its
          last n-1 positions.  e[0] is arbitrary	             

   (output)
   d      contains the eigenvalues in ascending order.  if an error
            exit is made, the eigenvalues are correct and ordered for
            indices 0,1,...ierr, but may not be the smallest eigenvalues.
   e      has been destroyed.					    
   ierr   set to zero for normal return, j if the j-th eigenvalue has
            not been determined after 30 iterations.		    

   Functions used
   --------------

   UTILITY	fsign
   MISC		pythag

 ***********************************************************************/

void dlas2::imtqlb(long n, double d[], double e[], double bnd[])

{
   long last, l, m, i, iteration;

   /* various flags */
   long exchange, convergence, underflow;	

   double b, test, g, r, s, c, p, f;

   if (n == 1) return;
   ierr = 0;
   bnd[0] = 1.0;
   last = n - 1;
   for (i = 1; i < n; i++) {
      bnd[i] = 0.0;
      e[i-1] = e[i];
   }
   e[last] = 0.0;
   for (l = 0; l < n; l++) {
      iteration = 0;
      while (iteration <= 30) {
	 for (m = l; m < n; m++) {
	    convergence = FALSE;
	    if (m == last) break;
	    else {
	       test = fabs(d[m]) + fabs(d[m+1]);
	       if (test + fabs(e[m]) == test) convergence = TRUE;
	    }
	    if (convergence) break;
	 }
	    p = d[l]; 
	    f = bnd[l]; 
	 if (m != l) {
	    if (iteration == 30) {
	       ierr = l;
	       return;
	    }
	    iteration += 1;
	    /*........ form shift ........*/
	    g = (d[l+1] - p) / (2.0 * e[l]);
	    r = pythag(g, 1.0);
	    g = d[m] - p + e[l] / (g + fsign(r, g));
	    s = 1.0;
	    c = 1.0;
	    p = 0.0;
	    underflow = FALSE;
	    i = m - 1;
	    while (underflow == FALSE && i >= l) {
	       f = s * e[i];
	       b = c * e[i];
	       r = pythag(f, g);
	       e[i+1] = r;
	       if (r == 0.0) underflow = TRUE;
	       else {
		  s = f / r;
		  c = g / r;
		  g = d[i+1] - p;
		  r = (d[i] - g) * s + 2.0 * c * b;
		  p = s * r;
		  d[i+1] = g + p;
		  g = c * r - b;
		  f = bnd[i+1];
		  bnd[i+1] = s * bnd[i] + c * f;
		  bnd[i] = c * bnd[i] - s * f;
		  i--;
	       }
	    }       /* end while (underflow != FALSE && i >= l) */
	    /*........ recover from underflow .........*/
	    if (underflow) {
	       d[i+1] -= p;
	       e[m] = 0.0;
	    }
	    else {
	       d[l] -= p;
	       e[l] = g;
	       e[m] = 0.0;
	    }
	 } 		       		   /* end if (m != l) */
	 else {

            /* order the eigenvalues */
	    exchange = TRUE;
	    if (l != 0) {
	       i = l;
	       while (i >= 1 && exchange == TRUE) {
	          if (p < d[i-1]) {
		     d[i] = d[i-1];
		     bnd[i] = bnd[i-1];
	             i--;
	          }
	          else exchange = FALSE;
	       }
	    }
	    if (exchange) i = 0;
	    d[i] = p;
	    bnd[i] = f; 
	    iteration = 31;
	 }
      }			       /* end while (iteration <= 30) */
   }				   /* end for (l=0; l<n; l++) */
   return;
}


/***********************************************************************
 *                                                                     *
 *			lanczos_step()                                 *
 *                                                                     *
 ***********************************************************************/
/***********************************************************************

   Description
   -----------

   Function embodies a single Lanczos step

   Arguments 
   ---------

   (input)
   n        dimension of the eigenproblem for matrix B
   first    start of index through loop				      
   last     end of index through loop				     
   wptr	    array of pointers pointing to work space		    
   alf	    array to hold diagonal of the tridiagonal matrix T
   eta      orthogonality estimate of Lanczos vectors at step j   
   oldeta   orthogonality estimate of Lanczos vectors at step j-1
   bet      array to hold off-diagonal of T                     
   ll       number of intitial Lanczos vectors in local orthog. 
              (has value of 0, 1 or 2)			
   enough   stop flag			

   Functions used
   --------------

   BLAS		cblas_ddot, cblas_dscal, cblas_daxpy, datx, cblas_dcopy
   USER		store
   LAS		purge, ortbnd, startv
   UTILITY	imin, imax

 ***********************************************************************/

void dlas2::lanczos_step(long n, long first, long last, double *wptr[],
		  double *alf, double *eta, double *oldeta,
		  double *bet, long *ll, long *enough)

{
   double t, *mid;
   long i;

   for (j=first; j<last; j++) {
      mid     = wptr[2];
      wptr[2] = wptr[1];
      wptr[1] = mid;
      mid     = wptr[3];
      wptr[3] = wptr[4];
      wptr[4] = mid;

      store(n, STORQ, j-1, wptr[2]);
      if (j-1 < MAXLL) store(n, STORP, j-1, wptr[4]);
      bet[j] = rnm;

      /* restart if invariant subspace is found */
      if (!bet[j]) {
	 rnm = startv(n, wptr);
	 if (ierr) return;
	 if (!rnm) *enough = TRUE;
      }
      if (*enough) break;

      /* take a lanczos step */
      t = 1.0 / rnm;
      datx(n, t, wptr[0], 1, wptr[1], 1);
      cblas_dscal(n, t, wptr[3], 1);
      mopb(n, wptr[3], wptr[0]);
      cblas_daxpy(n, -rnm, wptr[2], 1, wptr[0], 1);
      alf[j] = cblas_ddot(n, wptr[0], 1, wptr[3], 1);
      cblas_daxpy(n, -alf[j], wptr[1], 1, wptr[0], 1);

      /* orthogonalize against initial lanczos vectors */
      if (j <= MAXLL && (fabs(alf[j-1]) > 4.0 * fabs(alf[j])))
	 *ll = j;  
      for (i=0; i < std::min(*ll, j-1); i++) {
	 store(n, RETRP, i, wptr[5]);
	 t = cblas_ddot(n, wptr[5], 1, wptr[0], 1);
	 store(n, RETRQ, i, wptr[5]);
         cblas_daxpy(n, -t, wptr[5], 1, wptr[0], 1);
	 eta[i] = eps1;
	 oldeta[i] = eps1;
      }

      /* extended local reorthogonalization */
      t = cblas_ddot(n, wptr[0], 1, wptr[4], 1);
      cblas_daxpy(n, -t, wptr[2], 1, wptr[0], 1);
      if (bet[j] > 0.0) bet[j] = bet[j] + t;
      t = cblas_ddot(n, wptr[0], 1, wptr[3], 1);
      cblas_daxpy(n, -t, wptr[1], 1, wptr[0], 1);
      alf[j] = alf[j] + t;
      cblas_dcopy(n, wptr[0], 1, wptr[4], 1);
      rnm = sqrt(cblas_ddot(n, wptr[0], 1, wptr[4], 1));
      anorm = bet[j] + fabs(alf[j]) + rnm;
      tol = reps * anorm;

      /* update the orthogonality bounds */
      ortbnd(alf, eta, oldeta, bet);

      /* restore the orthogonality state when needed */
      purge(n,*ll,wptr[0],wptr[1],wptr[4],wptr[3],wptr[5],eta,oldeta);
      if (rnm <= tol) rnm = 0.0;
   }
   return;
}



/***********************************************************************
 *                                                                     *
 *                          ortbnd()                                   *
 *                                                                     *
 ***********************************************************************/
/***********************************************************************

   Description
   -----------

   Funtion updates the eta recurrence

   Arguments 
   ---------

   (input)
   alf      array to hold diagonal of the tridiagonal matrix T         
   eta      orthogonality estimate of Lanczos vectors at step j        
   oldeta   orthogonality estimate of Lanczos vectors at step j-1     
   bet      array to hold off-diagonal of T                          
   n        dimension of the eigenproblem for matrix B		    
   j        dimension of T					  
   rnm	    norm of the next residual vector			 
   eps1	    roundoff estimate for dot product of two unit vectors

   (output)
   eta      orthogonality estimate of Lanczos vectors at step j+1     
   oldeta   orthogonality estimate of Lanczos vectors at step j        


   Functions used
   --------------

   BLAS		cblas_dswap

 ***********************************************************************/

void dlas2::ortbnd(double *alf, double *eta, double *oldeta, double *bet)

{
   long i;
   if (j < 1) return;
   if (rnm) {
      if (j > 1) {
	 oldeta[0] = (bet[1] * eta[1] + (alf[0]-alf[j]) * eta[0] -
		      bet[j] * oldeta[0]) / rnm + eps1;
      }
      for (i=1; i<=j-2; i++) 
	 oldeta[i] = (bet[i+1] * eta[i+1] + (alf[i]-alf[j]) * eta[i] +
		      bet[i] * eta[i-1] - bet[j] * oldeta[i])/rnm + eps1;
   }
   oldeta[j-1] = eps1;
   cblas_dswap(j, oldeta, 1, eta, 1);  
   eta[j] = eps1;
   return;
}


/***********************************************************************
 *                                                                     *
 *				purge()                                *
 *                                                                     *
 ***********************************************************************/
/***********************************************************************

   Description
   -----------

   Function examines the state of orthogonality between the new Lanczos
   vector and the previous ones to decide whether re-orthogonalization 
   should be performed


   Arguments 
   ---------

   (input)
   n        dimension of the eigenproblem for matrix B		       
   ll       number of intitial Lanczos vectors in local orthog.       
   r        residual vector to become next Lanczos vector            
   q        current Lanczos vector			           
   ra       previous Lanczos vector
   qa       previous Lanczos vector
   wrk      temporary vector to hold the previous Lanczos vector
   eta      state of orthogonality between r and prev. Lanczos vectors 
   oldeta   state of orthogonality between q and prev. Lanczos vectors
   j        current Lanczos step				     

   (output)
   r	    residual vector orthogonalized against previous Lanczos 
	      vectors
   q        current Lanczos vector orthogonalized against previous ones


   Functions used
   --------------

   BLAS		cblas_daxpy,  cblas_dcopy,  idamax,  cblas_ddot
   USER		store

 ***********************************************************************/

void dlas2::purge(long n, long ll, double *r, double *q, double *ra,  
	   double *qa, double *wrk, double *eta, double *oldeta)

{
   double t, tq, tr, reps1;
   long k, iteration, flag, i;

   if (j < ll+2) return; 

   k = idamax(j - (ll+1), &eta[ll], 1) + ll;
   if (fabs(eta[k]) > reps) {
      reps1 = eps1 / reps;
      iteration = 0;
      flag = TRUE;
      while (iteration < 2 && flag) {
	 if (rnm > tol) {

	    /* bring in a lanczos vector t and orthogonalize both 
	     * r and q against it */
	    tq = 0.0;
	    tr = 0.0;
            for (i = ll; i < j; i++) {
	       store(n,  RETRQ,  i,  wrk);
	       t   = -cblas_ddot(n, qa, 1, wrk, 1);
	       tq += fabs(t);
	       cblas_daxpy(n,  t,  wrk,  1,  q,  1);
	       t   = -cblas_ddot(n, ra, 1, wrk, 1);
	       tr += fabs(t);
	       cblas_daxpy(n, t, wrk, 1, r, 1);
	    }
	    cblas_dcopy(n, q, 1, qa, 1);
	    t   = -cblas_ddot(n, r, 1, qa, 1);
	    tr += fabs(t);
	    cblas_daxpy(n, t, q, 1, r, 1);
	    cblas_dcopy(n, r, 1, ra, 1);
	    rnm = sqrt(cblas_ddot(n, ra, 1, r, 1));
	    if (tq <= reps1 && tr <= reps1 * rnm) flag = FALSE;
	 }
	 iteration++;
      }
      for (i = ll; i <= j; i++) { 
	 eta[i] = eps1;
	 oldeta[i] = eps1;
      }
   }
   return;
}


/***************************************************************** 
 * Function finds the index of element having max. absolute value*
 * based on FORTRAN 77 routine from Linpack by J. Dongarra       *
 *****************************************************************/ 

long dlas2::idamax(long n,double *dx,long incx)

{
   long ix,i,imax;
   double dtemp, dmax;

   if (n < 1) return(-1);
   if (n == 1) return(0);
   if (incx == 0) return(-1);

   if (incx < 0) ix = (-n+1) * incx;
   else ix = 0;
   imax = ix;
   dx += ix;
   dmax = fabs(*dx);
   for (i=1; i < n; i++) {
      ix += incx;
      dx += incx;
      dtemp = fabs(*dx);
      if (dtemp > dmax) {
	 dmax = dtemp;
	 imax = ix;
      }
   }
   return(imax);
}


/***********************************************************************
 *                                                                     *
 *			error_bound()                                  *
 *                                                                     *
 ***********************************************************************/
/***********************************************************************

   Description
   -----------

   Function massages error bounds for very close ritz values by placing 
   a gap between them.  The error bounds are then refined to reflect 
   this.


   Arguments 
   ---------

   (input)
   endl     left end of interval containing unwanted eigenvalues
   endr     right end of interval containing unwanted eigenvalues
   ritz     array to store the ritz values
   bnd      array to store the error bounds
   enough   stop flag


   Functions used
   --------------

   BLAS		idamax
   UTILITY	dmin

 ***********************************************************************/

void dlas2::error_bound(long *enough, double endl, double endr, 
		 double *ritz, double *bnd)
{
   long mid, i;
   double gapl, gap;

   /* massage error bounds for very close ritz values */
   mid = idamax(j + 1, bnd, 1);

   for (i=((j+1) + (j-1)) / 2; i >= mid + 1; i -= 1)
      if (fabs(ritz[i-1] - ritz[i]) < eps34 * fabs(ritz[i])) 
         if (bnd[i] > tol && bnd[i-1] > tol) {
	    bnd[i-1] = sqrt(bnd[i] * bnd[i] + bnd[i-1] * bnd[i-1]);
	    bnd[i] = 0.0;
         }
	 

   for (i=((j+1) - (j-1)) / 2; i <= mid - 1; i +=1 ) 
      if (fabs(ritz[i+1] - ritz[i]) < eps34 * fabs(ritz[i])) 
	 if (bnd[i] > tol && bnd[i+1] > tol) {
	    bnd[i+1] = sqrt(bnd[i] * bnd[i] + bnd[i+1] * bnd[i+1]);
	    bnd[i] = 0.0;
         }

   /* refine the error bounds */
   neig = 0;
   gapl = ritz[j] - ritz[0];
   for (i = 0; i <= j; i++) {
      gap = gapl;
      if (i < j) gapl = ritz[i+1] - ritz[i];
      gap = std::min(gap, gapl);
      if (gap > bnd[i]) bnd[i] = bnd[i] * (bnd[i] / gap);
      if (bnd[i] <= 16.0 * eps * fabs(ritz[i])) {
	 neig += 1;
	 if (!*enough) *enough = endl < ritz[i] && ritz[i] < endr;
      }
   }   
   return;
}




/***********************************************************************
 *                                                                     *
 *                        ritvec()                                     *
 * 	    Function computes the singular vectors of matrix A	       *
 *                                                                     *
 ***********************************************************************/
/***********************************************************************

   Description
   -----------

   This function is invoked by landr() only if eigenvectors of the cyclic
   eigenproblem are desired.  When called, ritvec() computes the 
   singular vectors of A and writes the result to an unformatted file.


   Parameters
   ----------

   (input)
   n	      dimension of the eigenproblem for matrix B
   kappa      relative accuracy of ritz values acceptable as 
		eigenvalues of B (singular values of A)
   ritz       array of ritz values
   bnd        array of error bounds
   alf        array of diagonal elements of the tridiagonal matrix T
   bet        array of off-diagonal elements of T
   w1, w2     work space
   fp_out2    pointer to unformatted output file
   j	      number of Lanczos iterations performed

   (output)
   xv1        array of eigenvectors of B (singular vectors of A)
   ierr	      error code
	      0 for normal return from imtql2()
	      k if convergence did not occur for k-th eigenvalue in 
		imtql2()
   nsig       number of accepted ritz values based on kappa

   (local)
   s	      work array which is initialized to the identity matrix of
	      order (j + 1) upon calling imtql2().  After the call, 
	      s contains the orthonormal eigenvectors of the symmetric 
	      tridiagonal matrix T

   Functions used
   --------------

   BLAS		cblas_dscal, cblas_dcopy, cblas_daxpy
   USER		store
   		imtql2

 ***********************************************************************/

void dlas2::ritvec(long fpo2, FILE* fp, bool ascii, long n, 
		    double kappa, double *ritz, double *bnd,
		    double *alf, double *bet, double *w1, double *w2)

{
   long js, jsq, i, k, size, id, id2, tmp;
   double *s;

   js = j + 1;
   jsq = js * js;
   size = sizeof(double) * n;

   if(!(s = (double *) malloc (jsq * sizeof(double)))) {
      perror("MALLOC FAILED in RITVEC()");
      exit(errno);
   }

   /* initialize s as an identity matrix */
   for (i = 0; i < jsq; i++) s[i] = 0.0;
   for (i = 0; i < jsq; i+= (js + 1)) s[i] = 1.0;
   cblas_dcopy(js, alf, 1, w1, -1);
   cblas_dcopy(j, &bet[1], 1, &w2[1], -1);

   /* compute eigenvalues of T */
   imtql2(js, js, w1, w2, s);
   if (ierr) return;

   /* on return w1 contains eigenvalues in ascending order and s 
    * contains the corresponding eigenvectors */

   //   write(fpo2, (char *)&n, sizeof(n));
   //write(fpo2, (char *)&js, sizeof(js));
   //write(fpo2, (char *)&kappa, sizeof(kappa));

   nsig = 0;
   id = 0;
   // Count the number of valid singular values
   for (k = 0; k < js; k++) {
     if (bnd[k] <=  kappa*fabs(ritz[k]) )
       ++nsig;
   }

   write_header(fpo2, fp, ascii, nrow, ncol, nsig, "las2");       

   nsig = 0;
   id2 = jsq - js;
   ssize_t retval;
   for (k = 0; k < js; k++) {
     tmp = id2;
     // IT IS CRITICAL TO HAVE the TYPECAST BELOW....
     if (bnd[k] <= kappa * fabs(ritz[k]) && k > long(js-neig-1)) {
       for (i = 0; i < n; i++) w1[i] = 0.0;
        for (i = 0; i < js; i++) {
          store(n, RETRQ, i, w2);
          cblas_daxpy(n, s[tmp], w2, 1, w1, 1);
          tmp -= js;
        }
        if (ascii) {
          for (unsigned int zz = 0; zz < size/sizeof(double); zz++) {
            fprintf(fp, "%f ", w1[zz]);
          }
          fprintf(fp, "\n");
        }
        else
          retval=write(fpo2, (char *)w1, size);
        
        
        /* store the w1 vector row-wise in array xv1;   
         * size of xv1 is (j+1) * (nrow+ncol) elements 
         * and each vector, even though only ncol long,
         * will have (nrow+ncol) elements in xv1.      
         * It is as if xv1 is a 2-d array (j+1) by     
         * (nrow+ncol) and each vector occupies a row  */
        
        for (i = 0; i < n; i++) xv1[id++] = w1[i];
        id += nrow;
        nsig += 1;
      }
      id2++;
   }

   V = xv1;
   free(s);
   return;
}


/***********************************************************************
 *                                                                     *
 *                     store()                                         *
 *                                                                     *
 ***********************************************************************/
/***********************************************************************

   Description
   -----------

   store() is a user-supplied function which, based on the input
   operation flag, stores to or retrieves from memory a vector.


   Arguments 
   ---------

   (input)
   n       length of vector to be stored or retrieved
   isw     operation flag:
	     isw = 1 request to store j-th Lanczos vector q(j)
	     isw = 2 request to retrieve j-th Lanczos vector q(j)
	     isw = 3 request to store q(j) for j = 0 or 1
	     isw = 4 request to retrieve q(j) for j = 0 or 1
   s	   contains the vector to be stored for a "store" request 

   (output)
   s	   contains the vector retrieved for a "retrieve" request 

   Functions used
   --------------

   BLAS		cblas_dcopy

 ***********************************************************************/

void dlas2::store(long n, long isw, long j, double *s)

{
   switch(isw) {
   case STORQ:	cblas_dcopy(n, s, 1, &a[(j+MAXLL) * n], 1);
		break;
   case RETRQ:	cblas_dcopy(n, &a[(j+MAXLL) * n], 1, s, 1);
		break;
   case STORP:	if (j >= MAXLL) {
		   fprintf(stderr,"store: (STORP) j >= MAXLL \n");
		   break;
		}
		cblas_dcopy(n, s, 1, &a[j*n], 1);
		break;
   case RETRP:	if (j >= MAXLL) {
		   fprintf(stderr,"store: (RETRP) j >= MAXLL \n");
		   break;
		}
   		cblas_dcopy(n, &a[j*n], 1, s, 1);
		break;
   }
   return;
}



/***********************************************************************
 *                                                                     *
 *                          lanso()                                    *
 *                                                                     *
 ***********************************************************************/
/***********************************************************************

   Description
   -----------

   Function determines when the restart of the Lanczos algorithm should 
   occur and when it should terminate.

   Arguments 
   ---------

   (input)
   n         dimension of the eigenproblem for matrix B
   lanmax    upper limit of desired number of lanczos steps           
   maxprs    upper limit of desired number of eigenpairs             
   endl      left end of interval containing unwanted eigenvalues
   endr      right end of interval containing unwanted eigenvalues
   ritz      array to hold the ritz values                       
   bnd       array to hold the error bounds                          
   wptr      array of pointers that point to work space:            
  	       wptr[0]-wptr[5]  six vectors of length n		
  	       wptr[6] array to hold diagonal of the tridiagonal matrix T
  	       wptr[9] array to hold off-diagonal of T	
  	       wptr[7] orthogonality estimate of Lanczos vectors at 
		 step j
 	       wptr[8] orthogonality estimate of Lanczos vectors at 
		 step j-1

   (output)
   j         number of Lanczos steps actually taken
   neig      number of ritz values stabilized
   ritz      array to hold the ritz values
   bnd       array to hold the error bounds
   ierr      (globally declared) error flag
	     ierr = 8192 if stpone() fails to find a starting vector
	     ierr = k if convergence did not occur for k-th eigenvalue
		    in imtqlb()
	     ierr = 0 otherwise


   Functions used
   --------------

   LAS		stpone, error_bound, lanczos_step
   MISC		dsort2
   UTILITY	imin, imax

 ***********************************************************************/

void dlas2::lanso(long n, long lanmax, long maxprs, double endl,
	   double endr, double *ritz, double *bnd, double *wptr[])

{
   double *r, *alf, *eta, *oldeta, *bet, *wrk;
   long ll, first, last, ENOUGH, id1, id2, id3, i, l;

   r = wptr[0];
   alf = wptr[6];
   eta = wptr[7];
   oldeta = wptr[8];
   bet = wptr[9];
   wrk = wptr[5];
   j = 0;

   /* take the first step */
   stpone(n, wptr);
   if (!rnm || ierr) return;
   eta[0] = eps1;
   oldeta[0] = eps1;
   ll = 0;
   first = 1;
   last = std::min(maxprs + std::max(8L,maxprs), lanmax);
   ENOUGH = FALSE;
   id1 = 0;
   while (id1 < maxprs && !ENOUGH) {
      if (rnm <= tol) rnm = 0.0;

      /* the actual lanczos loop */
      lanczos_step(n, first, last, wptr, alf, eta, oldeta, bet, &ll,  &ENOUGH);
      if (ENOUGH) j = j - 1;
      else j = last - 1;
      first = j + 1;
      bet[j+1] = rnm;

      /* analyze T */
      l = 0;
      for (id2 = 0; id2 < j; id2++) {
	 if (l > j) break;
         for (i = l; i <= j; i++) if (!bet[i+1]) break;
	 if (i > j) i = j;

	 /* now i is at the end of an unreduced submatrix */
	 cblas_dcopy(i-l+1, &alf[l],   1, &ritz[l],  -1);
	 cblas_dcopy(i-l,   &bet[l+1], 1, &wrk[l+1], -1);

	 imtqlb(i-l+1, &ritz[l], &wrk[l], &bnd[l]);

	 if (ierr) {
	    printf("IMTQLB FAILED TO CONVERGE (IERR = %ld)\n",    ierr);
	    printf("L = %ld    I = %ld\n", l, i);
            for (id3 = l; id3 <= i; id3++) 
	       printf("%ld  %lg  %lg  %lg\n",
	       id3, ritz[id3], wrk[id3], bnd[id3]);
	 }
         for (id3 = l; id3 <= i; id3++) 
	    bnd[id3] = rnm * fabs(bnd[id3]);
	 l = i + 1;
      }

      /* sort eigenvalues into increasing order */
      dsort2((j+1) / 2, j + 1, ritz, bnd);

      /* massage error bounds for very close ritz values */
      error_bound(&ENOUGH, endl, endr, ritz, bnd);

      /* should we stop? */
      if (long(neig) < long(maxprs)) {
        if (!neig) 
          last = first + 9;
        else 
          last = first + std::max(3UL, 1 + ((j-5) * (maxprs-neig)) / neig);
        last = std::min(last, lanmax);
      } else 
        ENOUGH = TRUE;
      ENOUGH = ENOUGH || first >= lanmax;
      id1++;
   }
   store(n, STORQ, j, wptr[1]);
   return;
}





int dlas2::read_options_fromfile(char* optfil)
{
  options = new OptionsStruct();

  /* open files for input/output */
  if (!(options->ofp = fopen(optfil, "r"))) {
    msg_or_error(1, stderr, "DLAS2: could not open options file: `%s' for reading\n", optfil);
    return -1;
  }
  int r;
  r=fscanf(options->ofp, "%s", options->matrix_file); // read data matrix from
  r=fscanf(options->ofp, "%s", options->buf);         // sparse mtx type

  if ( strcmp(options->buf, "M_CCS") == 0) {
    options->data_typ = 0;
  } else if ( strcmp(options->buf, "M_CCS3TXX") == 0) {
    options->data_typ = 1;
  } else if ( strcmp(options->buf, "M_CCS3TFN") == 0) {
    options->data_typ = 2;
  } else if ( strcmp(options->buf, "M_MAT") == 0) {
    options->data_typ = 3;
  } else if ( strcmp(options->buf, "M_HB") == 0) {
    options->data_typ = 4;
  } else {
    msg_or_error(1, stderr, "DLAS2: dlas2::read_options_fromfile() Invalid matrix format: %s\n", options->buf);
    myFCLOSE (options->ofp);
    return -1;
  }
  
  r=fscanf(options->ofp, "%s", options->buf);         // bin data or not
  if ( strcmp(options->buf, "BIN") == 0)
    options->data_is_bin = true;
  else
    options->data_is_bin = false;


  r=fscanf(options->ofp, "%zu", &options->lanmax);    // max num. of lanczos
                                                    // steps allowed...
  r=fscanf(options->ofp, "%zu", &options->nums);      // num. of singular triples
  r=fscanf(options->ofp, "%lf", &options->endl);      // Left-edge of interval
  r=fscanf(options->ofp, "%lf", &options->endr);      // right end of interval
  r=fscanf(options->ofp, "%lf", &options->kappa);     // Accuracy / tolerance
  r=fscanf(options->ofp, "%s",  options->buf);        // vectors desired
  // If singular vectors should be output
  if ( strcmp(options->buf, "VECS") == 0)
    options->vectors = true;
  else
    options->vectors = false;

  r=fscanf(options->ofp, "%s",  options->buf);        // binary or ascii mode
  // Check if ascii mode is requested
  if ( strcmp(options->buf, "BINVECS") == 0)
    options->ascii = false;
  else
    options->ascii = true;
  
  sprintf(options->valfile, "%s.vals", options->matrix_file);
  sprintf(options->vecfile, "%s.vecs", options->matrix_file);

  r=fscanf(options->ofp, "%s", options->buf);
  if ( strcmp(options->buf, "VERBOSE") == 0)
    options->verbose = 1;
  else
    options->verbose = 0;

  /* E.g., options file
     matrix M_CCS TXT 10 4 -1e-10 1e-10 1e-6 VECS ASCII_VECS VERBOSE
  */
  return 0;
}


int dlas2::process_options()
{
  msg_or_error(verbose,stdout, "dlas2::process_options(): Loading data matrix...");

  // Now try to load the data matrix
  options->dA = new SparseMatrix<double>();
  options->dA->load(options->matrix_file, options->data_is_bin, options->data_typ);
  if (options->dA->status() < 0) {
    msg_or_error(verbose,stderr,"\ndlas2::process_options() Error loading matrix: %s\n", options->matrix_file);
    return options->dA->status();
  }
  msg_or_error(verbose,stdout, "Done\n");
  setsize(options->dA->nrows(), options->dA->ncols(), options->dA->nnz());
  setptrs(options->dA->get_colptrs(), options->dA->get_rowidx(), options->dA->get_val());

  if (options->vectors && options->ascii) {
    if ( !(options->vecs = fopen(options->vecfile, "w"))) {
      msg_or_error(verbose,stderr,"DLAS2: cannot open output file %s\n", options->vecfile);
      return -1;
    }
  }
  
  if (options->vectors && !options->ascii) {
    if ((options->vecfp = open(options->vecfile, O_TRUNC | O_RDWR | O_CREAT, DEFFILEMODE)) == -1) {
      msg_or_error(verbose,stderr,"DLAS2: cannot open output file %s\n", options->vecfile);
      msg_or_error(verbose,stderr, "%s\n", strerror(errno));
      return -1;
    }
  }
  
  if (!(options->valfp = fopen(options->valfile, "w"))) {
    msg_or_error(verbose,stderr,"DLAS2: cannot open output file %s\n", options->valfile);
    //if (options->vectors && !options->ascii) myFCLOSE(options->vecs);
    if (options->vectors && options->ascii) close(options->vecfp);
    return -1;
  }

  verbose = options->get_verbosity();
  return 0;
}

/*
 * Create somewhat decent default parameter values for lazy people.....
 */
void dlas2::setup_default()
{
  options = new OptionsStruct();

  strcpy(options->matrix_file, "matrix");
  options->data_typ = M_CCS; // CCS
  options->data_is_bin = false;
  strcpy(options->vecfile, "matrix.vecs");
  strcpy(options->valfile, "matrix.vals");
  options->lanmax = 10;
  options->endl    = -1e-30;
  options->endr    = 1e-30;
  options->nums  = 1;
  options->kappa   = 1e-6;
  options->ascii   = true;
  options->vectors = true;
}

int dlas2::run()
{
  int flg;
  if ( (flg = process_options()) < 0)
    return flg;

  float t0, exetime;
  double tmp0, tmp1, xnorm;
  double *r, *ritz, *bnd, *d, *tptr1;
  long k, i, id, n, size1, memory;
  long ida, count1, count2;
  long nn;


  memory = 0;
  n = ncol;
  nn   = ncol + nrow;

  /* write header of output file */
  write_data(stdout, options->lanmax, options->nums, options->endl,
	     options->endr, options->vectors, options->kappa,
	     nrow, ncol, n);

  ierr = 0;
  size1 = sizeof(double) * (6 * n + nrow  + (n * options->lanmax));
  tptr1 = (double*) malloc(size1);
  
  msg_or_error(verbose, stderr, "Trying to allocate %3.3lfMB of RAM\n", double(size1)/(1024.0*1024.0));
  if (!tptr1) {
    msg_or_error(verbose, stderr, "Could not allocate %zu bytes of memory in dlas2::run()\n", size1);
    return -1;
  }

  r      = tptr1;
  tptr1 += n;
  ritz   = tptr1;
  tptr1 += n;
  bnd    = tptr1;
  tptr1 += n;
  d      = tptr1;
  tptr1 += n;
  ztemp     = tptr1;
  tptr1 += nrow;
  a      = tptr1;

  sing   = (double*) malloc(sizeof(double)*options->nums); 	
  for (int zz = 0; zz < options->nums; zz++)
    sing[zz] = 0;

  exetime = timer();

  /* to get a random starting vector, the first n cells must be
   * initialize to zero */
  for (int i = 0; i < n; i++) r[i] = 0.;


  /* 
  

  landr(options->vecfp, options->vecs, options->ascii, n, options->lanmax,
	    options->nums, nnz, options->endl, options->endr, options->vectors, 
            options->kappa, ritz, bnd, r)
  */

  /*
  std::cout << options->vecfp << "\n"
            << options->vecs << "\n"
            << options->ascii << "\n"
            << n << "\n"
            << options->lanmax << "\n"
            << options->nums << "\n"
            << nnz << "\n"
            << options->endl << "\n"
            << options->endr << "\n"
            << options->vectors << "\n"
            << options->kappa << "\n"
            << ritz << "\n"
            << bnd << " " << r << "\n";
  */

  /* make a lanczos run; see landr for meaning of parameters */
  if ( landr(options->vecfp, options->vecs, options->ascii, 
             n, options->lanmax, options->nums, nnz, options->endl, 
             options->endr, options->vectors, options->kappa, ritz, 
             bnd, r) != 0) {
    msg_or_error(verbose, stderr, "dlas2::landr() failed\n");
    options->dA->free_data();
    
    myFCLOSE(options->ofp);
    
    if (options->vectors) {
      close(options->vecfp);
      if (xv1) free(xv1);
      if (xv2) free(xv2);
    }
    return -1;
  }
  
  exetime = timer() - exetime;

  /* memory allocated for xv1, xv2 and s in landr() */
  if (options->vectors) 
    memory += sizeof(double) * (nn * (j+1) + n + (j+1) * (j+1));
  
  /* print error code if not zero */
  if (ierr)
    msg_or_error(verbose, stderr, " ... RETURN FLAG = %9ld ...\n", ierr);

  /* print ritz values and error bounds */
  msg_or_error(verbose, stdout, "\n");
  msg_or_error(verbose, stdout, " ...... ALLOCATED MEMORY (BYTES)= %lf\n", (double)memory);
  msg_or_error(verbose, stdout, " ...... LANSO EXECUTION TIME=%10.2E\n", exetime);
  msg_or_error(verbose, stdout, " ...... \n");
  msg_or_error(verbose, stdout, " ...... NUMBER OF LANCZOS STEPS = %3ld       NEIG = %3ld\n", j+1, neig);
  msg_or_error(verbose, stdout, " ...... \n");
  msg_or_error(verbose, stdout, " ......         COMPUTED RITZ VALUES  (ERROR BNDS)\n");
  msg_or_error(verbose, stdout, " ...... \n");
  for (i = 0; i <= j; i++)
    msg_or_error(verbose, stdout, " ...... %3ld   %22.14E  (%11.2E)\n",
                 i + 1, ritz[i], bnd[i]);
  /* compute residual error when singular values and vectors are 
   * computed.  This is done only for singular values that are
   * within the relative accuracy (kappa) */
  if (options->vectors) {
    //msg_or_error(verbose, stderr, "dlas2::options->vectors\n");
    size1 = sizeof(double) * nrow;
    t0 = timer();
    id = 0;

    //msg_or_error(verbose, stderr, "dlas2::NSIG = %d\n", nsig);
    //DBG("dlas2::writing out: " << nsig << " vectors\n");
    U = xv1;
    V = xv1;
    ssize_t retval;
    //std::cerr << "Num of sig vectors == " << nsig << std::endl;
    for (i = 0; i < nsig; i++) {
      /* multiply by matrix B first */
      mopb(n, &xv1[id], xv2);
      tmp0 = cblas_ddot(n, &xv1[id], 1, xv2, 1);
      cblas_daxpy(n, -tmp0, &xv1[id], 1, xv2, 1);
      tmp0 = sqrt(tmp0);
      xnorm = sqrt(cblas_ddot(n, xv2, 1, xv2, 1));
      ida = id + ncol;

      /* multiply by matrix A to get (scaled) left s-vector */
      mopa(&xv1[id], &xv1[ida]);
      tmp1 = 1.0 / tmp0;
      cblas_dscal(nrow, tmp1, &xv1[ida], 1);
      xnorm *= tmp1;
      bnd[i] = xnorm;
      d[i] = tmp0;

      /* write left s-vector to output file */
      if (options->ascii) {
        for (int zz = 0; zz < nrow; zz++)
          msg_or_error(verbose, options->vecs, "%f ", xv1[ida+ zz]);
        msg_or_error(verbose, options->vecs, "\n");
      } 
      else
        retval=write(options->vecfp, (char *)&xv1[ida], size1);
      id += nn;
    }

    exetime += (timer() - t0);
    count1=(mxvcount-nsig)/2 + nsig;
    count2=(mxvcount-nsig)/2;
      msg_or_error(verbose, stdout, " ...... \n");
      msg_or_error(verbose, stdout, " ...... NO. MULTIPLICATIONS BY A  =%10ld\n", count1);
      msg_or_error(verbose, stdout, " ...... NO. MULT. BY TRANSPOSE(A) =%10ld\n", count2);
      msg_or_error(verbose, stdout, "\n");
      msg_or_error(verbose, stdout, " ...... LASVD EXECUTION TIME=%10.2E\n", exetime);
      msg_or_error(verbose, stdout, " ...... \n");
      msg_or_error(verbose, stdout, " ......        NSIG = %4ld\n", nsig);
      msg_or_error(verbose, stdout, " ...... \n");
      msg_or_error(verbose, stdout, " ......         COMPUTED S-VALUES     (RES. NORMS)\n");
      msg_or_error(verbose, stdout, " ...... \n");
      for (i = 0; i < nsig; i++) {
        msg_or_error(verbose, stdout, " ...... %3ld   %22.14E  (%11.2E)\n", i + 1, d[i], bnd[i]);
        sing[i] = d[i];
      }
  } else {
    for (i = j; i >= 0; i--)
      if (bnd[i] > options->kappa * fabs(ritz[i])) break;
    nsig = j - i;
    
    count1=(mxvcount-nsig)/2 + nsig;
    count2=(mxvcount-nsig)/2;
    msg_or_error(verbose, stdout," ...... \n");
    msg_or_error(verbose, stdout," ...... NO. MULTIPLICATIONS BY A  =%10ld\n", count1);
    msg_or_error(verbose, stdout," ...... NO. MULT. BY TRANSPOSE(A) =%10ld\n", count2);
    msg_or_error(verbose, stdout, "\n");
    msg_or_error(verbose, stdout," ...... LASVD EXECUTION TIME = %10.2E\n", exetime);
    msg_or_error(verbose, stdout," ...... \n");
    msg_or_error(verbose, stdout," ......         NSIG = %4ld\n" , nsig);
    msg_or_error(verbose, stdout," ...... \n");
    msg_or_error(verbose, stdout," ......         COMPUTED S-VALUES   (ERROR BNDS)\n");
    msg_or_error(verbose, stdout," ...... \n");
    k = j + 1 - nsig;
    for (i = 1 ; i <= nsig; i++) {
      sing[i-1] = sqrt(ritz[k]);
      msg_or_error(verbose, stdout," ...... %3ld   %22.14E  (%11.2E)\n", i, sing[i-1], bnd[k]);
      k++;
    }
  }

  // Write out the svals to file
  msg_or_error(verbose, options->valfp, "%d\n", nsig);

  // Sort sing in place ...
  std::sort(sing, sing + nsig, std::greater<double>());
  
  for (int zz = 0; zz < nsig; zz++) 
    msg_or_error(verbose, options->valfp, "%.14f\n", sing[zz]);

  
  myFCLOSE(options->ofp);
  //myFCLOSE(options->valfp);
  myFCLOSE(options->vecs);
  if (options->vecfp) close(options->vecfp);
  return 0;
}


int   dlas2::get_svals(double* s, size_t K)
{
  for (size_t i = 0; i < K; i++)
    s[i] = sing[i];
  return 0;
}

// TODO:::::::::::::::::::::::::::
int   dlas2::get_svecs(double* uU, double* uV, size_t K)
{
  for (size_t i = 0; i < K; i++) {
    uU[i] = U[i];
    uV[i] = V[i];
  }
  return 0;
}
