// -*- c++ -*-
// Copyright (C) 2003 Suvrit Sra (suvrit@cs.utexas.edu)

// Time-stamp: <19 February 2010 04:06:51 PM CET --  suvrit>

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#ifndef _SPARSEMATRIX_H
#define _SPARSEMATRIX_H

#include <iostream>
#include <fstream>
#include <string>
#include <cstdlib>
#include "my_vector.h"

#define M_CCS 0
#define M_CCS3TXX 1
#define M_CCS3TFN 2
#define M_MAT 3
#define M_HB 4

using namespace std;

template <typename ValType> class SparseMatrix {
private:
  size_t m_rows, m_cols;

  /* number of non-zeros */
  size_t m_nz;
     
  /* file name associated with this matrix */
  string m_fname;

  /* The actual non-zeroes themselves */
  ValType* m_val;

  /* Colptrs for CCs structure */
  size_t* m_colptrs;
  
  /* Row indices for CCs structure */
  size_t* m_rowindx;
  
  /* Is the data in externally allocated arrays */
  bool m_is_external;

  /* We compute norm only once to save time */
  double fnormA;

  /* Private function to carry out computation of fnorm of matrx */
  void compute_fnorm();
  bool   fnorm_avail;		// is it avail or needs to be computed.

  /* the error / success status */
  int m_state;

public:
  SparseMatrix() {
    m_rows = m_cols = m_nz = 0L; fnormA = -1.0; fnorm_avail = false;
  }

  SparseMatrix(long r, long c, long n) { 
    //assert (r > 0 and c > 0 and n >= 0);
    m_rows = r; 
    m_cols = c;
    m_nz   = n;

    // Allocate the arrays
    m_colptrs = new size_t[c+1];
    m_rowindx = new size_t[n];
    m_val     = new ValType[n];
    m_is_external = false;
    fnorm_avail = false;
    fnormA = -1.0;
    m_fname = "";
  }
  
  ~SparseMatrix() {
  }

  void free_data() {
    free(m_colptrs); free(m_rowindx); free(m_val);
  }
  void setsize(size_t m, size_t n, size_t nz) 
  { m_rows = m; m_cols = n; m_nz = nz;}

  SparseMatrix* clone();

  int status() const { return m_state;}
  bool isExternal() const { return m_is_external;}
  
  // Allocate the matrix
  int matrix_alloc (size_t m, size_t n, size_t nz);
  int matrix_calloc (size_t m, size_t n, size_t nz);
  int matrix_resize (size_t m, size_t n, size_t nz);

  int read_file_header_bin(FILE* fp, size_t* m, size_t* n, size_t* nz, int* mt, int emt);
  int load(char*, bool asbin, int typ);
  int load_bin(char*);
  int load_txt(char*);
  int load_matlab(char*);
  int load_hb(char*);
  int load_ccs3(char*, char*);

  int save (const char*, bool asbin, int typ);
  int save_as_matlab (const char*, bool);
  int save_matlab_bin(const char* fil);
  int save_matlab_txt(const char*);
  
  size_t*    get_colptrs() { return m_colptrs;}
  size_t*    get_rowidx()   { return m_rowindx;}
  ValType*   get_val()    { return m_val;    }

  long nrows() const { return m_rows;}
  long ncols() const { return m_cols;}
  long nnz  () const { return m_nz;  }


  // This operator multiplies This . x 
  my_vector<ValType>* operator * (const my_vector<ValType>* x);

  //  v'col dot prod.
  ValType dot(size_t col, my_vector<ValType>* v);

  // do a saxpy
  void saxpy(double alpha, int col, my_vector<ValType>* v);

  // This does a dot product of col i with col j
  ValType col_dotprod(size_t i, size_t j);

  // This computes the cosine between col i and and col j.
  double col_cosine(size_t i, size_t j);

  // This computes the cosine of col i with input vector v
  double col_cosine2(size_t i, my_vector<ValType>* v);

  // return the frob norm of matrix
  double fnorm()  { if (fnorm_avail) return fnormA; else compute_fnorm(); return fnormA;}

  double col_diff(int i, int j);
  double col_delta(int, double*);

  // Fill input vector with column of A
  void getcol(size_t, my_vector<ValType>*);

  // This multiplies A^T . x
  my_vector<ValType>* tran_times (const my_vector<ValType>* x);

  /// ------------------------- ROW / COL OPERATIONS -----------------------------------

  /// Return row 'r' of matrix
  int get_row(size_t r, my_vector<ValType>*&);

  /// Return col 'c' of matrix
  int get_col(size_t c, my_vector<ValType>*&);

  /// Sets the specified row to the given vector
  int set_row(size_t r, my_vector<ValType>*&);
  
  /// Sets the existing non-zeros, ignores other entries
  int fast_set_row(size_t r, my_vector<ValType>*&);
  
  
  /// Sets the specified col to the given vector
  int set_col(size_t c, my_vector<ValType>*&);
  
  /// Sets only the existing non-zeros, ignores others
  int fast_set_col(size_t c, my_vector<ValType>*&);
  
  /// ||c||
  double col_norm (size_t c);
  /// ||c||_p
  double col_norm (size_t c, double p);
  /// <c1,c2>
  ValType col_dot  (size_t c1, size_t c2) ;
  /// <c, v>
  ValType col_dot  (size_t c, my_vector<ValType>* v);
  /// c1 <- c1 + c2
  int    col_add  (size_t c1, size_t c2) ;
  /// c1 <- c1 - c2
  int    col_sub  (size_t c1, size_t c2) ;
  /// c1 <- s*c1
  int    col_scale(size_t c1, ValType s);
  /// c1 <- c1 - v
  int    col_sub  (size_t c1, my_vector<ValType>* v);
  /// c1 <- c1 + v
  int    col_add  (size_t c1, my_vector<ValType>* v) ;
  /// r <- s*c1
  int    col_scale(size_t c1, ValType s, my_vector<ValType>* r);
  /// r <- ones(1,m) . this == 1'A
  int    col_sum  (my_vector<ValType>* r);
  
  /// ||r||
  double row_norm (size_t r);
  /// ||r||_p
  double row_norm (size_t r, double p);
  /// <r1,r2>
  ValType row_dot  (size_t r1, size_t r2);
  /// <r,v>
  ValType row_dot  (size_t r, my_vector<ValType>* v);

  /// ------------------------ ELEMENTWISE OPS --------------------------------------
  /// return the (i, j) entry of the matrix
  ValType  get (size_t i, size_t j);
  ValType  get (size_t i, size_t j, size_t* posn);
  ValType  operator()(size_t i, size_t j);
  int      set (size_t i, size_t j, ValType val);
  int      set (size_t posn, double val);

  // Vector l_p norms for this matrix
  double norm (double p);
  double norm (char*  p);
  // Apply an arbitrary function elementwise to this matrix
  ValType apply (ValType (* fn)(ValType&));

};

template <typename ValType> 
int SparseMatrix<ValType>::load(char* fname, bool asbin, int typ)
{
  if (asbin)
    return load_bin(fname);
  switch (typ) {
  case M_CCS:
    return load_txt(fname);     // load CCS1 = file.ccs file
  case M_CCS3TXX:
    return load_ccs3(fname, "txx");    // load CCS3, txx 
  case M_CCS3TFN:
    return load_ccs3(fname, "tfn"); 
  case M_MAT:
    return load_matlab(fname);
  case M_HB:
    return load_hb(fname);
  default:
    fprintf(stderr,"SparseMatrix:load() -- Invalid load type requested\n");
    m_state = -1;
    return m_state;
  }
}


/* Loads a CCS matrix from disk */
/* The CCS matrix is specified by a file called, say
 * file.ccs
 * file.ccs has the following contents
 * m n nz
 * name of row_indices file
 * name of col_ptrs file
 * name of values file
 */
template <typename ValType> int SparseMatrix<ValType>::load_txt(char* fname)
{
  char buf[256];
  int r;
  FILE* fp, *fp2;
  if ( !(fp = fopen(fname, "r"))) {
    fprintf(stderr,"SparseMatrix::load_txt() could not open: %s\n", fname);
    m_state = -1; 
    return m_state;
  }
  
  size_t m, n, nz;

  /* Read the dimensions from the file */
  r=fscanf(fp, "%zu %zu %zu", &m, &n, &nz);
  
  setsize (m, n, nz);
  
  /* Now get filenames */
  /* This is the row_indx file name */
  r=fscanf(fp, "%s", buf);

  /* Load the row indices */
  if ( !(fp2 = fopen(buf, "r"))) {
    fprintf(stderr,"SparseMatrix::load_txt() could not open: %s\n", buf);
    m_state = -2;
    return m_state;
  }

  m_rowindx = (size_t*) malloc (sizeof (size_t) * nz);
  size_t* ptr = m_rowindx;
  for (size_t i = 0; i < nz; i++) {
    r=fscanf(fp2, "%zu\n", ptr);
    ++ptr;
  }
  fclose(fp2);

  /* This is the col_ptrs file name */
  r=fscanf(fp, "%s", buf);
  
  /* Load the col_ptrs */
  if ( !(fp2 = fopen(buf, "r"))) {
    fprintf(stderr,"SparseMatrix::load_ccs3() could not open: %s\n", buf);
    free(m_rowindx); m_rowindx = 0; fclose(fp);
    m_state = -3; return m_state;
  }

  m_colptrs = (size_t*) malloc (sizeof (size_t) * (n+1));
  ptr = m_colptrs;

  for (size_t i = 0; i <= n; i++) {
    r=fscanf(fp2, "%zu\n", ptr);
    ++ptr;
  }
  fclose(fp2);

  /* The values file */
  r=fscanf(fp, "%s", buf);
  fclose(fp);

  m_val     = (ValType*) malloc (sizeof (ValType) * nz);
  ValType* ptr2;
  ptr2 = m_val;
  std::ifstream infile; infile.clear();
  // Now read in the actual m_values
  infile.open(buf);
  if (infile.fail()) {
    fprintf(stderr,"SparseMatrix:load_ccs3() could not open: %s\n", buf);
    free(m_rowindx);
    free(m_colptrs);
    m_state = -4; return m_state;
  }
  
  for (size_t i = 0; i < nz; i++) {
    infile >> *ptr2;
    ++ptr2;
  }
  infile.close();
  m_state = 0; return m_state;
}

template <typename ValType>
int SparseMatrix<ValType>::load_bin(char* fname)
{
  m_fname = std::string(fname);
  FILE* fp = fopen (fname, "rb");
  if (fp == 0) {
    fprintf(stderr,"SparseMatrix:load_bin() Could not open: %s\n", fname);
    m_state = -1; return m_state;
  }

  size_t m;
  size_t n;
  size_t nz;
  int    mt;

  if (read_file_header_bin(fp, &m, &n, &nz, &mt, M_CCS) < 0) {
    fclose(fp);
    m_state = -2;
    return m_state;
  }
  m_rows=m; m_cols=n; m_nz=nz;
  setsize (m_rows, m_cols, m_nz);
  fprintf(stderr, "Found a matrix (%zu %zu %zu) in %s\n", m, n, nz, fname);
  m_colptrs = (size_t*) malloc (sizeof (size_t) * (n+1));
  m_rowindx = (size_t*) malloc (sizeof (size_t) * nz);
  m_val     = (ValType*) malloc (sizeof (ValType) * nz);
   
  // Read the colptrs
  if (fread(m_colptrs, sizeof(size_t), n+1, fp) != n+1) {
    fprintf(stderr,"SparseMatrix:load_bin() -- Error reading colptrs\n");
    m_state = -3;
    return m_state;
  }

  // Read the row indices
  // TODO: if nz happens to be very big, we break up the reads
  if (fread(m_rowindx, sizeof(size_t), nz, fp) != nz) {
    fprintf(stderr,"SparseMatrix:load_bin() -- Error reading rowindices\n");
    m_state = -3;
    return m_state;
  }

  // Read the nonzeros themselves
  if (fread(m_val, sizeof(ValType), nz, fp) != nz) {
    fprintf(stderr,"SparseMatrix:load_bin() -- Error reading nonzeros\n");
    m_state = -3;
    return m_state;
  }
  fprintf(stderr, "Finished reading matrix\n", m, n, nz);
  fclose(fp);
  m_state = 0;
  return m_state;
}

template <typename ValType> int SparseMatrix<ValType>::load_ccs3(char* prefix, char* txx)
{
  // This function assumes that m_type is already set
  m_fname = prefix;
  std::string m_type = std::string(txx);

  // Set up the file names that we are gonna open
  std::string dim(prefix);
  std::string rows_file(dim + "_row_ccs");
  std::string cols_file(dim + "_col_ccs");
  std::string nz_file (dim + "_"+ m_type + "_nz");
  
  dim +=  "_dim";
  
  std::ifstream infile;
  infile.clear();
  infile.open(dim.data());
  if (infile.fail()) {
    fprintf(stderr,"SparseMatrix::load_ccs3() could not open: %s\n", dim.data());
    return -1;
  }
  
  size_t m_rows, m_cols;
  size_t m_nz;

  infile >> m_rows >> m_cols >> m_nz;
  infile.close();
  infile.clear();

  // Allocate the arrays
  m_colptrs = (size_t*) malloc (sizeof (size_t) * (m_cols+1));
  m_rowindx = (size_t*) malloc (sizeof (size_t) * m_nz);
  m_val     = (ValType*) malloc (sizeof (ValType) * m_nz);

  
  setsize (m_rows, m_cols, m_nz);

  // Now read in all the columnpointers
  infile.open(cols_file.data());
  if (infile.fail()) {
    fprintf(stderr,"SparseMatrix::load_ccs3() could not open: %s\n" , cols_file.data());
    m_state = -2;
    return m_state;
  }

  // Make sure the UB on i is correct.
  for (size_t i = 0; i < m_cols; i++) 
    infile >> m_colptrs[i];
  infile.close();
  infile.clear();
  m_colptrs[m_cols] = m_nz; // Add the convenience value

  infile.open(rows_file.data());
  if (infile.fail()) {
    fprintf(stderr,"SparseMatrix::load_ccs3() could not open: %s\n", rows_file.data());
    m_state = -3;
    return m_state;
  }

  for (size_t i = 0; i < m_nz; i++)
    infile >> m_rowindx[i];
  infile.close();
  infile.clear();
  
  // Now read in the actual m_values
  infile.open(nz_file.data());
  if (infile.fail()) {
    fprintf(stderr,"SparseMatrix:load_ccs3() could not open: %s\n", nz_file.data());
    m_state = -4;
    return m_state;
  }
  
  for (size_t i = 0; i < m_nz; i++)
    infile >> m_val[i];
  infile.close();
  m_state = 0;
  return m_state;
}

/*
 * Load a (i,j,val) file, where the indices are sorted according to 'j' and 'i'
 */
template <typename ValType> int SparseMatrix<ValType>::load_matlab(char* fname)
{
  FILE* fp;
  if ( !(fp = fopen(fname, "r"))) {
    fprintf(stderr,"SparseMatrix::load_matlab() could not load: %s\n", fname);
    m_state = -1;
    return m_state;
  }
  int r;
  char buf[1024];

  // REMOVE THIS DEPENDENCY
  sprintf(buf, "%s.dim", fname);
  FILE* fpd = fopen(buf, "r");
  if (!fpd) {
    std::cerr << "Dimension file: " << buf << " is missing\n";
    std::cerr << "You need to supply m n nz in the dimension file\n";
    m_state = -2;
    return m_state;
  }

  size_t m, n, nz;
  r=fscanf(fpd, "%zu %zu %zu", &m, &n, &nz);
  setsize(m,n,nz);
  // Allocate the arrays
  m_colptrs = (size_t*) malloc (sizeof (size_t) * (n+1));
  m_rowindx = (size_t*) malloc (sizeof (size_t) * nz);
  m_val     = (ValType*) malloc (sizeof (ValType) * nz);

  // The data structs are already allocated. so we can start fillin em up
  // Remember, matlab indexes matrices from 1 instd. of 0, so have to do the
  // appropriate translation.
  size_t idx = 0;
  long pcolumn = -1;
  m_colptrs[0] = 0;		// Init it!

  size_t col, row;
  ValType val;
  std::cout << "Beginning to load matrix...\n";
  while (!feof(fp)) {
    // I am assuming that matlb sparse matrix as a text file is saved as
    // (rowx, col1, val1)
    // (rowy, col1, val2)
    // ...
    // (rowz, colk, valk)
    // If it is the other way round, just call transpose in matlab OR use the
    // class CRS to load the matrix and transpose it....or just use the load_mat_rowmajor
    r=fscanf(fp, "%zu %zu %lf", &row, &col, &val);
    if (col > (long)ncols() || row > (long)nrows()) {
      std::cerr << "Input matrix too big, stopping at ("
                << row << " , " << col << ") entry\n";
      break;
    }
    // Since we are readin in matlab indices, subtract one
    --col;               
    --row;
    

    if (col > pcolumn) {
      for (long c = pcolumn + 1; c <= col; c++)
        m_colptrs[c] = idx;
      pcolumn = col;
    }
    m_rowindx[idx] = row;
    m_val[idx]     = val;
    ++idx;
  }
  
  m_colptrs[n] = nz + 1;     // Convenience value
  //std::cerr << "Built matrix " << m << " x " << (doc+1) << " with " << (idx-1) << " nonzeros\n";
  m_state = 0;
  return m_state;
}

template <typename ValType> int SparseMatrix<ValType>::load_hb(char* fname)
{
  FILE* fp = fopen(fname, "r");
  if (!fp) {
    fprintf(stderr,"SparseMatrix::load_hb() could not open: %s\n", fname);
    m_state = -1;
    return m_state;
  }
  int r;
  size_t m,n,nz;
  char buf[1024];
  // We skip over all the strings in the file.
  r=fscanf(fp, "%72c%*s", buf);
  r=fscanf(fp, "%*s");
  r=fscanf(fp, "%*s%zu%zu%zu%*d", &m, &n, &nz);
  
  //std::cout << "Read file header (" << nrow << "," << ncol << "," << nnz << ")\n";
  /* as of now i am skipping the data format line
   * might implement it if the demand arises.
   */
  r=fscanf(fp, "%*s %*s %*s %*s");

  setsize(m,n,nz);
  // Allocate the arrays
  m_colptrs = (size_t*) malloc (sizeof (size_t) * (n+1));
  m_rowindx = (size_t*) malloc (sizeof (size_t) * nz);
  m_val     = (ValType*) malloc (sizeof (ValType) * nz);
  // Read the col pointers
  for (size_t i = 0; i <= n; i++) 
    r=fscanf(fp, "%zu", &m_colptrs[i]);

  // We need to subtract 1 from the values, because traditional HB format
  // has arrays starting from 1
  for (size_t i = 0; i < n; i++)
    m_colptrs[i] -= 1;

  m_colptrs[n] = nz;

  for (size_t i = 0; i < nz; i++) {
    r=fscanf(fp, "%zu", &m_rowindx[i]);
    m_rowindx[i]--;
  }

  for (size_t i = 0; i < nz; i++) {
    r=fscanf(fp, "%lf", &m_val[i]);
  }
  fclose(fp);
  m_state = 0; 
  return m_state;
}

template <typename ValType>
int SparseMatrix<ValType>::read_file_header_bin(FILE* fp, size_t* m, size_t* n, size_t* nz, int* mt, int emt)
{
  // read the dimension info
  if (fread(m,  sizeof(size_t), 1, fp) != 1) {
    fprintf(stderr,"SparseMatrix::read_file_header_bin() -- Error reading  (numrows) from file\n");
    m_state = -2;
    return m_state;
  }

  if (fread(n,  sizeof(size_t), 1, fp) != 1) {
    fprintf(stderr,"SparseMatrix::read_file_header_bin() -- Error reading N (numcols) from file\n");
    m_state = -2;
    return m_state;
  }

  if (fread(nz, sizeof(size_t), 1, fp) != 1) {
    fprintf(stderr,"SparseMatrix::read_file_header_bin() -- Error reading NZ from file\n");
    m_state = -2;
    return m_state;
  }
  return 0;

  if (fread(mt, sizeof(int), 1, fp) != 1) {
    fprintf(stderr,"SparseMatrix::read_file_header_bin() -- Error reading matrix type from file\n");
    m_state = -2;
    return m_state;
  }

  if (*mt != emt) {
    fprintf(stderr, "SparseMatrix::read_file_header_bin() -- Input file is not in expected format!\n");
    fprintf(stderr,"Expected: %d GOT: %d\n", emt, *mt);
    m_state = -1;
    return m_state;
  }

  m_state = 0;
  return m_state;
}

#endif
