// -*- c++ -*-
// File: util.h
// Author: Suvrit Sra
// Date: 14 Jan 04

// Copyright (C) 2003 Suvrit Sra (suvrit@cs.utexas.edu)

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


#ifndef UTIL_H
#define UTIL_H

#include <gsl/gsl_matrix.h>
#include <gsl/gsl_rng.h>
#include <ostream>


/**
 * MATRIX, I/O FUNCTIONS. These function are wrappers around existing GSL
 * function since the GSL functions for I/O assume that the size of the
 * matrix is known in advance, which need not be the case. The matrix
 * expected by these functions is of the form:
 * M N
 * row1
 * row2
 * etc.
 * where M, N denote the number of rows and columns in the matrix
 * respectively. The matrices are all stored and accessed in C-style
 * (row-major) unless otherwise specified.
 */

// Read an augmented GSL matrix from a file of the specified name.
// This function not only reads the matrix, it allocate a gsl_matrix and
// returns a pointer to it (with the contents being read from the file.
gsl_matrix* read_gsl_matrix(char* filename);

// This function can be used to speed up the reading of a matrix that is
// STORED in the format described above (which is clearly the row-major
// format) and reading it and converting it to column major format (done by
// doing a transpose operation and then tweaking the size1, size2 fields by
// hand. If the input matrix is stored on the disk in column major format
// then one should use read_gsl_matrix_raw
gsl_matrix* read_gsl_matrix_fortran(char* filename);

// Basically all this function does is, is to read in the dimensions of the
// matrix into memory and then just read the entire m*n array into
// memory...which if was stored in col major order will now be col major in
// memory!
gsl_matrix* read_gsl_matrix_raw(char* filename);


// The following functions are just binary file counterparts of the
// functions above...
gsl_matrix* fread_gsl_matrix        (char* filename);
gsl_matrix* fread_gsl_matrix_fortran(char* filename);
gsl_matrix* fread_gsl_matrix_raw    (char* filename);


// The following functions read the matrix into a preallocated GSL matrix
// of appropriate dimensions. If dimensions do not match an error will
// occur. 

int read_gsl_matrix        (gsl_matrix*, FILE*);
int read_gsl_matrix_raw    (gsl_matrix*, FILE*);
int read_gsl_matrix_fortran(gsl_matrix*, FILE*);

int fread_gsl_matrix       (gsl_matrix*, FILE*);
int fread_gsl_matrix_raw   (gsl_matrix*, FILE*);
int fread_gsl_matrix_fortran(gsl_matrix*,FILE*);

int read_gsl_matrix        (gsl_matrix*, char*);
int read_gsl_matrix_raw    (gsl_matrix*, char*);
int read_gsl_matrix_fortran(gsl_matrix*, char*);

int fread_gsl_matrix       (gsl_matrix*, char*);
int fread_gsl_matrix_raw   (gsl_matrix*, char*);
int fread_gsl_matrix_fortran(gsl_matrix*,char*);

int read_gsl_matrix        (gsl_matrix*, std::istream&);
int read_gsl_matrix_raw    (gsl_matrix*, std::istream&);
int read_gsl_matrix_fortran(gsl_matrix*, std::istream&);


// The following functions write out the specified GSL matrix onto a file
// or a filestream....they behave as exact counter parts to the read
// functions listed above....
int write_gsl_matrix        (gsl_matrix*, FILE*);
int write_gsl_matrix_raw    (gsl_matrix*, FILE*);
int write_gsl_matrix_fortran(gsl_matrix*, FILE*);

int fwrite_gsl_matrix       (gsl_matrix*, const char*);
int fwrite_gsl_matrix       (gsl_matrix*, FILE*);
int fwrite_gsl_matrix_raw   (gsl_matrix*, FILE*);
int fwrite_gsl_matrix_fortran(gsl_matrix*,FILE*);

int write_gsl_matrix        (gsl_matrix*, std::ostream&);
int write_gsl_matrix_raw    (gsl_matrix*, std::ostream&);
int write_gsl_matrix_fortran(gsl_matrix*, std::ostream&);


/*
 * UTILITY FUNCTIONS
 */

// A uniformly random matrix with entries in [0..1]
gsl_matrix* random_matrix (int m, int n);

// A random matrix, with various type of random distributions supported by
// means of allowing one to pass a gsl_rng object
gsl_matrix* random_matrix (size_t m, size_t n, gsl_rng*);

// Converts a row major matrix to col major and vice versa!
gsl_matrix* toggle_rowcol_major(gsl_matrix*);


/* ******************************************
 * OPERATIONS, FUNCTIONALS, ACTIONS etc...
 * ******************************************/

// If no = 0, we use the default norm, which is the L_2 norm for
// normlizing...else we normalize as per the specified norm....
int normalize_matrix_cols(gsl_matrix*, int no = 0);
int normalize_matrix_rows(gsl_matrix*, int no = 0);

double fnorm(gsl_matrix*   );	// Frobenius norm
double norm (gsl_matrix*, int p);	// p, norm.

/* **********************************************
 * Interface and utility functions for converting to SparseMatrix
 * Format... and other such functions...
 * **********************************************/

// Writes out the given gsl matrix as a sparse matrix in CCS format, with
// file name being specified by prefix.
int  gsl_to_sparse(gsl_matrix*, char* prefix);
long nnzero       (gsl_matrix*);

/* ************************************
 * TOTALLY MISCELLANEOUS FUNCTIONS....
 * ************************************/
// Converts a GSL matrix to a PGM image!!! Useful when doing simple image
// processing work on PGM's that have been read in as a GSL matrix
// See the tools pgmtogsl.cc pgmtogsl.pl
int gsl_matrix_to_pgms(gsl_matrix*, char*, int r, int c, int scale);

// Deprecated functions, held in here till code over-haul happens
int         printon(gsl_matrix*, std::ostream&);

// Change a row major matrix to a column major matrix
gsl_matrix* prepare_for_clapack(gsl_matrix*);


// Apply scalar function to each element of the matrix and return result
// in the second matrix
int apply(double (*f)(double), gsl_matrix* src, gsl_matrix* dest);


/* ******************************************
 * Utility functions for arrays of doubles
 * *****************************************/

#endif
