// util.cc: Implements functions in util.h
// Copyright (C) 2003 Suvrit Sra

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#include <cstdio>
#include <cmath>
#include <iostream>
#include <fstream>
#include "util.h"

/**
 * Function to read in a gsl matrix from file specified by the global
 * 'fname'. This function allocates the matrix and returns a pointer to the
 * allocated matrix.
 */
gsl_matrix* read_gsl_matrix(char* fname)
{
  FILE* fp = fopen(fname, "r");
  if (fp == NULL) {
    std::cerr << "read_gsl_matrix: Could not open " << fname << std::endl;
    return 0;
  }

  // This is to allow future additions where the first line of the matrix
  // file might have some other useful information aboutthe matrix...such
  // as whether matrix is triangular, symmetric, diagonal, rowmajor etc., etc.
  char line[1024];
  if (!fgets(line, 1000, fp)) {
    std::cerr << "read_gsl_matrix: Error reading matrix\n";
    return 0;
  }
  
  int m, n;
  sscanf(line, "%d %d", &m, &n);
  gsl_matrix* mat = gsl_matrix_alloc(m,n);
  if (mat == 0) {
    std::cerr << "read_gsl_matrix: Error allocating memory. Check matrix format " << fname << std::endl;
    return 0;
  }
  gsl_matrix_fscanf(fp, mat);
  fclose(fp);
  //cerr << mat->size1 << "***" << mat->size2 << endl;
  //gsl_matrix_fprintf(stdout, mat, "%f");
  return mat;
}

/**
 * This is the function to use for reading a matrix for use with CLAPACK
 * routines, it automatically reads in a matrix stored in row-major (C)
 * format into a FORTRAN format....thus avoiding wrapper calls and wasted
 * memory for a copy of the input matrix
 */
gsl_matrix* read_gsl_matrix_fortran(char* fname)
{
  gsl_matrix* tmp = read_gsl_matrix(fname);
  gsl_matrix* mat = 0;

  if (tmp != 0) {
    mat = gsl_matrix_alloc(tmp->size2, tmp->size1);
    gsl_matrix_transpose_memcpy(mat, tmp);
    size_t t = mat->size1;
    mat->size1 = mat->size2;
    mat->size2 = t;
    gsl_matrix_free(tmp);
  }
  return mat;
}

int read_gsl_matrix(gsl_matrix* mat, char* fname)
{
   FILE* fp = fopen(fname, "r");
   if (fp == NULL) {
     std::cerr << "read_gsl_matrix: Could not open " << fname << std::endl;
     return -1;
   }
   
   int m, n;
   fscanf(fp, "%d %d", &m, &n);
   if ((int)mat->size1 != m or (int)mat->size2 != n) {
     std::cerr << "read_gsl_matrix: Incorrect matrix size " << m << " x " << n << std::endl;
     return -1;
   }
   gsl_matrix_fscanf(fp, mat);
   fclose(fp);
   return 0;
}


/*
 * Read matrix as 1-D array into memory
 */

gsl_matrix* read_gsl_matrix_raw(char* file)
{
   FILE* fp = fopen(file, "r");
   if (fp == NULL) {
     std::cerr << "read_gsl_matrix: Could not open " << file << std::endl;
     return 0;
   }
   
   size_t m, n;
   fscanf(fp, "%u %u", &m, &n);
   gsl_matrix* mat = gsl_matrix_alloc(m, n);

   for (size_t i = 0; i < m*n; i++)
     fscanf(fp, "%lf", &mat->data[i]);
   
   fclose(fp);
   return mat;
}

/**
 * Does and fwrite of a GSL matrix.
 * TBD: Include error checking....
 */
int write_gsl_matrix(gsl_matrix* m, char* fname)
{
  FILE* fp = fopen(fname, "wb");
  if (fp == NULL) {
    std::cerr << "write_gsl_matrix: could not open " << fname << std::endl;
    return -1;
  }

  gsl_matrix_fwrite(fp, m);
  return 0;
}

/*
 * TBD: Include error checking....
 * do a binary (fread) of a prealloced gsl matrix
 */
int fread_gsl_matrix(gsl_matrix* m, char* fname)
{
  FILE* fp = fopen(fname, "rb");
  if (fp == NULL) {
    std::cerr << "fread_gsl_matrix: could not open " << fname << std::endl;
    return -1;
  }
  return   gsl_matrix_fread(fp, m);
}

/*
 * TBD: Include error checking....
 */
gsl_matrix* fread_gsl_matrix(char* file)
{
  FILE* fp = fopen(file, "rb");
  if (!fp) {
    std::cerr << "gsl_matrix* fread_gsl_matrix(" << file << "): failed\n";
    return 0;
  }
  unsigned int m, n;
  fread(&m, sizeof(size_t), 1, fp);
  fread(&n, sizeof(size_t), 1, fp);
  gsl_matrix* mat = gsl_matrix_alloc(m, n);
  if (mat == 0) {
    std::cerr << "gsl_matrix* fread_gsl_matrix(" << file << "): Error allocating matrix\n";
    return 0;
  }
  gsl_matrix_fread(fp, mat);
  return mat;
}


// Of course, since it is fread, it is raw....
gsl_matrix* fread_gsl_matrix_raw(char* file)
{
  return fread_gsl_matrix(file);
}

// fortran style fread?
gsl_matrix* fread_gsl_matrix_fortran(char* file)
{
  gsl_matrix* tmp = fread_gsl_matrix(file);
  gsl_matrix* mat = 0;
  if (tmp != 0) {
    mat = gsl_matrix_alloc(tmp->size2, tmp->size1);
    gsl_matrix_transpose_memcpy(mat, tmp);
    size_t t = mat->size1;
    mat->size1 = mat->size2;
    mat->size2 = t;
    gsl_matrix_free(tmp);
  }
  return mat;
}

/*
 * TBD: Include error checking....
 */
/*
 * fwrite a gsl matrix. This function also stores the size of the matrix in
 * the binary file allowing for the fread_gsl_matrix function above...
 */
int fwrite_gsl_matrix(gsl_matrix* mat, const char* file)
{
  FILE* fp = fopen(file, "wb");
  if (!fp) {
    std::cerr << "int fwrite_gsl_matrix(" << file << "): file open failed\n";
    return -1;
  }
  fwrite(&mat->size1, sizeof(size_t), 1, fp);
  fwrite(&mat->size2, sizeof(size_t), 1, fp);
  return gsl_matrix_fwrite(fp, mat);
}


int fwrite_gsl_matrix(gsl_matrix* mat, FILE* fp)
{
  if (!fp) {
    std::cerr << "int fwrite_gsl_matrix(): file error\n";
    return -1;
  }
  fwrite(&mat->size1, sizeof(size_t), 1, fp);
  fwrite(&mat->size2, sizeof(size_t), 1, fp);
  return gsl_matrix_fwrite(fp, mat);
}
/**
 * Returns a matrix with random entries. Useful function. Uses simulation strength random number
 * generator.
 */
gsl_matrix* random_matrix(int m, int n)
{
  gsl_matrix* mat = gsl_matrix_alloc(m, n);
  gsl_rng* r = gsl_rng_alloc(gsl_rng_mt19937);
  gsl_rng_set(r, time(0));
  if (mat != 0) {
    for (int i = 0; i < m*n; i++)
      mat->data[i] = gsl_rng_uniform(r);
  }
  return mat;
  gsl_rng_free(r);
}

/**
 * Print out contents of gsl matrix A onto ostream os
 */
int printon(gsl_matrix* A, std::ostream& os)
{
  char buf[255];

  os << A->size1 << " " << A->size2 << std::endl;
  for (uint i = 0; i < A->size1; i++) {
    for (uint j = 0; j < A->size2; j++) {
      sprintf(buf, "%.6f", gsl_matrix_get(A, i, j));
      //os << gsl_matrix_get(A, i, j) << " ";
      os << buf << " ";
    }
    os << std::endl;
  }
  return 0;
}

/**
 * Writes out each column of the input matrix g into files
 */
int gsl_matrix_to_pgms(gsl_matrix* g, char* prefix, int r, int c, int scale)
{
  if (r*c != (int)g->size1) {
    std::cerr << "gsl_to_pgm: " << r << " x " << c << " != " << g->size1 << std::endl;
    return -1;
  }
  
  char filename[255];
  for (int j = 0; j < (int)g->size2; j++) {
    sprintf(filename, "%s.%d.pgm", prefix, j);
    FILE* fp = fopen(filename, "w");
    
    // Write pgm header now
    fprintf(fp, "P5\n");		// pgm magic number
    fprintf(fp, "%d %d\n", c, r);	// cols, rows (width, height)
    fprintf(fp, "255\n");		// maxgray level
    
    for (int h = 0; h < r; h++) {
      for (int w = 0; w < c; w++)
	fprintf(fp, "%c", (unsigned char)(gsl_matrix_get(g, h*c+w, j)*scale));
    }
    fclose(fp);
  }
  return 0;

}

/**
 * Returns number of non-zeros in matrix A
 */
long nnzero(gsl_matrix* A)
{
  long res = 0;
  for (size_t i = 0; i < A->size1*A->size2; i++)
    if (A->data[i] != 0) ++res;
  return res;
}

/**
 * @args: gsl_matrix* A -- Input matrix
 * @args: char* prefix  -- Filename prefix 
 */
int gsl_to_sparse(gsl_matrix* A, char* prefix)
{
  size_t cols = A->size2;
  size_t rows = A->size1;

  std::string dim(prefix);
  std::string rows_file(dim + "_row_ccs");
  std::string cols_file(dim + "_col_ccs");
  std::string nz_file  (dim + "_txx_nz");

  dim += "_dim";

  std::ofstream ofile(dim.data());
  //ofile.open(dim.data());
  if (ofile.fail()) {
    std::cerr << "gsl_to_sparse: Error opening " << dim << std::endl;
    return -1;
  }

  long nz = nnzero(A);

  ofile << rows << " " << cols << " " << nz << std::endl;
  ofile.close();
  ofile.clear();

  ofile.open(rows_file.data());
  std::ofstream colptrs;
  colptrs.open(cols_file.data());

  if (ofile.fail() or colptrs.fail()) {
    std::cerr << "gsl_to_sparse: Error opening file\n";
    return -1;
  }

  std::ofstream nzfile;
  nzfile.open(nz_file.data());
  if (nzfile.fail()) {
    std::cerr << "gsl_to_sparse: Error opening file " << nz_file << "\n";
    return -1;
  }

  // Now we write out the CCS matrix appropriately.
  int colcnt = 0;
  colptrs << colcnt << " "; 	// Start of col_ccs file

  for (size_t i = 0; i < rows; i++) {
    for (size_t j = 0; j < cols; j++) {
      double x = gsl_matrix_get(A, i, j);
      if (x != 0) {
	nzfile << x << "\n";
	ofile << i << "\n";
	++colcnt;
      }
    }
    colptrs << colcnt << "\n";
  }

  colptrs << nz << std::endl;
  ofile.close();
  colptrs.close();
  nzfile.close();
  return 0;
}

/**
 * This function is a wrapper for converting a C-style matrix to a FORTRAN
 * style matrix so that we can pass it to CLAPACK routines. Nothing else
 * prolly needs to be changed before calling a CLAPACK routine!
 **/
gsl_matrix* prepare_for_clapack(gsl_matrix* tmp)
{
  gsl_matrix* mat = gsl_matrix_alloc(tmp->size2, tmp->size1);
  gsl_matrix_transpose_memcpy(mat, tmp);
  // gsl_matrix_free(tmp); // We leave it untouched...
  size_t s1 = mat->size1;
  mat->size1 = mat->size2;	// BAD LOW LEVEL changes to a GSL mat!!
  mat->size2 = s1;
  return mat;
}

double fnorm(gsl_matrix* X)
{

  if (X == 0) {
    std::cerr << "Factorization::fnorm(X) called for unallocated X\n";
    return -1;
  }

  double no = 0.0;
  for (uint i = 0; i < X->size1 * X->size2; i++) {
    double t = X->data[i];
    no += (t*t);
  }
  return sqrt(no);
}

/**
 * apply
 */

int apply(double (*f)(double), gsl_matrix* src, gsl_matrix* dest)
{
  if (dest->size1 != src->size1 or dest->size2 != src->size2)
    return -1;

  for (size_t i = 0; i < src->size1*src->size2; i++)
    dest->data[i] = f(src->data[i]);
  return 0;
}


