// File: mkcplex.cc
// Author: Suvrit Sra
// (c) 2004, Suvrit Sra


#include "util.h"

#include <iostream>
#include <ostream>

#include <cstdio>
#include <cstdlib>


void outputTriangle (uint tn, uint i, uint j, uint k, gsl_matrix* d);
void generateLP(gsl_matrix*);
void generateQP(gsl_matrix*);
void common(gsl_matrix*);
void bounds(gsl_matrix*, int = 2);
void printHeader();
uint indexOf (uint i, uint j, uint N);

int main (int argc, char** argv) 
{

  // Type 1 means L1 LP
  // Type 2 means QP
  // Type ?
  if (argc != 3) {
    std::cerr << "mkcplex: <type=1 or 2> <matrix>\n";
    return 1;
  }

  
  gsl_matrix* d = read_gsl_matrix(argv[2]);

  if (d == 0) {
    std::cerr << "mkcplex: Could not open " << argv[2] << "\n";
    return -2;
  }

  // Now that d is loaded we do the processing.
  int type = atoi(argv[1]);

  printHeader();

  if (type == 1)
    generateLP(d);
  else 
    generateQP(d);

  return 0;
}

void printHeader()
{
  std::cout << "\\Metric Nearness LP\n"
            << "\nMinimize\n";

}

void generateQP (gsl_matrix* d) 
{
  std::cout << "  obj: [ ";
  
  uint N = d->size1 * (d->size1 - 1) / 2;

  for (uint i = 0; i < N-1; i++) {
    if ( (i+1) % 9 == 0)
      std::cout << "\n\t";
    std::cout << "x" << (i+1) << "^2 + ";
    
  }
  std::cout << "x" << N << "^2 ] / 2\n";

  // Ae <= b
  common(d);
  // bounds on the variables ...
  bounds(d);
  

  std::cout << "end\n";

}

/**
 * Function prints out the bounds section onthe variables.
 * Default arg. prints out bounds info for QP.
 */
void bounds (gsl_matrix* d, int lporqp)
{
  uint N = d->size1 * (d->size1-1)/2;
  if (lporqp == 1) 
    N = 2*N;

  std::cout << "Bounds\n";
  for (uint i = 0; i < N; i++) {
    std::cout << "\tx" << (i+1) << " free\n";
  }
}

/**
 * Generates the cplex lp file for L1 LP
 */
void generateLP (gsl_matrix* d) 
{
  std::cout << "  obj: ";
  
  uint N = d->size1 * (d->size1 - 1) / 2;

  for (uint i = 0; i < N-1; i++) {
    if ( (i+1) % 13 == 0)
      std::cout << "\n\t";
    std::cout << "x" << (N + i + 1) << " + ";
    
  }
  std::cout << "x" << 2*N << "\n";

  // Ae <= b constraints
  common(d);
  uint n = d->size1;

  uint r = n*(n-1)*(n-2)/2;

  for (uint i = 0; i < N; i++) {
  // e + f >= 0 or equiv. -e - f <= 0
    std::cout << "  c" << (r + 1 + i) 
              << ": " << "-x" << (i+1) 
              << " - " << "x" << (i+N+1) 
              << " <= 0\n";
      
  }
  for (uint i = 0; i < N; i++) {
    // e - f <= 0
    std::cout << "  c" << (r + N + 1 + i) 
              << ": " << "x" << (i+1) 
              << " - " << "x" << (i+N+1) 
              << " <= 0\n";
  }

  bounds(d, 1);
  std::cout << "end\n";
}

void common (gsl_matrix* d) 
{
  std::cout << "Subject to\n";
  uint n = d->size1;

  uint counter = 1;

  for (uint i = 0; i < n; i++) {
    for (uint j = i+1; j < n; j++) {
      for (uint k = j+1; k < n; k++) {
        outputTriangle(counter, i, j, k, d);
        counter += 3;
      }
    } // End for j
  } // End for i

}

void outputTriangle (uint tn, uint i, uint j, uint k, gsl_matrix* d) 
{
  double ab = gsl_matrix_get(d, i, j);
  double bc = gsl_matrix_get(d, j, k);
  double ac = gsl_matrix_get(d, i, k);
  
  uint N = d->size1;

  double b1 = -ab + bc + ac;
  double b2 = ab - bc + ac;
  double b3 = ab + bc - ac;

  uint idx1 = indexOf(i, j, N);
  uint idx2 = indexOf(j, k, N);
  uint idx3 = indexOf(i, k, N);

  std::cout << "  c" << tn << ": "
            << "x" << idx1 << " - x" << idx2 << " - x" << idx3
            << " <= " << b1 << std::endl;
  std::cout << "  c" << (tn+1) << ": "
            << "x" << idx2 << " - x" << idx1 << " - x" << idx3
            << " <= " << b2 << std::endl;
  std::cout << "  c" << (tn+2) << ": "
            << "x" << idx3 << " - x" << idx1 << " - x" << idx2
            << " <= " << b3 << std::endl;
}

uint indexOf (uint i, uint j, uint N)
{
  uint idx = i*(N-1) - (i+1)*i/2 + j;
  return idx;
}
