// File: metricL2FW.cc
// Author: Suvrit Sra
// (c) 2004, Suvrit Sra
// Impl. of l2 metric nearness, with FW ordering of triangles

#include "metricL2FW.h"

/**
 * Perform fast L2 metric nearness
 *
 */
int MetricL2FW::execute()
{
    std::cout << "MetricL2FW::execute() Entered\n";

    uint n = d->size1;
    uint N = n * (n - 1)/2;

    e = new double[N];

    if (e == 0) {
      std::cerr << "MetricL2FW::execute() Out of memory!\n";
      return -2;
    }
    uint r = N * (n - 2);

    memset(e, 0, N * sizeof(double));

    // Allocate z
    if (alloc_dual_variables() < 0) {
      return -2;
    }
    double zsum = 10;
    uint iter    = 1;
    if (maxiter == 0)
        maxiter = 5*N;

    // Record time here
    clock_t elapse = clock();
    uint t;                     // Current triangle


    while (zsum > tol && iter < maxiter) {
      zsum = 0.0;
      t = 0;
      // Go thru all the triangles in the FW order
      for (uint i = 0; i < n-1; i++) {
        for (uint k = i+2; k < n; k++) {
          for (uint j = i+1; j < k; j++) {
            fixOneTriangle(i, j, k, t, zsum);
          }
        }
      }
      ++iter;
    }

    elapse = clock() - elapse;
    unsigned long int flops = iter * r * 20;
    double ns = elapse * 1.0 / CLOCKS_PER_SEC;

    double obj = compute_obj();

    std::cout << "MetricL2FW::execute(): Elapsed time = " 
              << ns   << " secs, "
              << "Iterations:= " << iter
              << ", zsum:= " << zsum << "\n";
    std::cout << "Objective value = " << obj 
              << ", FLOPS = " << flops << std::endl;
        //<< ", RelError = " << relerror << "\n";
    //printon(d, std::cout);
    std::cout << "SUMMARY:" 
              << n << " " << ns 
              << " " << obj << std::endl;
    return 0;
}


/**
 * This procedure is different from that in other files because it
 * fixes all the three directed triangles at the same time. When we enter
 * it, we have i < j < k guaranteed.
 */
double MetricL2FW::fixOneTriangle(uint i, uint j, 
                                  uint k, uint& t, double& zsum) 
{
    double ab, bc, ca;
    double eab, ebc, eca;
    double v1, v2, v3;
    double mu1, mu2, mu3;
    double theta;

    ab = gsl_matrix_get( d, i, j );
    bc = gsl_matrix_get( d, j, k );
    ca = gsl_matrix_get( d, i, k );

    eab = vecget(e, i, j, d->size1);
    ebc = vecget(e, j, k, d->size1);
    eca = vecget(e, i, k, d->size1);
    
    v1 = ca + bc - ab;
    v2 = ca + ab - bc;
    v3 = ab + bc - ca;

    mu1 = (eab - ebc - eca - v1)/3.0;
    theta = std::max(mu1, -z[t]);
    eab -= theta;
    ebc += theta;
    eca += theta;
    z[t] += theta;
    ++t;
    zsum += fabs(theta);

    mu2 = (ebc - eca - eab - v2)/3.0;
    theta = std::max(mu2, -z[t]);
    ebc -= theta;
    eab += theta;
    eca += theta;
    z[t] += theta;
    ++t;
    zsum += fabs(theta);

    mu3 = (eca - eab - ebc - v3)/3.0;
    theta = std::max(mu3, -z[t]);
    eca -= theta;
    ebc += theta;
    eab += theta;
    z[t] += theta;
    ++t;
    zsum += fabs(theta);

    // Store back the eab etc. correctly.
    vecset(e, i, j, eab, d->size1);
    vecset(e, j, k, ebc, d->size1);
    vecset(e, i, k, eca, d->size1);

    return zsum;
}


inline double MetricL2FW::compute_obj ()
{
    double obj = 0;
    uint N = d->size1 * (d->size1 - 1)/2;
    for (uint i = 0; i < N; i++) {
      double tmp = e[i];
      obj += (tmp * tmp);
    }
    return 0.5*obj;
}
