// File: metricL1V.cc
// Author: Suvrit Sra
// (c) 2004, Suvrit Sra
// Implements l1 metric nearness -- vectorized version.

#include "metricL1V.h"

int MetricL1V::execute() 
{
    std::cout << "MetricL1V::execute() Entered\n";
    
    uint N = d->size1 * (d->size1 - 1)/2;
    e      = new double[N];
    lambda = new double[N];
    mu     = new double[N];

    memset (e,      0, N * sizeof(double));
    memset (lambda, 0, N * sizeof(double));
    memset (mu,     0, N * sizeof(double));



    f = new double[N];
    // Now init 'f' to -eps^{-1}
    // This setting is crucial ... and it is an art to select it.
    double param = -scaleEps * gsl_matrix_max(d);
    
    for (uint i = 0; i < N; i++)
      f[i] = param;

    alloc_dual_variables();     // allocates 'z'

    if (z == 0) return -1;
    

    double zsum = 10;
    uint iter    = 1;
    if (maxiter == 0)
        maxiter = 4 * d->size1;

    // Record time here
    clock_t elapse = clock();
    
    uint t;                     // Current triangle
    uint n = d->size1;

    double* b = new double[N];
    
    while (zsum > tol && iter < maxiter) {
      zsum = 0.0;
      t = 0;
      /**
       * First we run the triangles
       */
      for (uint i = 0; i < n; i++) 
        for (uint j = i+1; j < n; j++) 
          for (uint k = j+1; k < n; k++) 
            // Fixes inequalities (ijk), (jki), (kij)
            bregTriangleFix(i, j, k, t, zsum);
      
      /*
       * Then we apply the remaining inequalities.
       */

      // The vector operations below can be replaced by BLASified
      // operations such as dscale etc.
      // b = 0.5*(e+f)
      memcpy  (b, e, N * sizeof(double));
      vecadd  (b, f, N);
      vecscale(b, 0.5, N);
      vecmin  (b, lambda, N);
      vecsub  (lambda, b, N);
      vecsub  (e, b, N);
      vecsub  (f, b, N);

      // b = 0.5*(f - e)
      memcpy  (b, f, N * sizeof(double));
      vecsub  (b, e, N);
      vecscale(b, 0.5, N);
      vecmin  (b, mu, N);
      vecsub  (mu, b, N);
      vecadd  (e, b, N);
      vecsub  (f, b, N);

      ++iter;
    }
    

    elapse = clock() - elapse;
    double ns = elapse * 1.0 / CLOCKS_PER_SEC;

    double obj = compute_obj();

    std::cout << "MetricL1V::execute(): Elapsed time = " 
              << ns   << " secs, "
              << "Iterations:= " << iter
              << ", zsum:= " << zsum << "\n";
    std::cout << "Error value = " << obj << std::endl;
    std::cout << "SUMMARY:" 
              << d->size1 << " " << ns 
              << " " << obj << std::endl;
    
    //printon(d, std::cout);
    return 0;
}


/**
 * This procedure is different from that in other files because it
 * fixes all the three directed triangles at the same time. When we enter
 * it, we have i < j < k guaranteed.
 */
double MetricL1V::bregTriangleFix(uint i, uint j, 
                                  uint k, uint& t, double& zsum) 
{
    double ab, bc, ca;
    double eab, ebc, eca;
    double v1, v2, v3;
    double mu1, mu2, mu3;
    double theta;

    ab = gsl_matrix_get( d, i, j );
    bc = gsl_matrix_get( d, j, k );
    ca = gsl_matrix_get( d, i, k );

    eab = vecget(e, i, j, d->size1);
    ebc = vecget(e, j, k, d->size1);
    eca = vecget(e, i, k, d->size1);
    
    v1 = ca + bc - ab;
    v2 = ca + ab - bc;
    v3 = ab + bc - ca;

    mu1 = (eab - ebc - eca - v1)/3.0;
    theta = std::max(mu1, -z[t]);
    eab -= theta;
    ebc += theta;
    eca += theta;
    z[t] += theta;
    ++t;
    zsum += fabs(theta);

    mu2 = (ebc - eca - eab - v2)/3.0;
    theta = std::max(mu2, -z[t]);
    ebc -= theta;
    eab += theta;
    eca += theta;
    z[t] += theta;
    ++t;
    zsum += fabs(theta);

    mu3 = (eca - eab - ebc - v3)/3.0;
    theta = std::max(mu3, -z[t]);
    eca -= theta;
    ebc += theta;
    eab += theta;
    z[t] += theta;
    ++t;
    zsum += fabs(theta);

    // Store back the eab etc. correctly.
    vecset(e, i, j, eab, d->size1);
    vecset(e, j, k, ebc, d->size1);
    vecset(e, i, k, eca, d->size1);

    return zsum;
}

double MetricL1V::compute_obj()
{
  double obj = 0.0;
  double N = d->size1 * (d->size1 - 1)/2;

  for (uint i = 0; i < N; i++)
    obj += fabs(e[i]);

  return obj;
}


