// File: metricL1.cc
// Author: Suvrit Sra
// (c) 2004, Suvrit Sra
// Implements l1 metric nearness

#include "metricL1.h"

int MetricL1::execute() 
{
    std::cout << "MetricL1::execute() Entered\n";
    
    uint N = d->size1;
    //uint r = N*(N-1)*(N-2)/2;

    f = gsl_matrix_alloc(N, N);

    // Now init 'f' to -eps^{-1}
    // This setting is crucial ... and it is an art to select it.
    double param = -scaleEps * gsl_matrix_max(D);
    gsl_matrix_set_all(f, param);
    gsl_matrix_set_all(d, 0);   // we store 'e' in 'd' itself.
    alloc_dual_variables();     // allocates 'z'

    if (z == 0) {
        std::cerr << "MetricL1::execute(): Out of memory!\n";
        return -1;
    }
    
    gsl_matrix* lambda = gsl_matrix_calloc(N, N);
    gsl_matrix* mu     = gsl_matrix_calloc(N, N);

    double zsum = 10;
    uint iter    = 1;
    if (maxiter == 0)
        maxiter = 5*N;

    // Record time here
    clock_t elapse = clock();
    
    uint t;                     // Current triangle

    gsl_matrix* b = gsl_matrix_alloc(N, N);

    while (zsum > tol && iter < maxiter) {
        zsum = 0.0;
        t = 0;
        
        /**
         * First we run the triangles
         */
        for (uint i = 0; i < N; i++) {
            for (uint j = i+1; j < N; j++) {
                for (uint k = j+1; k < N; k++) {
                    bregTriangleFix(i, j, k, z[t], zsum);
                    ++t;
                    bregTriangleFix(j, k, i, z[t], zsum);
                    ++t;
                    bregTriangleFix(k, i, j, z[t], zsum);
                    ++t;
                }
            }
        }

        /*
         * Then we apply the remaining inequalities.
         */

        // b = 0.5*(e+f)
        gsl_matrix_memcpy(b, d);
        gsl_matrix_add(b, f);
        gsl_matrix_scale(b, 0.5);

        for (uint i = 0; i < b->size1*b->size1; i++) {
          b->data[i] = std::min(b->data[i], lambda->data[i]);
        }
        gsl_matrix_sub(lambda, b); // lambda -= theta
        gsl_matrix_sub(d, b);
        gsl_matrix_sub(f, b);

        // b = 0.5*(f - e)
        gsl_matrix_memcpy(b, f);
        gsl_matrix_sub(b, d);
        gsl_matrix_scale(b, 0.5);

        for (uint i = 0; i < b->size1*b->size1; i++) {
          b->data[i] = std::min(b->data[i], mu->data[i]);
        }

        gsl_matrix_sub(mu, b);
        gsl_matrix_add(d, b);
        gsl_matrix_sub(f, b);

        ++iter;
    }
    
    gsl_matrix_add(d, D);

    elapse = clock() - elapse;
    double ns = elapse * 1.0 / CLOCKS_PER_SEC;

    double obj = compute_obj();

    std::cout << "MetricL1::execute(): Elapsed time = " 
              << ns   << " secs, "
              << "Iterations:= " << iter
              << ", zsum:= " << zsum << "\n";
    std::cout << "Error value = " << obj << std::endl;
    std::cout << "SUMMARY:" 
              << N << " " << ns 
              << " " << obj << std::endl;
    
    printon(d, std::cout);
    return 0;
}


double MetricL1::bregTriangleFix(uint i, uint j, uint k, 
                                       double& err, double& zsum) 
{
    double ab, bc, ca;
    double eab, ebc, eca;

    double del;

    ab = gsl_matrix_get( D, i, j );
    bc = gsl_matrix_get( D, j, k );
    ca = gsl_matrix_get( D, i, k );

    eab = gsl_matrix_get( d, i, j );
    ebc = gsl_matrix_get( d, j, k );
    eca = gsl_matrix_get( d, i, k );
                    
    del = ca + bc - ab;
    double mu = (eab - ebc - eca - del)/3.0;
    double theta = std::max(mu, -err);

    eab -= theta;
    ebc += theta;
    eca += theta;

    err += theta;

    gsl_matrix_set( d, i, j, eab);
    gsl_matrix_set( d, j, i, eab);

    gsl_matrix_set( d, j, k, ebc);
    gsl_matrix_set( d, k, j, ebc);

    gsl_matrix_set( d, k, i, eca);
    gsl_matrix_set( d, i, k, eca);
 
    zsum += fabs(theta);
    return zsum;
}

double MetricL1::compute_obj()
{
  double obj = 0.0;

  for (uint i = 0; i < d->size1*d->size1; i++)
    obj += f->data[i];

  return 0.5*obj;
}
